﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/util/timesrv_UtilityApi.h>
#include <nn/time/time_Result.h>
#include <nn/time/time_ResultPrivate.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace util
{

// 以下ファイルから、libnn_time にあるユーティリティをそのままコピー.
// (依存の関係で libnn_timesrv で libnn_time はリンクしない)
//
// - Eris/Sources/Libraries/time/time_Api.cpp
// - Eris/Sources/Libraries/time/time_TimeZoneApi.cpp
//

Result GetSpanBetween(int64_t* pOutSeconds, const nn::time::SteadyClockTimePoint& from, const nn::time::SteadyClockTimePoint& to) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSeconds);

    NN_RESULT_THROW_UNLESS(pOutSeconds != nullptr, nn::time::ResultInvalidPointer());
    NN_RESULT_THROW_UNLESS(to.sourceId == from.sourceId, nn::time::ResultNotComparable());

    NN_RESULT_THROW_UNLESS(
        from.value >= 0 ?
            ( to.value >= (INT64_MIN + from.value) ) :
            ( to.value <= (INT64_MAX + from.value) ),
        nn::time::ResultOverflowed() );

    *pOutSeconds = to.value - from.value;

    NN_RESULT_SUCCESS;
}

bool IsLeapYear(int year) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(year , 0);

    if ( year % 400 == 0 )
    {
        return true;
    }
    else if ( year % 100 == 0 )
    {
        return false;
    }
    else if ( year % 4 == 0 )
    {
        return true;
    }
    else
    {
        return false;
    }
}

int GetDaysInMonth(int year, int month) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(year , 0);
    NN_SDK_REQUIRES_MINMAX(month, 1, 12);

    static const int Days[12] =
    {
        31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
    };

    if(month == 2 && IsLeapYear(year))
    {
        return Days[month - 1] + 1;
    }
    else
    {
        return Days[month - 1];
    }
}

bool IsValidDate(int year, int month, int day) NN_NOEXCEPT
{
    return
        1 <= year &&
        1 <= month && month <= 12 &&
        1 <= day && day <= GetDaysInMonth(year, month);
}

int DateToDays(int inYear, int inMonth, int inDay) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(inYear, 0);
    NN_SDK_ASSERT_GREATER(inMonth, 0);
    NN_SDK_ASSERT_GREATER(inDay, 0);

    int year = inYear;
    int month = inMonth;
    int day = inDay;

    while(month > 12)
    {
        month -= 12;
        year++;
    }

    while(NN_STATIC_CONDITION(true))
    {
        int daysInMonth = GetDaysInMonth(year, month);
        if(day > daysInMonth)
        {
            day -= daysInMonth;

            month++;
            if(month > 12)
            {
                year ++;
                month = 1;
            }
        }
        else
        {
            break;
        }
    }

    NN_SDK_ASSERT_MINMAX(month, 1, 12);
    NN_SDK_ASSERT_MINMAX(day, 1, GetDaysInMonth(year, month));

    // ひと月ごとの累計経過日数(うるう年無視)
    const int DaysSum[] =
    {
        0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
    };

    int result;
    result = (year - 1) * 365;// 去年までの合計(うるう年無視)
    result += (year / 4) - (year / 100) + (year / 400);// うるう年考慮
    result += DaysSum[month - 1] + day; // 今年の日数加算(うるう年無視)
    if(month < 3 && IsLeapYear(year)) // 今年のうるう年考慮
    {
        result -= 1;
    }
    result -= 1; // ○日目じゃなくて経過日数(1/1 は 0 にしたい)
    return result;
}

nn::time::PosixTime ToPosixTimeFromUtc(const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidDate(
        static_cast<int>(calendarTime.year),
        static_cast<int>(calendarTime.month),
        static_cast<int>(calendarTime.day)));

    NN_SDK_REQUIRES_MINMAX(calendarTime.hour, 0, 23);
    NN_SDK_REQUIRES_MINMAX(calendarTime.minute, 0, 59);
    NN_SDK_REQUIRES_MINMAX(calendarTime.second, 0, 59);

    const int EpochDays = static_cast<int64_t>(DateToDays(1970, 1, 1));
    int64_t days = static_cast<int64_t>(DateToDays(calendarTime.year, calendarTime.month, calendarTime.day)) - EpochDays;
    int64_t hour = static_cast<int64_t>(calendarTime.hour);
    int64_t minute = static_cast<int64_t>(calendarTime.minute);
    int64_t second = static_cast<int64_t>(calendarTime.second);

    nn::time::PosixTime posix =
    {
        ((days * 24LL + hour) * 60LL + minute) * 60LL + second
    };

    return posix;
}

}
}
}
}
