﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_TimeZone.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/factory/settings_ConfigurationId.h>

#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include <mutex>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace settings
{

namespace
{
#if !defined (NN_BUILD_CONFIG_OS_WIN)
    template <typename T>
    void GetTimeFwdbgSettings(T* pOut, const char* key) NN_NOEXCEPT
    {
        auto outBytes = nn::settings::fwdbg::GetSettingsItemValue(
            pOut,
            sizeof(T),
            "time",
            key);
        NN_ABORT_UNLESS_EQUAL(outBytes, sizeof(T));
    }
#endif
}

nn::time::SourceId ReadSteadyClockSourceId() NN_NOEXCEPT
{
    // 初期値と、内部での失敗時(NAND壊れのとき)には nn::util::InvalidUuid が返るように
    // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-18156 で調整済。
    nn::settings::system::ClockSourceId clockSourceId;
    nn::settings::system::GetExternalSteadyClockSourceId(&clockSourceId);

    return clockSourceId;
}

void WriteSteadyClockSourceId(const nn::time::SourceId& sourceId) NN_NOEXCEPT
{
    nn::settings::system::SetExternalSteadyClockSourceId(sourceId);
}

nn::TimeSpan ReadSteadyClockInternalOffset() NN_NOEXCEPT
{
    const nn::TimeSpan internalOffset
        = nn::TimeSpan::FromSeconds(nn::settings::system::GetExternalSteadyClockInternalOffset());

    return internalOffset;
}

void WriteSteadyClockInternalOffset(const nn::TimeSpan internalOffset) NN_NOEXCEPT
{
    nn::settings::system::SetExternalSteadyClockInternalOffset(internalOffset.GetSeconds());
}

nn::time::SystemClockContext ReadLocalSystemClockContext() NN_NOEXCEPT
{
    nn::settings::system::SystemClockContext context;
    nn::settings::system::GetUserSystemClockContext(&context);
    return context;
}

void WriteLocalSystemClockContext(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    nn::settings::system::SetUserSystemClockContext(systemClockContext);
}

nn::time::SystemClockContext ReadNetworkSystemClockContext() NN_NOEXCEPT
{
    nn::settings::system::SystemClockContext context;
    nn::settings::system::GetNetworkSystemClockContext(&context);
    return context;
}

void WriteNetworkSystemClockContext(const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    nn::settings::system::SetNetworkSystemClockContext(systemClockContext);
}

nn::TimeSpan ReadNetworkClockSufficientAccuracyThreshold() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromDays(30);
#else
    uint32_t minutes;
    GetTimeFwdbgSettings(&minutes, "standard_network_clock_sufficient_accuracy_minutes");
    NN_DETAIL_TIME_INFO("[TIMESRV] ReadNetworkClockSufficientAccuracyThreshold: %lu [minutes]\n", minutes);

    return nn::TimeSpan::FromMinutes(static_cast<int64_t>(minutes));
#endif
}

bool ReadUserSystemClockAutomaticCorrectionEnabled() NN_NOEXCEPT
{
    return nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled();
}

void WriteUserSystemClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
{
    nn::settings::system::SetUserSystemClockAutomaticCorrectionEnabled(value);
}

nn::time::SteadyClockTimePoint ReadUserSystemClockAutomaticCorrectionUpdatedTime() NN_NOEXCEPT
{
    nn::time::SteadyClockTimePoint tp;
    nn::settings::system::GetUserSystemClockAutomaticCorrectionUpdatedTime(&tp);
    return tp;
}

void WriteUserSystemClockAutomaticCorrectionUpdatedTime(const nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT
{
    nn::settings::system::SetUserSystemClockAutomaticCorrectionUpdatedTime(value);
}

nn::time::LocationName ReadDeviceLocationName() NN_NOEXCEPT
{
    nn::time::LocationName locationName;
    nn::settings::system::GetDeviceTimeZoneLocationName(&locationName);
    return locationName;
}

void WriteDeviceLocationName(const nn::time::LocationName& value) NN_NOEXCEPT
{
    nn::settings::system::SetDeviceTimeZoneLocationName(value);
}

nn::time::SteadyClockTimePoint ReadDeviceLocationUpdatedTime() NN_NOEXCEPT
{
    nn::time::SteadyClockTimePoint tp;
    nn::settings::system::GetDeviceTimeZoneLocationUpdatedTime(&tp);
    return tp;
}

void WriteDeviceLocationUpdatedTime(const nn::time::SteadyClockTimePoint& value) NN_NOEXCEPT
{
    nn::settings::system::SetDeviceTimeZoneLocationUpdatedTime(value);
}

nn::TimeSpan ReadStandardSteadyClockTestOffset() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan(0);
#else
    uint32_t minutes;
    GetTimeFwdbgSettings(&minutes, "standard_steady_clock_test_offset_minutes");

    if(minutes != 0)
    {
        NN_DETAIL_TIME_INFO("[TIMESRV] ReadStandardSteadyClockTestOffset: %lu [minutes]\n", minutes);
    }

    return nn::TimeSpan::FromMinutes(static_cast<int64_t>(minutes));
#endif
}

nn::TimeSpan ReadStandardSteadyClockRtcUpdateInterval() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromMinutes(5);
#else
    uint32_t minutes;
    GetTimeFwdbgSettings(&minutes, "standard_steady_clock_rtc_update_interval_minutes");
    NN_DETAIL_TIME_INFO("[TIMESRV] ReadStandardSteadyClockRtcUpdateInterval: %lu [minutes]\n", minutes);

    return nn::TimeSpan::FromMinutes(static_cast<int64_t>(minutes));
#endif
}

void ReadBoardType(char* pOut, size_t outSize) NN_NOEXCEPT
{
    nn::settings::factory::ConfigurationId1 config;
    nn::settings::factory::GetConfigurationId1(&config);
    nn::util::Strlcpy(pOut, config.string, static_cast<int>(outSize));
}

nn::TimeSpan ReadNotifyTimeToFsInterval() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return nn::TimeSpan::FromMinutes(10);
#else
    uint32_t seconds;
    GetTimeFwdbgSettings(&seconds, "notify_time_to_fs_interval_seconds");
    NN_DETAIL_TIME_INFO("[TIMESRV] ReadNotifyTimeToFsInterval: %lu [seconds]\n", seconds);

    return nn::TimeSpan::FromSeconds(static_cast<int64_t>(seconds));
#endif
}

int ReadStandardUserSystemClockInitialYear() NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    return 2018;
#else
    uint32_t year;
    GetTimeFwdbgSettings(&year, "standard_user_clock_initial_year");
    NN_DETAIL_TIME_INFO("[TIMESRV] ReadStandardUserSystemClockInitialYear: %lu\n", year);
    return static_cast<int>(year);
#endif
}

}
}
}
}
