﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/time/time_LocationName.h>
#include <nn/timesrv/detail/service/timesrv_TimeZoneServiceServer.h>
#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>
#include <nn/timesrv/detail/core/timesrv_TimeZoneServiceCore.h>
#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace service
{

TimeZoneServiceServer::TimeZoneServiceServer(
    core::StandardSteadyClockCore* pStandardSteadyClockCore, core::TimeZoneServiceCore* pTimeZoneServiceCore, bool isAdministrator) NN_NOEXCEPT:
        m_pStandardSteadyClockCore(pStandardSteadyClockCore),
        m_pTimeZoneServiceCore(pTimeZoneServiceCore),
        m_IsAdministrator(isAdministrator)
{
}

Result TimeZoneServiceServer::GetDeviceLocationName(nn::sf::Out<nn::time::LocationName> outValue) NN_NOEXCEPT
{
    m_pTimeZoneServiceCore->GetDeviceLocationName(outValue.GetPointer());

    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceServer::SetDeviceLocationName(const nn::time::LocationName& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsAdministrator, nn::time::ResultNoCapability());

    NN_RESULT_DO( m_pTimeZoneServiceCore->SetDeviceLocationName(value) );

    // LocationNameの変更時刻を更新
    nn::time::SteadyClockTimePoint currentTimePoint;
    NN_RESULT_DO(m_pStandardSteadyClockCore->GetCurrentTimePoint(&currentTimePoint));
    m_pTimeZoneServiceCore->SetDeviceLocationUpdatedTime(currentTimePoint);

    // 永続化
    settings::WriteDeviceLocationName(value);
    settings::WriteDeviceLocationUpdatedTime(currentTimePoint);

    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceServer::GetTotalLocationNameCount(nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    outValue.Set( m_pTimeZoneServiceCore->GetTotalLocationNameCount() );
    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceServer::LoadTimeZoneRule(
    nn::sf::Out<nn::time::TimeZoneRule> outValue,
    const nn::time::LocationName& locationName) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->LoadTimeZoneRule(outValue.GetPointer(), locationName);
}

Result TimeZoneServiceServer::LoadLocationNameList(
    nn::sf::Out<int32_t> outCount,
    nn::sf::OutArray<nn::time::LocationName> pOutLocationNameList,
    int32_t offset) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->LoadLocationNameList(
            outCount.GetPointer(),
            pOutLocationNameList.GetData(),
            pOutLocationNameList.GetLength(),
            offset);
}

Result TimeZoneServiceServer::GetTimeZoneRuleVersion(nn::sf::Out<nn::time::TimeZoneRuleVersion> outValue) NN_NOEXCEPT
{
    m_pTimeZoneServiceCore->GetTimeZoneRuleVersion(outValue.GetPointer());

    NN_RESULT_SUCCESS;
}

Result TimeZoneServiceServer::GetDeviceLocationNameAndUpdatedTime(
    nn::sf::Out<nn::time::LocationName> outLocationName,
    nn::sf::Out<nn::time::SteadyClockTimePoint> outUpdatedTime) NN_NOEXCEPT
{
    m_pTimeZoneServiceCore->GetDeviceLocationName(outLocationName.GetPointer());
    outUpdatedTime.Set(m_pTimeZoneServiceCore->GetDeviceLocationUpdatedTime());

    NN_RESULT_SUCCESS;
}

// ToCalendarTime ---------------------------------------------------
Result TimeZoneServiceServer::ToCalendarTime(
    nn::sf::Out<nn::time::CalendarTime> outCalendarTime,
    nn::sf::Out<nn::time::sf::CalendarAdditionalInfo> outCalendarAdditionalInfo,
    const nn::time::PosixTime& posixTime,
    const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->ToCalendarTime(
        outCalendarTime.GetPointer(),
        outCalendarAdditionalInfo.GetPointer(),
        posixTime,
        rule);
}

Result TimeZoneServiceServer::ToCalendarTimeWithMyRule(
    nn::sf::Out<nn::time::CalendarTime> outCalendarTime,
    nn::sf::Out<nn::time::sf::CalendarAdditionalInfo> outCalendarAdditionalInfo,
    const nn::time::PosixTime& posixTime) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->ToCalendarTimeWithMyRule(
        outCalendarTime.GetPointer(),
        outCalendarAdditionalInfo.GetPointer(),
        posixTime);
}

// ToPosixTime ---------------------------------------------
Result TimeZoneServiceServer::ToPosixTime(
    nn::sf::Out<int32_t> outCount,
    nn::sf::OutArray<nn::time::PosixTime> outPosixTimeList,
    const nn::time::CalendarTime& calendarTime,
    const nn::time::TimeZoneRule& rule) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->ToPosixTime(
        outCount.GetPointer(),
        outPosixTimeList.GetData(),
        outPosixTimeList.GetLength(),
        calendarTime,
        rule);
}

Result TimeZoneServiceServer::ToPosixTimeWithMyRule(
    nn::sf::Out<int32_t> outCount,
    nn::sf::OutArray<nn::time::PosixTime> outPosixTimeList,
    const nn::time::CalendarTime& calendarTime) NN_NOEXCEPT
{
    return m_pTimeZoneServiceCore->ToPosixTimeWithMyRule(
        outCount.GetPointer(),
        outPosixTimeList.GetData(),
        outPosixTimeList.GetLength(),
        calendarTime);
}

}
}
}
}
