﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/fs/timesrv_TimeZoneBinaryManager.h>
#include <nn/time/time_ResultPrivate.h>

#include <nn/timesrv/detail/fs/timesrv_Fs.h>
#include <siglo/Include/time_TzApiForTimeZoneBinary.h>
#include <siglo/Include/time_TimeZoneResources.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace fs
{

char TimeZoneBinaryManager::m_WorkBuffer[1024 * 20]; // タイムゾーンバイナリ、タイムゾーンリストの読み込み用バッファ

Result TimeZoneBinaryManager::MountFs() NN_NOEXCEPT
{
    return  nn::timesrv::detail::fs::Mount();
}

constexpr bool TimeZoneBinaryManager::HasSourceLinkTimeZoneBinary() NN_NOEXCEPT
{
#if NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL == NN_DETAIL_TIME_CONFIG_TIMEZONE_BINARY_MODEL_SOURCE_LINK
    return true;
#else
    return false;
#endif
}

Result TimeZoneBinaryManager::ReadTimeZoneBinary(
    char** pOutTimeZoneBinary,
    size_t* pOutSize,
    const nn::time::LocationName locationName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutTimeZoneBinary);
    NN_SDK_ASSERT_NOT_NULL(pOutSize);

    nn::timesrv::detail::fs::BinaryPath path;
    nn::timesrv::detail::fs::GetFilePath(&path, locationName);
    size_t fileSize = 0;
    NN_RESULT_DO(
        nn::timesrv::detail::fs::ReadFile(&fileSize , m_WorkBuffer, sizeof(m_WorkBuffer), path) );

    *pOutTimeZoneBinary = m_WorkBuffer;
    *pOutSize = fileSize;

    NN_RESULT_SUCCESS;
}

Result TimeZoneBinaryManager::ReadTimeZoneListData(
    size_t* pOutSize,
    char* pOutBuffer,
    size_t outBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSize);
    NN_SDK_ASSERT_NOT_NULL(pOutBuffer);

    nn::timesrv::detail::fs::BinaryPath path;
    nn::timesrv::detail::fs::GetFilePathOfTimeZoneListData(&path);

    size_t fileSize;
    NN_RESULT_DO(
        nn::timesrv::detail::fs::ReadFile(&fileSize, pOutBuffer, outBufferSize, path));

    *pOutSize = fileSize;

    NN_RESULT_SUCCESS;
}

int32_t TimeZoneBinaryManager::GetLocationNameCount() NN_NOEXCEPT
{
    if(NN_STATIC_CONDITION(HasSourceLinkTimeZoneBinary()))
    {
        return static_cast<int32_t>(nne::tz::BinaryResourceCount);
    }
    else
    {
        int32_t locationNameCount = 0;

        size_t fileSize;
        auto result = ReadTimeZoneListData(&fileSize, m_WorkBuffer, sizeof(m_WorkBuffer));
        if(result.IsSuccess())
        {
            CountLocationNameFromListBuffer(
                &locationNameCount ,
                m_WorkBuffer,
                fileSize);
        }

        return locationNameCount;
    }
}

bool TimeZoneBinaryManager::IsValidLocationName(const nn::time::LocationName& locationName) NN_NOEXCEPT
{
    // ファイルサイズを取得できるかで、存在する地域名かをチェックする
    nn::timesrv::detail::fs::BinaryPath path;
    nn::timesrv::detail::fs::GetFilePath(&path, locationName);

    size_t fileSize;
    return nn::timesrv::detail::fs::GetFileSize(&fileSize, path).IsSuccess();
}

Result TimeZoneBinaryManager::LoadLocationNameList(
    int32_t* pOutCount,
    nn::time::LocationName* pOutLocationNameList,
    size_t locationNameListCount,
    int32_t offset) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT_NOT_NULL(pOutLocationNameList);

    if(NN_STATIC_CONDITION(HasSourceLinkTimeZoneBinary()))
    {
        NN_RESULT_DO(RetrieveLocationNameList(
            pOutCount,
            pOutLocationNameList,
            locationNameListCount,
            offset,
            nne::tz::GetBinaryResourceListPtr(),
            nne::tz::BinaryResourceCount));
    }
    else
    {
        size_t fileSize;
        NN_RESULT_DO(ReadTimeZoneListData(&fileSize, m_WorkBuffer, sizeof(m_WorkBuffer)));

        NN_RESULT_DO(RetrieveLocationNameList(
            pOutCount,
            pOutLocationNameList,
            locationNameListCount,
            offset,
            m_WorkBuffer,
            fileSize));
    }
    NN_RESULT_SUCCESS;
}

Result TimeZoneBinaryManager::GetTimeZoneRuleVersion(nn::time::TimeZoneRuleVersion* pOut) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOut);

    if (NN_STATIC_CONDITION(HasSourceLinkTimeZoneBinary()))
    {
        nn::util::Strlcpy(pOut->data, nne::tz::VersionString, pOut->Size);
    }
    else
    {
        nn::timesrv::detail::fs::BinaryPath path;
        nn::timesrv::detail::fs::GetFilePathOfTimeZoneVersion(&path);

        size_t fileSize;
        NN_RESULT_DO(
            nn::timesrv::detail::fs::ReadFile(&fileSize, pOut->data, pOut->Size, path));
        pOut->data[fileSize] = '\0';
    }

    NN_RESULT_SUCCESS;
}

void TimeZoneBinaryManager::CountLocationNameFromListBuffer(
    int32_t* pOutCount,
    const char* pListBuffer,
    size_t listBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT_NOT_NULL(pListBuffer);

    const char* p = pListBuffer;
    int32_t foundLocationNameCount = 0;

    for(size_t i = 0 ; i < listBufferSize ; ++i)
    {
        // CRLF(\r\n) の数が LocationName 数となる
        if(*p == '\0')
        {
            break;
        }
        else if(*p == '\r')
        {
            // do nothing
        }
        else if(*p == '\n')
        {
            foundLocationNameCount++;
        }
        p++;
    }

    *pOutCount = foundLocationNameCount;
}

Result TimeZoneBinaryManager::RetrieveLocationNameList(
    int32_t* pOutCount,
    nn::time::LocationName* pOutLocationNameList,
    size_t count,
    int32_t offset,
    const char* pListBuffer,
    size_t listBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT_NOT_NULL(pOutLocationNameList);
    NN_SDK_ASSERT_NOT_NULL(pListBuffer);

    struct LocationNameHolder
    {
        nn::time::LocationName name;
        size_t index;

        LocationNameHolder() NN_NOEXCEPT
        {
            this->Reset();
        }
        bool Push(uint8_t data) NN_NOEXCEPT
        {
            if( !(this->index < this->name.Size - 1) )
            {
                return false;
            }
            this->name._value[this->index++] = static_cast<char>(data);
            return true;
        }
        void Reset() NN_NOEXCEPT
        {
            std::memset(&(this->name), 0, sizeof(nn::time::LocationName));
            this->index = 0;
        }
    };
    LocationNameHolder locationNameHolder;

    const char* p = pListBuffer;
    size_t pushIndex = 0;
    int foundLocationNameCount = 0;
    for(size_t i = 0 ; i < listBufferSize ; ++i)
    {
        // CRLF(\r\n) で LocationName をスプリット
        if(*p == '\0')
        {
            break;
        }
        else if(*p == '\r')
        {
            // do nothing
        }
        else if(*p == '\n')
        {
            foundLocationNameCount++;

            if(foundLocationNameCount > offset)
            {
                pOutLocationNameList[pushIndex++] = locationNameHolder.name;
                if(count <= pushIndex)
                {
                    break;
                }
            }

            locationNameHolder.Reset();
        }
        else
        {
            auto ret =locationNameHolder.Push(*p);
            NN_RESULT_THROW_UNLESS(ret, nn::time::ResultOutOfMemory());
        }
        p++;
    }

    *pOutCount = static_cast<int32_t>(pushIndex);
    NN_RESULT_SUCCESS;
}

Result TimeZoneBinaryManager::RetrieveLocationNameList(
    int32_t* pOutCount,
    nn::time::LocationName* pOutLocationNameList,
    size_t count,
    int32_t offset,
    const nne::tz::BinaryResource* pBinaryResource,
    size_t binaryResourceCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT_NOT_NULL(pOutLocationNameList);
    NN_SDK_ASSERT_NOT_NULL(pBinaryResource);

    size_t pushIndex = 0;
    for(size_t i = static_cast<size_t>(offset) ; i < binaryResourceCount ; ++i)
    {
        if(count <= pushIndex)
        {
            break;
        }
        pOutLocationNameList[pushIndex++] = pBinaryResource[i].name;
    }

    *pOutCount = static_cast<int32_t>(pushIndex);

    NN_RESULT_SUCCESS;
}

}
}
}
}

