﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/core/timesrv_WorkerThread.h>
#include <nn/timesrv/detail/core/timesrv_SharedMemoryManager.h>
#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>
#include <nn/timesrv/detail/service/timesrv_ServiceProvider.h>
#include <nn/nn_SystemThreadDefinition.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace core
{

namespace
{
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_WorkerThreadStackBuffer[1024 * 4];
}

WorkerThread::WorkerThread(
    core::StandardSteadyClockCore* pStandardSteadyClockCore,
    core::SharedMemoryManager* pSharedMemoryManager,
    nn::timesrv::detail::service::PosixTimeNotifierToFs* pPosixTimeNotifierToFs) NN_NOEXCEPT:
    m_ThreadStopEvent(nn::os::EventClearMode_ManualClear),
    m_pStandardSteadyClockCore(pStandardSteadyClockCore),
    m_pSharedMemoryManager(pSharedMemoryManager),
    m_pPosixTimeNotifierToFs(pPosixTimeNotifierToFs),
    m_SteadyClockBaseTimeUpdateEvent(nn::os::EventClearMode_ManualClear),
    m_NotifyPosixTimeEvent(nn::os::EventClearMode_ManualClear)
{
    auto rtcUpdateInterval = settings::ReadStandardSteadyClockRtcUpdateInterval();
    m_SteadyClockBaseTimeUpdateEvent.StartPeriodic(rtcUpdateInterval, rtcUpdateInterval);

    // FSへの時刻通知設定
    // 時刻変更、タイムゾーン変更時に通知しているが、夏時間などの影響でそういったトリガーなしに時差が変化することへのカバー.
    // 時差の時差が遅れて通知されるが、それは良しとする.
    auto notifyPosixTimeEventInterval = settings::ReadNotifyTimeToFsInterval();
    m_NotifyPosixTimeEvent.StartPeriodic(notifyPosixTimeEventInterval, notifyPosixTimeEventInterval);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &m_Thread,
            ThreadFunction,
            this,
            g_WorkerThreadStackBuffer,
            sizeof(g_WorkerThreadStackBuffer),
            NN_SYSTEM_THREAD_PRIORITY(time, Worker)));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(time, Worker));
    nn::os::StartThread(&m_Thread);
}

WorkerThread::~WorkerThread() NN_NOEXCEPT
{
    m_ThreadStopEvent.Signal();
    nn::os::DestroyThread(&m_Thread);
}

void WorkerThread::WriteLocalSystemClockContextAsync(const nn::time::SystemClockContext& value) NN_NOEXCEPT
{
    m_LocalSystemClockContextEventData.SetAndEventSignal(value);
}

void WorkerThread::WriteNetworkSystemClockContextAsync(const nn::time::SystemClockContext& value) NN_NOEXCEPT
{
    m_NetworkSystemClockContextEventData.SetAndEventSignal(value);
}

void WorkerThread::ThreadFunctionImpl() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        auto waitResult = nn::os::WaitAny(
            m_ThreadStopEvent.GetBase(),                            // 0
            m_LocalSystemClockContextEventData.GetEventBase(),      // 1
            m_NetworkSystemClockContextEventData.GetEventBase(),    // 2
            m_SteadyClockBaseTimeUpdateEvent.GetBase(),             // 3
            m_NotifyPosixTimeEvent.GetBase());                      // 4

        switch (waitResult)
        {
        case 0:
            return;
        case 1:
            {
                const auto value = m_LocalSystemClockContextEventData.GetAndEventClear();
                settings::WriteLocalSystemClockContext( value );
                // NN_DETAIL_TIME_TRACE("[WorkerThread::ThreadFunctionImpl] SetUserSystemClockContext\n");
            }
            break;
        case 2:
            {
                const auto value = m_NetworkSystemClockContextEventData.GetAndEventClear();
                settings::WriteNetworkSystemClockContext( value );
                // NN_DETAIL_TIME_TRACE("[WorkerThread::ThreadFunctionImpl] SetNetworkSystemClockContext\n");
            }
            break;
        case 3:
            {
                m_SteadyClockBaseTimeUpdateEvent.Clear();
                m_pStandardSteadyClockCore->UpdateBaseTimeWithRtc();

                // 更新された SteadyClockTimePoint の値を共有メモリへ反映
                {
                    auto value = m_pStandardSteadyClockCore->GetCurrentTimePointValue();
                    m_pSharedMemoryManager->SetStandardSteadyClockTimePointValue(value);
                }
            }
            break;
        case 4:
            {
                m_NotifyPosixTimeEvent.Clear();
                m_pPosixTimeNotifierToFs->Notify();
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

}
}
}
}
