﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_Api.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/time/time_ResultPrivate.h>
#include <nn/time/detail/util/time_UtilApi.h>

#include <nn/timesrv/detail/service/timesrv_ITimeZoneService.sfdl.h>
#include <nn/sf/sf_ISharedObject.h>

#include <algorithm> // std::min

namespace nn { namespace time {

extern nn::sf::SharedPointer<nn::timesrv::detail::service::ITimeZoneService> g_pTimeZoneService;

// TODO: GetLocationName に改名？
void GetDeviceLocationName(LocationName* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_ABORT_UNLESS(g_pTimeZoneService != nullptr, "[TIME] The time library is not initialized.");
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_pTimeZoneService->GetDeviceLocationName(pOut) );
}

nn::Result LoadTimeZoneRule(TimeZoneRule* pOut, const LocationName& name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_RESULT_THROW_UNLESS(g_pTimeZoneService != nullptr, ResultNotInitialized());

    return g_pTimeZoneService->LoadTimeZoneRule(pOut, name);
}

nn::Result ToCalendarTime(
    CalendarTime* pOutCalendar,
    CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const PosixTime& posixTime,
    const TimeZoneRule& rule) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCalendar);
    NN_RESULT_THROW_UNLESS(g_pTimeZoneService != nullptr, ResultNotInitialized());

    nn::time::sf::CalendarAdditionalInfo sfCalendarAdditionalInfo; // enum を排除した型
    NN_RESULT_DO(g_pTimeZoneService->ToCalendarTime(
        pOutCalendar,
        &sfCalendarAdditionalInfo,
        posixTime,
        rule));

    if(pOutCalendarAdditionalInfo)
    {
        pOutCalendarAdditionalInfo->dayOfWeek = static_cast<nn::time::DayOfWeek>(sfCalendarAdditionalInfo.dayOfWeek);
        pOutCalendarAdditionalInfo->timeZone = sfCalendarAdditionalInfo.timeZone;
        pOutCalendarAdditionalInfo->yearDay = sfCalendarAdditionalInfo.yearDay;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ToCalendarTime(
    CalendarTime* pOutCalendar,
    CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const PosixTime& posixTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCalendar);
    NN_RESULT_THROW_UNLESS(g_pTimeZoneService != nullptr, ResultNotInitialized());

    nn::time::sf::CalendarAdditionalInfo sfCalendarAdditionalInfo; // enum を排除した型
    NN_RESULT_DO(g_pTimeZoneService->ToCalendarTimeWithMyRule(
        pOutCalendar,
        &sfCalendarAdditionalInfo,
        posixTime));

    if(pOutCalendarAdditionalInfo)
    {
        pOutCalendarAdditionalInfo->dayOfWeek = static_cast<nn::time::DayOfWeek>(sfCalendarAdditionalInfo.dayOfWeek);
        pOutCalendarAdditionalInfo->timeZone = sfCalendarAdditionalInfo.timeZone;
        pOutCalendarAdditionalInfo->yearDay = sfCalendarAdditionalInfo.yearDay;
    }

    NN_RESULT_SUCCESS;
}

CalendarTime ToCalendarTimeInUtc(const PosixTime& posixTime) NN_NOEXCEPT
{
    return detail::util::ToCalendarTimeInUtc(posixTime);
}

nn::Result ToPosixTime(
    int* pOutCount,
    PosixTime* pOutPosixTimeList,
    int count,
    const CalendarTime& calendarTime,
    const TimeZoneRule& rule) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutPosixTimeList);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_SDK_REQUIRES_MINMAX(calendarTime.month, 1, 12);
    NN_SDK_REQUIRES_MINMAX(calendarTime.day, 1, 31);
    NN_SDK_REQUIRES_MINMAX(calendarTime.hour, 0, 23);
    NN_SDK_REQUIRES_MINMAX(calendarTime.minute, 0, 59);
    NN_SDK_REQUIRES_MINMAX(calendarTime.second, 0, 59);

    NN_RESULT_THROW_UNLESS(g_pTimeZoneService != nullptr, ResultNotInitialized());

    nn::sf::OutArray<PosixTime> outPosixTimeList(pOutPosixTimeList, static_cast<size_t>(std::min(count, 2)));
    return g_pTimeZoneService->ToPosixTime(
        pOutCount,
        outPosixTimeList,
        calendarTime,
        rule);
}

nn::Result ToPosixTime(
    int* pOutCount,
    PosixTime* pOutPosixTimeList,
    int count,
    const CalendarTime& calendarTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutPosixTimeList);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_SDK_REQUIRES_MINMAX(calendarTime.month, 1, 12);
    NN_SDK_REQUIRES_MINMAX(calendarTime.day, 1, 31);
    NN_SDK_REQUIRES_MINMAX(calendarTime.hour, 0, 23);
    NN_SDK_REQUIRES_MINMAX(calendarTime.minute, 0, 59);
    NN_SDK_REQUIRES_MINMAX(calendarTime.second, 0, 59);

    NN_RESULT_THROW_UNLESS(g_pTimeZoneService != nullptr, ResultNotInitialized());

    nn::sf::OutArray<PosixTime> outPosixTimeList(pOutPosixTimeList, static_cast<size_t>(std::min(count, 2)));
    return g_pTimeZoneService->ToPosixTimeWithMyRule(
        pOutCount,
        outPosixTimeList,
        calendarTime);
}

PosixTime ToPosixTimeFromUtc(const CalendarTime& calendarTime) NN_NOEXCEPT
{
    return detail::util::ToPosixTimeFromUtc(calendarTime);
}

/*
// いったん作らない
nn::Result AdjustCalendarTime(
    int* pOutCount,
    CalendarTime* pOutCalendarTimeList,
    int count,
    const CalendarTime& calendarTime,
    const TimeZoneRule& rule) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutCalendarTimeList);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_RESULT_DO( detail::tz::AdjustCalendarTime(pOutCount, pOutCalendarTimeList, count, calendarTime, rule) );

    NN_RESULT_SUCCESS;
}
*/

}} // nn::time

