﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/detail/util/time_UtilApi.h>

namespace nn { namespace time { namespace detail { namespace util {

namespace
{
    const int EpochDays = nn::time::detail::util::DateToDays(1970, 1, 1);
}

Result GetSpanBetween(int64_t* pOutSeconds, const SteadyClockTimePoint& from, const SteadyClockTimePoint& to) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSeconds);

    NN_RESULT_THROW_UNLESS(pOutSeconds != nullptr, ResultInvalidPointer());
    NN_RESULT_THROW_UNLESS(to.sourceId == from.sourceId, ResultNotComparable());

    NN_RESULT_THROW_UNLESS(
        from.value >= 0 ?
            ( to.value >= (INT64_MIN + from.value) ) :
            ( to.value <= (INT64_MAX + from.value) ),
        ResultOverflowed() );

    *pOutSeconds = to.value - from.value;

    NN_RESULT_SUCCESS;
}

bool IsLeapYear(int year) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(year , 0);

    if ( year % 400 == 0 )
    {
        return true;
    }
    else if ( year % 100 == 0 )
    {
        return false;
    }
    else if ( year % 4 == 0 )
    {
        return true;
    }
    else
    {
        return false;
    }
}

bool IsValidDate(int year, int month, int day) NN_NOEXCEPT
{
    return
        1 <= year &&
        1 <= month && month <= 12 &&
        1 <= day && day <= GetDaysInMonth(year, month);
}

int GetDaysInMonth(int year, int month) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(year , 0);
    NN_SDK_REQUIRES_MINMAX(month, 1, 12);

    static const int Days[12] =
    {
        31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
    };

    if(month == 2 && IsLeapYear(year))
    {
        return Days[month - 1] + 1;
    }
    else
    {
        return Days[month - 1];
    }
}

int DateToDays(int inYear, int inMonth, int inDay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(inYear, 0);
    NN_SDK_REQUIRES_GREATER(inMonth, 0);
    NN_SDK_REQUIRES_GREATER(inDay, 0);

    int year = inYear;
    int month = inMonth;
    int day = inDay;

    // うるう年計算が狂うので month は 1～12 に. day はそのまま加算すれば良い.
    year += month / 12;
    month %= 12;
    if(month == 0)
    {
        month = 12;
        year--;
    }
    NN_SDK_ASSERT_MINMAX(month, 1, 12);

    // ひと月ごとの累計経過日数(うるう年無視)
    const int DaysSum[] =
    {
        0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
    };

    int result;
    result = (year - 1) * 365;// 去年までの合計(うるう年無視)
    result += (year / 4) - (year / 100) + (year / 400);// うるう年考慮
    result += DaysSum[month - 1] + day; // 今年の日数加算(うるう年無視)
    if(month < 3 && IsLeapYear(year)) // 今年のうるう年考慮
    {
        result -= 1;
    }
    result -= 1; // ○日目じゃなくて経過日数(1/1 は 0 にしたい)
    return result;
}

void DaysToDate(int *pOutYear, int *pOutMonth, int *pOutDay, int inDays) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(inDays, 0);

    int days = inDays;

    // CTR nn::fnd::DateTime::DaysToDate から拝借.
    // フェアフィールドの公式を変形して利用している.
    // 2000年1月1日からの経過日数を1年1月1日に改修.

    const int c4days = (((365 * 4 + 1) * 25 - 1) * 4 + 1);
    const int c1days =  ((365 * 4 + 1) * 25 - 1);
    const int y4days =   (365 * 4 + 1);
    const int y1days =    365;
    int year, month, day;

    days -= 31 + 28;
    days += 365; // フェアフィールドの公式は西暦0年がベースになってるので1年をベースにする

    int c4    = days / c4days;
    int c4ds  = days % c4days;

    if (c4ds < 0)
    {
        c4ds += c4days;
        c4 -= 1;
    }

    int c1    = c4ds / c1days;
    int c1ds  = c4ds % c1days;
    int y4    = c1ds / y4days;
    int y4ds  = c1ds % y4days;
    int y1    = y4ds / y1days;
    int ydays = y4ds % y1days;

    {
        // CTR で2000年からになっていたのを0年にする.
        // ここで2000年ベースを単純に1年ベースにするとうるう年判定が狂うので、
        // 0にして(2000年も0年もうるう年)、↑の days+=365 で調整

        // year = 2000 + c4 * 400 + c1 * 100 + y4 * 4 + y1;
        year = c4 * 400 + c1 * 100 + y4 * 4 + y1;
    }
    month = (5 * ydays + 2) / 153;
    day = ydays - (153 * month + 2) / 5 + 1;

    if (y1 == 4 || c1 == 4)
    {
        month = 2 + (12 - 3);
        day = 29;
        year -= 1;
    }

    if ( month <= (12 - 3) )
    {
        month += 3;
    }
    else
    {
        month -= 12 - 3;
        year += 1;
    }

    if ( pOutYear )
    {
        *pOutYear = year;
    }
    if ( pOutMonth )
    {
        *pOutMonth = month;
    }
    if ( pOutDay )
    {
        *pOutDay = day;
    }
}

nn::time::DayOfWeek GetDayOfWeek(int year, int month, int day) NN_NOEXCEPT
{
    // 西暦1年1月1日が月曜日(DayOfWeek_Monday = 1)なので、
    // DateToDays に1をプラスして補正して、7の剰余が答え
    int days = nn::time::detail::util::DateToDays(year, month, day) + 1;
    return static_cast<nn::time::DayOfWeek>(days % 7);
}

nn::time::PosixTime ToPosixTimeFromUtc(const CalendarTime& calendarTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidDate(
        static_cast<int>(calendarTime.year),
        static_cast<int>(calendarTime.month),
        static_cast<int>(calendarTime.day)));

    NN_SDK_REQUIRES_MINMAX(calendarTime.hour, 0, 23);
    NN_SDK_REQUIRES_MINMAX(calendarTime.minute, 0, 59);
    NN_SDK_REQUIRES_MINMAX(calendarTime.second, 0, 59);

    int64_t days = static_cast<int64_t>(nn::time::detail::util::DateToDays(calendarTime.year, calendarTime.month, calendarTime.day)) - EpochDays;
    int64_t hour = static_cast<int64_t>(calendarTime.hour);
    int64_t minute = static_cast<int64_t>(calendarTime.minute);
    int64_t second = static_cast<int64_t>(calendarTime.second);

    nn::time::PosixTime posix =
    {
        ((days * 24LL + hour) * 60LL + minute) * 60LL + second
    };

    return posix;
}

CalendarTime ToCalendarTimeInUtc(const PosixTime& posixTime) NN_NOEXCEPT
{
    int year, month, day, hour, minute, second;

    int days = static_cast<int>( posixTime.value / (24LL * 60LL * 60LL) ) + EpochDays;

    DaysToDate(&year, &month, &day, days);

    int64_t posixAbsoluteValue = posixTime.value >= 0 ? posixTime.value : (-1LL * posixTime.value);
    int64_t remain = static_cast<int64_t>( posixAbsoluteValue % (24LL * 60LL * 60LL) );
    if(posixTime.value < 0)
    {
        remain *= -1LL;
    }

    if (remain < 0)
    {
        day--;
        if (day <= 0)
        {
            month--;
            if (month <= 0)
            {
                month = 12;
                year--;
            }
            day = nn::time::detail::util::GetDaysInMonth(year, month);
        }

        remain += 60LL * 60LL * 24LL;
    }

    hour = static_cast<int>( remain / (60LL * 60LL) );
    remain %= (60LL * 60LL);

    minute = static_cast<int>( remain / 60LL );
    remain %= 60LL;

    second = static_cast<int>( remain );

    CalendarTime calendar =
    {
        static_cast<int16_t>(year),
        static_cast<int8_t>(month),
        static_cast<int8_t>(day),
        static_cast<int8_t>(hour),
        static_cast<int8_t>(minute),
        static_cast<int8_t>(second)
    };
    return calendar;
}

}}}}
