﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>

#include <nn/stdfu/stdfu_Result.public.h>
#include "../detail/stdfu_Log.h"
#include "stdfu_DfuFile.h"
#include "stdfu_Crc.h"

namespace nn {
namespace stdfu {
//////////////////////////////////////////////////////////////////////////////
// Public
//////////////////////////////////////////////////////////////////////////////
Result DfuFile::Initialize(void *pBinary, uint32_t totalBytes, uint16_t *pOutVid, uint16_t *pOutPid, uint16_t *pOutBcd, uint8_t *pOutImageCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutImageCount);

    m_pDfuPrefix    = reinterpret_cast<DfuPrefix*>(pBinary);
    m_pDfuSuffix    = reinterpret_cast<DfuSuffix*>(reinterpret_cast<uint8_t*>(pBinary) + (totalBytes - sizeof(DfuSuffix)));
    m_pImage        = reinterpret_cast<uint8_t*>(pBinary) + sizeof(DfuPrefix);

    Result result = ValidateDfuPrefix(totalBytes);

    if (result.IsSuccess())
    {
        result = ValidateDfuSuffix();

        if (result.IsSuccess())
        {
            if (pOutVid)
            {
                *pOutVid    = SwapBytes16(&m_pDfuSuffix->idVendorLo);
            }

            if (pOutPid)
            {
                *pOutPid    = SwapBytes16(&m_pDfuSuffix->idProductLo);
            }

            if (pOutBcd)
            {
                *pOutBcd    = SwapBytes16(&m_pDfuSuffix->bcdDeviceLo);
            }

            *pOutImageCount = m_pDfuPrefix->bTargets;

            NN_DETAIL_STDFU_INFO("DFU binary intended to update vid: %04x, pid: %04x, bcd: %04x\n", SwapBytes16(&m_pDfuSuffix->idVendorLo), SwapBytes16(&m_pDfuSuffix->idProductLo), SwapBytes16(&m_pDfuSuffix->bcdDeviceLo));
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuFile::GetImage(DfuTargetPrefix **pOutDfuTargetPrefix, uint8_t *pOutAlternateSetting, uint8_t **pOutTargetName, uint32_t *pOutElementsCount, uint8_t imageIndex) NN_NOEXCEPT
{
    uint8_t *p = m_pImage;

    for (int iImage = 0; iImage < m_pDfuPrefix->bTargets; iImage++)
    {
        DfuTargetPrefix *pDfuTargetPrefix = reinterpret_cast<DfuTargetPrefix*>(p);

        if (iImage == imageIndex)
        {
            if (
                    (pDfuTargetPrefix->szSignature[0] == 'T') &&
                    (pDfuTargetPrefix->szSignature[1] == 'a') &&
                    (pDfuTargetPrefix->szSignature[2] == 'r') &&
                    (pDfuTargetPrefix->szSignature[3] == 'g') &&
                    (pDfuTargetPrefix->szSignature[4] == 'e') &&
                    (pDfuTargetPrefix->szSignature[5] == 't')
                )
            {
                *pOutDfuTargetPrefix    = pDfuTargetPrefix;
                *pOutAlternateSetting   = pDfuTargetPrefix->bAlternatesetting;
                *pOutTargetName         = pDfuTargetPrefix->szTargetName;
                *pOutElementsCount      = SwapBytes32(pDfuTargetPrefix->dwNbElements);

                return ResultSuccess();
            }
            else
            {
                return ResultInvalidDfuImageSignature();
            }
        }

        p += sizeof(DfuTargetPrefix);

        for (int iElement = 0; iElement < SwapBytes32(pDfuTargetPrefix->dwNbElements); iElement++)
        {
            DfuImageElement *pDfuImageElement = reinterpret_cast<DfuImageElement*>(p);
            p += (sizeof(DfuImageElement) + SwapBytes32(pDfuImageElement->dwElementSize));
        }
    }

    return ResultDfuImageNotFound();
}


//////////////////////////////////////////////////////////////////////////////
Result DfuFile::GetElement(uint32_t *pOutAddress, uint32_t *pOutSize, uint8_t **pOutData, uint32_t iElement, DfuTargetPrefix *pDfuTargetPrefix) NN_NOEXCEPT
{
    uint8_t *p = reinterpret_cast<uint8_t*>(pDfuTargetPrefix);

    p += sizeof(DfuTargetPrefix);

    for (uint32_t i = 0; i < SwapBytes32(pDfuTargetPrefix->dwNbElements); i++)
    {
        DfuImageElement *pDfuImageElement = reinterpret_cast<DfuImageElement*>(p);

        if (i == iElement)
        {
            *pOutAddress    = SwapBytes32(pDfuImageElement->dwElementAddres);
            *pOutSize       = SwapBytes32(pDfuImageElement->dwElementSize);
            *pOutData       = p + sizeof(DfuImageElement);

            return ResultSuccess();
        }

        p += (sizeof(DfuImageElement) + SwapBytes32(pDfuImageElement->dwElementSize));
    }

    return ResultDfuElementNotFound();

}


//////////////////////////////////////////////////////////////////////////////
// Private
//////////////////////////////////////////////////////////////////////////////
Result DfuFile::ValidateDfuPrefix(uint32_t totalBytes) NN_NOEXCEPT
{
    if (
            (m_pDfuPrefix->szSignature[0] != 'D') ||
            (m_pDfuPrefix->szSignature[1] != 'f') ||
            (m_pDfuPrefix->szSignature[2] != 'u') ||
            (m_pDfuPrefix->szSignature[3] != 'S') ||
            (m_pDfuPrefix->szSignature[4] != 'e')
        )
    {
        return ResultInvalidDfuPrefixSignature();
    }

    if (m_pDfuPrefix->bVersion != 0x01)
    {
        return ResultInvalidDfuPrefixVersion();
    }

    if (SwapBytes32(m_pDfuPrefix->DFUImageSize) != totalBytes)
    {
        return ResultInvalidDfuPrefixImageSize();
    }

    if (!m_pDfuPrefix->bTargets)
    {
        return ResultInvalidDfuPrefixTargets();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DfuFile::ValidateDfuSuffix() NN_NOEXCEPT
{
    if (
            (m_pDfuSuffix->ucDfuSignature[0] != 'U') ||
            (m_pDfuSuffix->ucDfuSignature[1] != 'F') ||
            (m_pDfuSuffix->ucDfuSignature[2] != 'D')
        )
    {
        return ResultInvalidDfuSuffixSignature();
    }

    if (IsValidCrc() == false)
    {
        return ResultInvalidDfuSuffixCrc();
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
bool DfuFile::IsValidCrc() NN_NOEXCEPT
{
    if (SwapBytes32(m_pDfuSuffix->dwCRC) == ComputeCrc())
    {
        return true;
    }

    return false;
}


//////////////////////////////////////////////////////////////////////////////
uint32_t DfuFile::ComputeCrc() NN_NOEXCEPT
{
    uint32_t fullcrc    = 0xffffffff;
    uint32_t bytes      = SwapBytes32(m_pDfuPrefix->DFUImageSize) - 4;
    uint8_t *p          = reinterpret_cast<uint8_t*>(m_pDfuPrefix);

    while (bytes--)
    {
        CRC(fullcrc, *p++);
    }

    return fullcrc;
}


//////////////////////////////////////////////////////////////////////////////
uint32_t DfuFile::SwapBytes32(uint8_t *p) NN_NOEXCEPT
{
    uint32_t value;

    value = *p++;
    value |= (*p++ << 8);
    value |= (*p++ << 16);
    value |= (*p++ << 24);

    return value;
}


//////////////////////////////////////////////////////////////////////////////
uint16_t DfuFile::SwapBytes16(uint8_t *p) NN_NOEXCEPT
{
    uint16_t value;

    value = *p++;
    value |= (*p++ << 8);

    return value;
}


//////////////////////////////////////////////////////////////////////////////
} // namespace stdfu
} // namespace nn

