﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
  * @file
  * @brief   STDFU API
  */

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>

#include <nn/stdfu/stdfu.h>
#include <nn/stdfu/stdfu_DfuUpdateStatus.h>

#include "stdfu_DfuFile.h"
#include "stdfu_DfuDevice.h"
#include "stdfu_AhidDevice.h"
#include "../detail/stdfu_Log.h"

namespace nn { namespace stdfu {

//////////////////////////////////////////////////////////////////////////////
Result StDfu::Initialize() NN_NOEXCEPT
{
    m_DfuUpdateStatus.Initialize();

    return ResultSuccess();
}

//////////////////////////////////////////////////////////////////////////////
Result StDfu::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}

//////////////////////////////////////////////////////////////////////////////
Result StDfu::UpdateDfuDevice(void *pBinary, uint32_t bytes, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    uint16_t    vid1;
    uint16_t    pid1;
    uint16_t    bcd1;
    uint8_t     imageCount;

    DfuFile     dfuFile;
    Result result = dfuFile.Initialize(pBinary, bytes, &vid1, &pid1, &bcd1, &imageCount);

    if (result.IsSuccess())
    {
        DfuDevice dfu;
        result = dfu.PerformUpdate(vid, pid, bcd, &dfuFile, imageCount, &m_DfuUpdateStatus);
    }

    return result;
}

//////////////////////////////////////////////////////////////////////////////
Result StDfu::Update(void *pBinary, uint32_t bytes, uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    Result      result;
    uint16_t    vid1;
    uint16_t    pid1;
    uint16_t    bcd1;
    uint8_t     imageCount;
    DfuFile     dfuFile;

    m_DfuUpdateStatus.Initialize();

    result = dfuFile.Initialize(pBinary, bytes, &vid1, &pid1, &bcd1, &imageCount);

    if (result.IsSuccess())
    {
        AhidDevice ahid;

        // If the caller specified vid, pid, bcd, then attach to that AHID device
        if (vid != UpdateParameters::UpdateParameters_IdFromBinary)
        {
            vid1 = vid;
        }

        if (pid != UpdateParameters::UpdateParameters_IdFromBinary)
        {
            pid1 = pid;
        }

        if (bcd != UpdateParameters::UpdateParameters_IdFromBinary)
        {
            bcd1 = bcd;
        }

        result = ahid.ChangeToDfuDevice(vid1, pid1, bcd1);

        // The DFU device has a battery. This means if someone unplugs the
        // device in the middle an update it will leave the device as a DFU
        // device. So we should try to find the DFU device regardless of if we
        // found the AHID device.
        //if (result.IsSuccess())
        {
            DfuDevice dfu;

            // Sleep here for awhile to allow DFU device to enumerate
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));

            result = dfu.PerformUpdate(0x057e, 0x200f, 0x0200, &dfuFile, imageCount, &m_DfuUpdateStatus);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result StDfu::Update(void *pBinary, uint32_t bytes, nn::ahid::Ahid *pAhid) NN_NOEXCEPT
{
    Result      result;
    uint8_t     imageCount;
    DfuFile     dfuFile;

    m_DfuUpdateStatus.Initialize();

    result = dfuFile.Initialize(pBinary, bytes, NULL, NULL, NULL, &imageCount);

    if (result.IsSuccess())
    {
        AhidDevice ahid;

        result = ahid.ChangeToDfuDevice(pAhid);

        // The DFU device has a battery. This means if someone unplugs the
        // device in the middle an update it will leave the device as a DFU
        // device. So we should try to find the DFU device regardless of if we
        // found the AHID device.
        //if (result.IsSuccess())
        {
            DfuDevice dfu;

            // Sleep here for awhile to allow DFU device to enumerate
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));

            result = dfu.PerformUpdate(0x057e, 0x200f, 0x0200, &dfuFile, imageCount, &m_DfuUpdateStatus);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result StDfu::GetUpdateStatus(UpdateStatus *pUpdateStatus) NN_NOEXCEPT
{
    m_DfuUpdateStatus.GetUpdateStatus(pUpdateStatus);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
} // end of namespace stdfu
} // end of namespace nn
