﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include <nn/socket.h>
#include <nn/sf/sf_ISharedObject.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/ssl_Connection.h>

#include <nn/ssl/detail/ssl_ISslContext.h>
#include <nn/ssl/detail/ssl_ISslConnection.h>

#include "detail/ssl_CertChain.h"
#include "detail/ssl_ServiceSession.h"
#include "server/ssl_MemoryManager.h"
#include "server/ssl_Util.h"


using namespace nn::sf;
using namespace nn::ssl::sf;
using namespace nn::ssl::detail;

namespace nn { namespace ssl {

// -------------------------------------------------------------------------------------------------
// Connection
// -------------------------------------------------------------------------------------------------
Connection::Connection() NN_NOEXCEPT :
    m_ConnectionId(0),
    m_ContextId(0),
    m_SockToClose(-1),
    m_CertBuf(nullptr),
    m_CertBufLen(0),
    m_IoLastError(nn::ResultSuccess())
{
}

Connection::~Connection() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ConnectionId == 0);
}


nn::Result Connection::Create(Context* pInSslContext) NN_NOEXCEPT
{
    nn::Result                      result = ResultSuccess();
    SharedPointer<ISslConnection>   *newConn = nullptr;
    SharedPointer<ISslContext>      *ctx = nullptr;
    SslContextId                    ctxId;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pInSslContext == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        result = pInSslContext->GetContextId(&ctxId);
        if (result.IsFailure())
        {
            result = ResultInvalidContext();
            break;
        }

        NN_DETAIL_SSL_GET_PTR_FROM_ID(ctx, ctxId, SharedPointer<ISslContext>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(ctx, result, ResultInvalidContext());

        void *ptr = SslMemoryManager::AllocateChunk(sizeof(SharedPointer<ISslContext>), 0);
        if (ptr == nullptr)
        {
            result = ResultInsufficientMemory();
            break;
        }

        newConn = new(ptr) SharedPointer<ISslConnection>();

        result = (*ctx)->CreateConnection(newConn);
        if (result.IsFailure())
        {
            break;
        }

        m_ContextId = ctxId;
        NN_DETAIL_SSL_GET_ID_FROM_PTR(m_ConnectionId, newConn);
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure() && (newConn != nullptr))
    {
        newConn->~SharedPointer<ISslConnection>();
        SslMemoryManager::Free(newConn, 0);
    }

    return result;
}

nn::Result Connection::Destroy() NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        if (m_SockToClose >= 0)
        {
            NN_DETAIL_SSL_DBG_PRINT("[Connection::Destroy] closing client socket %d\n", m_SockToClose);
            nn::socket::Close(m_SockToClose);
            m_SockToClose = -1;
        }

        NN_DETAIL_SSL_DBG_PRINT("[Connection::Destroy] cleaning up\n");
        m_ConnectionId = 0;
        m_ContextId    = 0;
        m_CertBuf      = nullptr;
        m_CertBufLen   = 0;
        m_IoLastError  = nn::ResultSuccess();
        *conn          = nullptr;
        conn->~SharedPointer<ISslConnection>();
        SslMemoryManager::Free(conn, 0);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetSocketDescriptor(int socketDescriptor) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        result = (*conn)->SetSocketDescriptor(socketDescriptor, &m_SockToClose);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetHostName(const char* pInHostName, uint32_t hostNameLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pInHostName == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        InBuffer hostBuf(pInHostName, hostNameLength);
        result = (*conn)->SetHostName(hostBuf);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetVerifyOption(VerifyOption optionValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::VerifyOption opts;
        opts.options = static_cast<uint32_t>(optionValue);
        result = (*conn)->SetVerifyOption(opts);
        NN_SDK_ASSERT(nn::ssl::ResultOperationNotAllowed::Includes(result) == false,
                      "Cannot disable (Current Connection::OptionType_SkipDefaultVerify = false\n");
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetServerCertBuffer(char* pOutBuffer, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if ((pOutBuffer == nullptr) && (m_CertBuf == nullptr))
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        //  Only allow a buffer to be set 1x or cleared
        if ((pOutBuffer != nullptr) && (m_CertBuf != nullptr))
        {
            result = ResultBufferAlreadyRegistered();
            break;
        }

        m_CertBuf = pOutBuffer;
        m_CertBufLen = bufferLength;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetIoMode(IoMode mode) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::IoMode sfMode;
        sfMode.mode = static_cast<uint32_t>(mode);
        result = (*conn)->SetIoMode(sfMode);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetSessionCacheMode(SessionCacheMode mode) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::SessionCacheMode sfMode;
        sfMode.mode = static_cast<uint32_t>(mode);
        result = (*conn)->SetSessionCacheMode(sfMode);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetRenegotiationMode(RenegotiationMode mode) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::RenegotiationMode sfMode;
        sfMode.mode = static_cast<uint32_t>(mode);
        result = (*conn)->SetRenegotiationMode(sfMode);
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result Connection::GetSocketDescriptor(int* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        result = (*conn)->GetSocketDescriptor(pOutValue);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetHostName(char* pOutHostNameBuffer, uint32_t* pOutHostNameLength, uint32_t hostNameBufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if ((pOutHostNameBuffer == nullptr) || (pOutHostNameLength == nullptr))
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        OutBuffer sfBuf(pOutHostNameBuffer, hostNameBufferLength);
        result = (*conn)->GetHostName(sfBuf, pOutHostNameLength);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetVerifyOption(VerifyOption* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::VerifyOption opt;
        result = (*conn)->GetVerifyOption(&opt);
        if (result.IsSuccess())
        {
            *pOutValue = static_cast<VerifyOption>(opt.options);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetIoMode(IoMode* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::IoMode mode;
        result = (*conn)->GetIoMode(&mode);
        if (result.IsSuccess())
        {
            *pOutValue = static_cast<IoMode>(mode.mode);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetSessionCacheMode(SessionCacheMode* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::SessionCacheMode sfMode;
        result = (*conn)->GetSessionCacheMode(&sfMode);
        if (result.IsSuccess())
        {
            *pOutValue = static_cast<SessionCacheMode>(sfMode.mode);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetRenegotiationMode(RenegotiationMode* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::RenegotiationMode sfMode;
        result = (*conn)->GetRenegotiationMode(&sfMode);
        if (result.IsSuccess())
        {
            *pOutValue = static_cast<RenegotiationMode>(sfMode.mode);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::FlushSessionCache() NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        result = (*conn)->FlushSessionCache();
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::DoHandshake() NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        result = (*conn)->DoHandshake();
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::DoHandshake(uint32_t* pOutServerCertSize, uint32_t* pOutServerCertCount) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        //  Call through to the overloaded version, providing the buffer and
        //  buffer length previously provided by the caller.
        result = DoHandshake(pOutServerCertSize,
                             pOutServerCertCount,
                             m_CertBuf,
                             m_CertBufLen);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::DoHandshake(uint32_t  *pOutCertSize,
                                   uint32_t  *pOutServerCertCount,
                                   char      *pOutBuffer,
                                   uint32_t  bufferLen)
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        //  If no buffer has previously been provided, just fall back to handshake
        //  with no cert copy.
        if ((pOutBuffer == nullptr) || (bufferLen == 0))
        {
#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT_IO
            NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] no buf provided, drop to simple\n");
#endif
            result = DoHandshake();
            if (pOutCertSize != nullptr)
            {
                *pOutCertSize = 0;
            }

            if (pOutServerCertCount != nullptr)
            {
                *pOutServerCertCount = 0;
            }
        }
        else
        {
#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT_IO
            NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] calling w/user buf %p, %d\n",
                                    pOutBuffer,
                                    bufferLen);
#endif
            OutBuffer buf(pOutBuffer, bufferLen);
            uint32_t certSize = 0;
            uint32_t certCount = 0;
            result = (*conn)->DoHandshakeGetServerCert(&certSize,
                                                       &certCount,
                                                       buf);
            if (result.IsSuccess())
            {
                 if (pOutCertSize != nullptr)
                 {
                     *pOutCertSize = certSize;
                 }

                 if (pOutServerCertCount != nullptr)
                 {

                     *pOutServerCertCount = certCount;
                 }
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetServerCertDetail(ServerCertDetail  *pOutCertDetail,
                                           const char        *pInBuffer,
                                           uint32_t          index)
{
    nn::Result                  result = nn::ResultSuccess();
    const CertChainInfo         *pCertChainInfo = nullptr;
    const CertChainEntry        *pCertEntries = nullptr;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutCertDetail == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetServerCertDetail] no ouput buffer provided\n");
            result = ResultInvalidPointer();
            break;
        }

        if (pInBuffer == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetServerCertDetail] no input buffer provided\n");
            result = ResultInvalidPointer();
            break;
        }

        pCertChainInfo = reinterpret_cast<const CertChainInfo *>(pInBuffer);
        if (pCertChainInfo->magicNum != nn::ssl::detail::CertChainMagicNum)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetServerCertDetail] no chain found in buffer\n");
            result = ResultNoServerChain();
            break;
        }

        if (index >= pCertChainInfo->certCount)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetServerCertDetail] invalid index %u specified. Max is %u.\n",
                                    index,
                                    pCertChainInfo->certCount);
            result = ResultInvalidIndex();
            break;
        }

        //  The certs each get an entry after the info area.  Each entry
        //  contains where to find the cert data itself within the provided
        //  buffer.  The offset is taken from the start of the buffer.
        pCertEntries = reinterpret_cast<const CertChainEntry *>(pCertChainInfo + 1);
        pOutCertDetail->dataSize = pCertEntries[index].len;
        pOutCertDetail->pDerData = pInBuffer + pCertEntries[index].offset;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


int Connection::Read(char* pOutBuffer, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    int                         ret;

    do
    {
        result = Read(pOutBuffer, &ret, bufferLength);
        if (result.IsFailure())
        {
            ret = -1;
        }
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result Connection::Read(char* pOutBuffer, int* pOutReadSizeCourier, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    bytesRead;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutBuffer == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (pOutReadSizeCourier == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (bufferLength == 0)
        {
            result = ResultBufferTooShort();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        OutBuffer buf(pOutBuffer, bufferLength);
        result = (*conn)->Read(&bytesRead, buf);
        if (result.IsSuccess())
        {
            *pOutReadSizeCourier = static_cast<int>(bytesRead);
        }
        else
        {
            *pOutReadSizeCourier = -1;
        }
    } while (NN_STATIC_CONDITION(false));

    m_IoLastError = result;
    return result;
}


int Connection::Write(const char* pInBuffer, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    int                         ret;

    do
    {
        result = Write(pInBuffer, &ret, bufferLength);
        if (result.IsFailure())
        {
            ret = -1;
        }
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result Connection::Write(const char* pInBuffer, int* pOutWrittenSizeCourier, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    bytesWritten;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pInBuffer == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (pOutWrittenSizeCourier == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (bufferLength == 0)
        {
            result = ResultBufferTooShort();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        InBuffer buf(pInBuffer, bufferLength);
        result = (*conn)->Write(&bytesWritten, buf);
        if (result.IsSuccess())
        {
            *pOutWrittenSizeCourier = static_cast<int>(bytesWritten);
        }
        else
        {
            *pOutWrittenSizeCourier = -1;
        }
    } while (NN_STATIC_CONDITION(false));

    m_IoLastError = result;
    return result;
}


int Connection::Pending() NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    int                         ret;

    do
    {
        result = Pending(&ret);
        if (result.IsFailure())
        {
            ret = -1;
        }
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result Connection::Pending(int* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        result = (*conn)->Pending(pOutValue);
    } while (NN_STATIC_CONDITION(false));

    m_IoLastError = result;
    return result;
}


nn::Result Connection::Peek(char* pOutBuffer, int* pOutReadSizeCourier, uint32_t bufferLength) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    byteCount;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutBuffer == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (pOutReadSizeCourier == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (bufferLength == 0)
        {
            result = ResultBufferTooShort();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        OutBuffer buf(pOutBuffer, bufferLength);
        result = (*conn)->Peek(&byteCount, buf);
        if (result.IsSuccess())
        {
            *pOutReadSizeCourier = static_cast<int>(byteCount);
        }
    } while (NN_STATIC_CONDITION(false));

    m_IoLastError = result;
    return result;
}


nn::Result Connection::Poll(PollEvent* pOutEvent, PollEvent* pInEvent, uint32_t msecTimeout) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();
    nn::ssl::sf::PollEvent      inEvt;
    nn::ssl::sf::PollEvent      outEvt;

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutEvent == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        if (pInEvent == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        inEvt.events = static_cast<uint32_t>(*pInEvent);
        outEvt.events = 0;
        result = (*conn)->Poll(&outEvt, inEvt, msecTimeout);
        if (result.IsSuccess())
        {
            *pOutEvent = static_cast<PollEvent>(outEvt.events);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetLastError(nn::Result* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        *pOutValue = m_IoLastError;
        m_IoLastError = ResultSuccess();
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetVerifyCertError(nn::Result* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        *pOutValue = (*conn)->GetVerifyCertError();
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result Connection::GetVerifyCertErrors(nn::Result* pOutResultArray, uint32_t* pOutResultCountWritten, uint32_t* pOutTotalResultCount, uint32_t resultArrayMaxCount) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        if (pOutResultArray == nullptr || pOutResultCountWritten == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        OutBuffer resultBuf(reinterpret_cast<char*>(pOutResultArray), resultArrayMaxCount * sizeof(nn::Result));
        uint32_t totalResultCount = 0;
        result = (*conn)->GetVerifyCertErrors(resultBuf, pOutResultCountWritten, &totalResultCount);

        if(pOutTotalResultCount != nullptr)
        {
            *pOutTotalResultCount = (result.IsSuccess()) ? totalResultCount : 0;
        }

        // If we succeeded but had more errors than what there was room for in the array.
        if(result.IsSuccess() && totalResultCount != *pOutResultCountWritten)
        {
            result = nn::ssl::ResultBufferTooShort();
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetNeededServerCertBufferSize(uint32_t *pOutValue)
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        result = (*conn)->GetNeededServerCertBufferSize(pOutValue);
    } while (NN_STATIC_CONDITION(false));

    return result;

}


nn::Result Connection::GetContextId(SslContextId* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        *pOutValue = m_ContextId;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetConnectionId(SslConnectionId* pOutValue) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        *pOutValue = m_ConnectionId;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::SetOption(OptionType optionType, bool enable) NN_NOEXCEPT
{
    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::OptionType sfOptionType;
        sfOptionType.type = static_cast<uint32_t>(optionType);
        result = (*conn)->SetOption(sfOptionType, enable);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result Connection::GetOption(bool* pOutIsEnabled, OptionType optionType) NN_NOEXCEPT
{

    nn::Result                  result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutIsEnabled == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;

        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        nn::ssl::sf::OptionType sfOptionType;
        sfOptionType.type = static_cast<uint32_t>(optionType);
        result = (*conn)->GetOption(pOutIsEnabled, sfOptionType);
    } while (NN_STATIC_CONDITION(false));

    return result;
}

Connection::CipherInfo::CipherInfo() NN_NOEXCEPT
{
    memset(cipherName, 0, NameBufLen);
    memset(versionName, 0, VersionBufLen);
}

nn::Result Connection::GetCipherInfo(CipherInfo* pOutValue) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        if (!ServiceSession::IsInitialized())
        {
            result = ResultLibraryNotInitialized();
            break;
        }

        if (pOutValue == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        SharedPointer<ISslConnection>   *conn;
        NN_DETAIL_SSL_GET_PTR_FROM_ID(conn, m_ConnectionId, SharedPointer<ISslConnection>);
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(conn, result, ResultInvalidConnectionContext());

        OutBuffer sfCipherInfoBuf(reinterpret_cast<char*>(pOutValue), sizeof(CipherInfo));
        result = (*conn)->GetCipherInfo(CipherInfo::StructureVersion, sfCipherInfoBuf);
        if (result.IsFailure())
        {
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    // If there is an error, null-terminate the strings
    if(result.IsFailure() && pOutValue != nullptr)
    {
        pOutValue->cipherName[0]  = '\0';
        pOutValue->versionName[0] = '\0';
    }

    return result;
}

}}
