﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ssl_StaticDataReader.h"
#include "ssl_Util.h"

namespace nn { namespace ssl { namespace detail {

//////////////////////////////////////////////////////////////////////////////
// StaticDataReader - Built-in data reader based on BDF located in a C data array.
//////////////////////////////////////////////////////////////////////////////
StaticDataReader::StaticDataReader(const uint8_t *pData, uint32_t dataLen) : m_pData(pData),
                                                                             m_dataLen(dataLen)
{
}


StaticDataReader::~StaticDataReader()
{
}


nn::Result StaticDataReader::Open() NN_NOEXCEPT
{
    nn::Result                  result = ResultSuccess();

    do
    {
        if (m_pData == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[StaticDataReader::Open] invalid id, nothing to open\n");
            result = ResultErrorLower();
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result StaticDataReader::Read(size_t   *pOutBytesRead,
                                 int64_t  offset,
                                 uint8_t  *pOutBuf,
                                 size_t   size) NN_NOEXCEPT
{
    nn::Result                  result = ResultSuccess();
    size_t                      remaining;

    do
    {
        //  Make sure our inputs are good: we have been provided a buffer
        //  and there is enough space to hold all of the requested data.
        if (pOutBytesRead == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[DataArrayReader::Read] null out buffer\n");
            result = ResultInvalidPointer();
            break;
        }

        remaining = static_cast<size_t>(m_dataLen) - static_cast<size_t>(offset);
        if (size > remaining)
        {
            NN_DETAIL_SSL_DBG_PRINT("[DataArrayReader::Read] requested size is more than avail\n");
            result = ResultInsufficientMemory();
            break;
        }

        memcpy(pOutBuf, &m_pData[offset], size);
        *pOutBytesRead = size;
    } while (NN_STATIC_CONDITION(false));

    return result;
}

} } }    //  nn::ssl::detail
