﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/detail/ssl_Common.h>
#include <algorithm> // For std::find

#include "server/ssl_SslSfObservable.h"
#include "server/ssl_Util.h"


namespace nn { namespace ssl { namespace detail {

SslSfObserver::~SslSfObserver() NN_NOEXCEPT
{
}

SslSfObservable::SslSfObservable() NN_NOEXCEPT : m_Lock(false)
{
    NN_DETAIL_SSL_DBG_PRINT("[SslSfObservable] ctor %p\n", this);
}


SslSfObservable::~SslSfObservable() NN_NOEXCEPT
{
    //  Let any observers know that we are being destroyed
    NN_DETAIL_SSL_DBG_PRINT("[~SslSfObservable] dctor %p\n", this);
    NotifyObservers();

    while (m_Observers.size() > 0)
    {
        SslSfObserverListElement* pData = &m_Observers.front();
        m_Observers.pop_front();

        delete pData;
    }
}


void SslSfObservable::NotifyObservers() NN_NOEXCEPT
{
    m_Lock.Lock();

    for (auto iter = m_Observers.begin(); iter != m_Observers.end(); ++iter)
    {
        SslSfObserver& curObserver = iter->Get();
        curObserver.OnDestroy(this);
    }

    m_Lock.Unlock();
}


nn::Result SslSfObservable::AddObserver(SslSfObserver *observer) NN_NOEXCEPT
{
    nn::Result ret = nn::ResultSuccess();

    do
    {
        SslSfObserverListElement* pNewObserver = new SslSfObserverListElement(*observer);
        if(nullptr == pNewObserver)
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddObserver] insufficient memory\n");
            ret = ResultInsufficientMemory();
            break;
        }

        m_Lock.Lock();
        m_Observers.push_back(*pNewObserver);
        m_Lock.Unlock();

    } while(NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result SslSfObservable::RemoveObserver(SslSfObserver *observer) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    m_Lock.Lock();

    do
    {
        auto iterFound = std::find(m_Observers.begin(), m_Observers.end(), SslSfObserverListElement(*observer));
        if(iterFound == m_Observers.end())
        {
            NN_DETAIL_SSL_DBG_PRINT("[AddObserver] Failed to find the observer\n");
            ret = ResultInvalidReference();
            break;
        }

        SslSfObserverListElement* pData = &*iterFound;
        m_Observers.erase(iterFound);
        delete pData;

    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    return ret;
}

SslSfObserverListElement::SslSfObserverListElement(SslSfObserver& observer) NN_NOEXCEPT
    : m_observer(observer) {}

SslSfObserver& SslSfObserverListElement::Get() NN_NOEXCEPT
{
    return m_observer;
}

bool SslSfObserverListElement::operator==(const SslSfObserverListElement& rhs) const NN_NOEXCEPT
{
    return (&m_observer == &rhs.m_observer);
}

bool SslSfObservable::operator==(const SslSfObservable& rhs) const NN_NOEXCEPT
{
    return (this == &rhs);
}

} } }
