﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/ssl/ssl_Connection.h>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>
#include "ssl_CertStore.h"
#include "ssl_MemoryManager.h"
#include "ssl_NssCommon.h"
#include "ssl_NssCore.h"
#include "detail/ssl_ISslServiceFactory.h"


namespace nn { namespace ssl { namespace detail {

class SslConnectionImpl;

class NssUtil
{
public:
    static nn::Result ConvertNssErrorToResult(PRErrorCode nssError);
    static nn::Result ConvertNssErrorToResult(PRErrorCode nssError, uint32_t interfaceVersion);
    static nn::Result ConvertPrErrorToResult(PRErrorCode prError);
    static nn::Result ConvertSslErrorToResult(PRErrorCode sslError);
    static nn::Result ConvertSecErrorToResult(PRErrorCode secError);

    static nn::Result GetDecryptedPendingBytes(PRFileDesc* pInDescriptor, uint32_t* pOutPendingBytes);
    static void DumpSupportedSslVersions();
    static void DumpPublicKeyInCert(CERTCertificate* pInCert);
    static void DumpDerCertificate(CERTCertificate* pInCert);

    //  Caller is responsible for calling delete[] on the return pointer
    static char* MakeIdString(uint64_t id);

    //  Caller must call DestroyRandIdName when done with the return pointer
    static char *CreateRandIdName(const char *prefix, uint32_t minBufSize);
    static void DestroyRandIdName(char *id);

    class ErrorList;

    class NssConverter
    {
    public:
        static nn::Result DerDataToPublicKey(char* pInDerKey, uint32_t dataLength);
        static nn::Result DerDataToCertificate(char* pInDerCert, uint32_t dataLength);
    };

    class CertTool
    {
    private:
        static const uint32_t m_sTimeStrLength = 128;

        struct SslChainVerifyArgs
        {
            SslConnectionImpl       *pConn;
            PRErrorCode             chainVerifyErr;
        };

        class VerifyLogger
        {
        private:
            static const uint32_t m_sArenaSize = 512;
            PRArenaPool*          m_pArena;
            CERTVerifyLog*        m_pLogger;
        public:
            VerifyLogger() : m_pArena(nullptr), m_pLogger(nullptr) {}
            ~VerifyLogger(){}

            nn::Result     Initialize();
            nn::Result     Finalize();
            CERTVerifyLog* GetCERTVerifyLog();
            void           Dump(CertTool* pInCertTool);
        };

        nn::Result SetupRevocation(PLArenaPool         *pArena,
                                   CERTRevocationFlags **pOutRevFlags);

        nn::Result SetupCrl(PLArenaPool         *pArena,
                            CERTRevocationTests *pTests);

        static nn::os::Mutex    g_VerifyLock;
        static NssCore::BgTask  g_CleanPkixHashTableTask;
        static void ScheduleCleanPkixHashTables();
        static void CleanPkixHashTables(void *pItemDetails);

        static SECStatus SslChainVerifyCb(void               *pArg,
                                          const CERTCertList *pCurrentChain,
                                          PRBool             *pChainOk);

    public:
        void DumpCertificateInformation(CERTCertificate* pInCert);
        nn::Result VerifyCertificate(
            CERTCertificate*   pInServerCert,
            PRInt64            dateToCheck,
            SslConnectionImpl* pInConnection);
        nn::Result VerifyName(
            CERTCertificate* pInServerCert,
            PRFileDesc*      pInSocket,
            char*            pInHostName,
            PRErrorCode*     pOutErrorCode);
        nn::Result VerifyDate(CERTCertificate* pInServerCert);
    };

    struct ObjListNode
    {
        PRCList                 links;
        void                    *pObj;
    };
};

class NssUtil::ErrorList
{
private:
    NN_DISALLOW_COPY(ErrorList);

public:
    struct ErrorNode;

    ErrorList() NN_NOEXCEPT;
    ~ErrorList() NN_NOEXCEPT;

    bool AddError(PRErrorCode error) NN_NOEXCEPT;
    void ClearErrors() NN_NOEXCEPT;
    const ErrorNode* GetHead() const NN_NOEXCEPT;

private:
    ErrorNode* m_pHead;
};

struct NssUtil::ErrorList::ErrorNode
{
    ErrorNode() NN_NOEXCEPT;

    PRErrorCode error;
    ErrorNode* pNext;
};

}}} // namespace nn { namespace ssl { namespace detail {
