﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/ssl_Connection.h>

#include "ssl_NssCommon.h"
#include "ssl_Util.h"
#include "ssl_NssUtil.h"
#include "ssl_EvCertUtil.h"
#include "ssl_SslConnectionImpl.h"
#include "ssl_SslContextImpl.h"
#include "detail/ssl_ISslServiceFactory.h"

extern "C"
{
//  Pick up PKIX cache hash tables.  These are not defined in a header
//  within PKIX, so they must be declared extern here.
extern PKIX_PL_HashTable *cachedCrlSigTable;
extern PKIX_PL_HashTable *cachedCertSigTable;
extern PKIX_PL_HashTable *cachedCertChainTable;
extern PKIX_PL_HashTable *cachedCertTable;
extern PKIX_PL_HashTable *cachedCrlEntryTable;
extern PKIX_PL_HashTable *aiaConnectionCache;
extern PKIX_PL_HashTable *httpSocketCache;
}

namespace nn { namespace ssl { namespace detail {

// ------------------------------------------------------------------------------------------------
// NssUtil
// ------------------------------------------------------------------------------------------------
nn::Result NssUtil::ConvertNssErrorToResult(PRErrorCode nssError)
{
    nn::Result result = ResultInternalLogicError();
    do
    {
        result = ConvertPrErrorToResult(nssError);
        if(!ResultInternalErrorCodeNotInRange::Includes(result))
        {
            break;
        }

        result = ConvertSslErrorToResult(nssError);
        if(!ResultInternalErrorCodeNotInRange::Includes(result))
        {
            break;
        }

        result = ConvertSecErrorToResult(nssError);
        if(!ResultInternalErrorCodeNotInRange::Includes(result))
        {
            break;
        }
    }while(NN_STATIC_CONDITION(false));

#ifdef NN_DETAIL_SSL_ENABLE_ERROR_PRINT
    NN_DETAIL_SSL_DBG_PRINT("[ConvertNssErrorToResult]NSS Error:%d (%s)-> Result(Desc):%d\n",
        nssError, PR_ErrorToName(nssError), result.GetDescription());
#endif

    return result;
}

nn::Result NssUtil::ConvertNssErrorToResult(PRErrorCode nssError, uint32_t interfaceVersion)
{
    nn::Result result = ConvertNssErrorToResult(nssError);

    if (ResultInternalLogicError::Includes(result))
    {
        // Error handling added in 3.0.0 NUP
        if (interfaceVersion >= ISslServiceFactory::ISslServiceVersionType_Nup300)
        {
            switch(nssError)
            {
                case SSL_ERROR_EXPORT_ONLY_SERVER:
                case SSL_ERROR_US_ONLY_SERVER:
                case SSL_ERROR_NO_CYPHER_OVERLAP:
                    result = ResultSslErrorNoCipher();
                    break;
                default:
                    // Leave it as ResultInternalLogicError()
                    break;
            }
        }
    }

    return result;
}

nn::Result NssUtil::ConvertPrErrorToResult(PRErrorCode prError)
{
    if((prError >= PR_MAX_ERROR) || (prError < PR_NSPR_ERROR_BASE))
    {
        return ResultInternalErrorCodeNotInRange();
    }

    nn::Result result = ResultInternalLogicError();
    switch(prError)
    {
    case PR_OUT_OF_MEMORY_ERROR:
    case PR_INSUFFICIENT_RESOURCES_ERROR:
    case PR_PROC_DESC_TABLE_FULL_ERROR:
        result = ResultInsufficientMemory();
        break;
    case PR_IO_TIMEOUT_ERROR:
        result = ResultIoTimeout();
        break;
    case PR_WOULD_BLOCK_ERROR:
        result = ResultIoWouldBlock();
        break;
    case PR_BAD_DESCRIPTOR_ERROR:
        result = ResultInvalidSocketDescriptor();
        break;
    case PR_NOT_CONNECTED_ERROR:
        result = ResultNoConnection();
        break;
    case PR_CONNECT_RESET_ERROR:
        result = ResultConnectionReset();
        break;
    case PR_OPERATION_ABORTED_ERROR:
        result = ResultIoAborted();
        break;
    case PR_CONNECT_ABORTED_ERROR:
        result = ResultConnectionAborted();
        break;
    case PR_SOCKET_SHUTDOWN_ERROR:
        result = ResultSocketShutdown();
        break;
    case PR_NETWORK_DOWN_ERROR:
        result = ResultNetworkDown();
        break;

    // TODO: should we have dedicate errors for file operation failures?
    //       May occur when something bad happens in persistent storage
    case PR_FILE_NOT_FOUND_ERROR:
    case PR_NOT_DIRECTORY_ERROR:
    case PR_FILESYSTEM_MOUNTED_ERROR:
    case PR_END_OF_FILE_ERROR:
        result = ResultErrorLower();
        break;

    // Errors happen in low level
    case PR_IO_ERROR:
    case PR_BUFFER_OVERFLOW_ERROR:
        result = ResultErrorLower();
        break;

    // Errors should not happen by design
    // TODO: Need to adjust some errors when we internaly create and establish TCP connection
    case PR_ADDRESS_NOT_AVAILABLE_ERROR: // should not happen because connection is established by apps
    case PR_ADDRESS_NOT_SUPPORTED_ERROR: // should not happen because connection is established by apps
    case PR_IS_CONNECTED_ERROR:          // should not happen because connection is established by apps
    case PR_BAD_ADDRESS_ERROR:           // should not happen because connection is established by apps
    case PR_ADDRESS_IN_USE_ERROR:        // should not happen because connection is established by apps
    case PR_CONNECT_REFUSED_ERROR:       // should not happen because connection is established by apps
    case PR_NETWORK_UNREACHABLE_ERROR:   // should not happen because connection is established by apps
    case PR_CONNECT_TIMEOUT_ERROR:       // should not happen because connection is established by apps
    case PR_HOST_UNREACHABLE_ERROR:      // should not happen because connection is established by apps
    case PR_NOT_SOCKET_ERROR:
    case PR_NOT_TCP_SOCKET_ERROR:
    case PR_SOCKET_ADDRESS_IS_BOUND_ERROR:
    case PR_ACCESS_FAULT_ERROR:
    case PR_INVALID_METHOD_ERROR:
    case PR_ILLEGAL_ACCESS_ERROR:
    case PR_PENDING_INTERRUPT_ERROR:
    case PR_NOT_IMPLEMENTED_ERROR:
    case PR_IO_PENDING_ERROR:
    case PR_DIRECTORY_OPEN_ERROR:
    case PR_INVALID_ARGUMENT_ERROR:
    case PR_LOAD_LIBRARY_ERROR:
    case PR_UNLOAD_LIBRARY_ERROR:
    case PR_FIND_SYMBOL_ERROR:
    case PR_DIRECTORY_LOOKUP_ERROR:
    case PR_TPD_RANGE_ERROR:
    case PR_NO_ACCESS_RIGHTS_ERROR:
    case PR_OPERATION_NOT_SUPPORTED_ERROR:
    case PR_PROTOCOL_NOT_SUPPORTED_ERROR:
    case PR_REMOTE_FILE_ERROR:
    case PR_RANGE_ERROR:
    case PR_DEADLOCK_ERROR:
    case PR_FILE_IS_LOCKED_ERROR:
    case PR_FILE_TOO_BIG_ERROR:
    case PR_NO_DEVICE_SPACE_ERROR:
    case PR_PIPE_ERROR:
    case PR_NO_SEEK_DEVICE_ERROR:
    case PR_IS_DIRECTORY_ERROR:
    case PR_LOOP_ERROR:
    case PR_NAME_TOO_LONG_ERROR:
    case PR_READ_ONLY_FILESYSTEM_ERROR:
    case PR_DIRECTORY_NOT_EMPTY_ERROR:
    case PR_NOT_SAME_DEVICE_ERROR:
    case PR_DIRECTORY_CORRUPTED_ERROR:
    case PR_FILE_EXISTS_ERROR:
    case PR_MAX_DIRECTORY_ENTRIES_ERROR:
    case PR_INVALID_DEVICE_STATE_ERROR:
    case PR_DEVICE_IS_LOCKED_ERROR:
    case PR_NO_MORE_FILES_ERROR:
    case PR_FILE_SEEK_ERROR:
    case PR_FILE_IS_BUSY_ERROR:
    case PR_IN_PROGRESS_ERROR:
    case PR_ALREADY_INITIATED_ERROR:
    case PR_GROUP_EMPTY_ERROR:
    case PR_LIBRARY_NOT_LOADED_ERROR:
    case PR_CALL_ONCE_ERROR:
        result = ResultInternalLogicError();
        break;
    default:
        result = ResultInternalLogicError();
        break;
    }
    return result;
} // NOLINT(impl/function_size)

nn::Result NssUtil::ConvertSecErrorToResult(PRErrorCode secError)
{
    if((secError >= SEC_ERROR_END_OF_LIST) || (secError < SEC_ERROR_BASE))
    {
        return ResultInternalErrorCodeNotInRange();
    }

    nn::Result result = ResultInternalLogicError();
    switch(secError)
    {
    case SEC_ERROR_INADEQUATE_KEY_USAGE:
    case SEC_ERROR_INADEQUATE_CERT_TYPE:
        result = ResultSslErrorUnknownCertificate();
        break;
    case SEC_ERROR_UNTRUSTED_CERT:
        result = ResultSslErrorAccessDenied();
        break;
    case SEC_ERROR_CERT_NOT_IN_NAME_SPACE:
    case SEC_ERROR_CA_CERT_INVALID:
    case SEC_ERROR_EXTENSION_VALUE_INVALID:
    case SEC_ERROR_UNKNOWN_CRITICAL_EXTENSION:  // cert_VerifyCertChainOld
    case SEC_ERROR_BAD_SIGNATURE:
    case SEC_ERROR_INVALID_AVA: // when CERT_DecodeAVAValue fails (AVA: attribute value assertion)
        result = ResultSslErrorBadCertificate();
        break;
    case SEC_ERROR_EXPIRED_ISSUER_CERTIFICATE:
        result = ResultSslErrorExpiredCertificate();
        break;
    case SEC_ERROR_EXPIRED_CERTIFICATE:
        result = ResultSslErrorExpiredCertificate();
        break;
    case SEC_ERROR_REVOKED_CERTIFICATE:
        result = ResultSslErrorRevokedCertificate();
        break;
    case SEC_ERROR_UNKNOWN_ISSUER:
    case SEC_ERROR_UNTRUSTED_ISSUER:
        result = ResultSslErrorUnkownCa();
        break;
    case SEC_ERROR_BAD_DER:
        result = ResultSslErrorUnsupportedCertificate();
        break;
    case SEC_ERROR_NO_MEMORY:
        result = ResultInsufficientMemory();
        break;
    case SEC_ERROR_UNSUPPORTED_KEYALG:
        result = ResultSslErrorReceivedUnexpectedHandshakeData();
        break;
    case SEC_ERROR_APPLICATION_CALLBACK_ERROR:
        //  This is returned if our custom cert validation check fails.
        result = ResultSslErrorUnkownCa();
        break;
    case SEC_ERROR_EXTENSION_NOT_FOUND:
        result = ResultSslErrorNoExtendedField();
        break;
    case SEC_ERROR_UNRECOGNIZED_OID:
        result = ResultSslErrorUntrustedOid();
        break;
    case SEC_ERROR_POLICY_VALIDATION_FAILED:
        result = ResultSslErrorInvalidPolicy();
        break;
    // --------------------------------------------------------------------------------------------
    // Errors in lower level
    // --------------------------------------------------------------------------------------------
    case SEC_ERROR_IO:
    case SEC_ERROR_LIBRARY_FAILURE:
    case SEC_ERROR_BAD_DATA:
    case SEC_ERROR_INVALID_ARGS:
    case SEC_ERROR_INVALID_ALGORITHM:
    case SEC_ERROR_OUTPUT_LEN:
    case SEC_ERROR_INPUT_LEN:
    case SEC_ERROR_INVALID_TIME:
    case SEC_ERROR_INVALID_KEY:
        result = ResultErrorLower();
        break;
    // --------------------------------------------------------------------------------------------
    // CRL
    // --------------------------------------------------------------------------------------------
    // TODO: need to review again when we support CRL
    case SEC_ERROR_CRL_EXPIRED:
    case SEC_ERROR_CRL_BAD_SIGNATURE:
    case SEC_ERROR_CRL_INVALID:
        result = ResultInternalLogicError();
        break;
    // --------------------------------------------------------------------------------------------
    // OCSP
    // --------------------------------------------------------------------------------------------
    // TODO: need to review again when we support OCSP
    case SEC_ERROR_UNKNOWN_CERT:
    case SEC_ERROR_UNKNOWN_SIGNER:
    case SEC_ERROR_CERT_BAD_ACCESS_LOCATION:
    case SEC_ERROR_OCSP_UNKNOWN_RESPONSE_TYPE:
    case SEC_ERROR_OCSP_BAD_HTTP_RESPONSE:
    case SEC_ERROR_OCSP_MALFORMED_REQUEST:
    case SEC_ERROR_OCSP_SERVER_ERROR:
    case SEC_ERROR_OCSP_TRY_SERVER_LATER:
    case SEC_ERROR_OCSP_REQUEST_NEEDS_SIG:
    case SEC_ERROR_OCSP_UNAUTHORIZED_REQUEST:
    case SEC_ERROR_OCSP_UNKNOWN_RESPONSE_STATUS:
    case SEC_ERROR_OCSP_UNKNOWN_CERT:
    case SEC_ERROR_OCSP_NOT_ENABLED:
    case SEC_ERROR_OCSP_NO_DEFAULT_RESPONDER:
    case SEC_ERROR_OCSP_MALFORMED_RESPONSE:
    case SEC_ERROR_OCSP_UNAUTHORIZED_RESPONSE:
    case SEC_ERROR_OCSP_FUTURE_RESPONSE:
    case SEC_ERROR_OCSP_OLD_RESPONSE:
    case SEC_ERROR_REVOKED_CERTIFICATE_OCSP:
    case SEC_ERROR_OCSP_RESPONDER_CERT_INVALID:
    case SEC_ERROR_OCSP_BAD_SIGNATURE:
        result = ResultInternalLogicError();
        break;
    // --------------------------------------------------------------------------------------------
    // PKCS12
    // --------------------------------------------------------------------------------------------
    // TODO: need to review again when we use PKCS12
    case SEC_ERROR_BAD_EXPORT_ALGORITHM:
    case SEC_ERROR_EXPORTING_CERTIFICATES:
    case SEC_ERROR_IMPORTING_CERTIFICATES:
    case SEC_ERROR_PKCS12_DECODING_PFX:
    case SEC_ERROR_PKCS12_INVALID_MAC:
    case SEC_ERROR_PKCS12_UNSUPPORTED_MAC_ALGORITHM:
    case SEC_ERROR_PKCS12_UNSUPPORTED_TRANSPORT_MODE:
    case SEC_ERROR_PKCS12_CORRUPT_PFX_STRUCTURE:
    case SEC_ERROR_PKCS12_UNSUPPORTED_PBE_ALGORITHM:
    case SEC_ERROR_PKCS12_UNSUPPORTED_VERSION:
    case SEC_ERROR_PKCS12_PRIVACY_PASSWORD_INCORRECT:
    case SEC_ERROR_PKCS12_CERT_COLLISION:
    case SEC_ERROR_USER_CANCELLED:
    case SEC_ERROR_PKCS12_DUPLICATE_DATA:
    case SEC_ERROR_PKCS12_UNABLE_TO_IMPORT_KEY:
    case SEC_ERROR_PKCS12_IMPORTING_CERT_CHAIN:
    case SEC_ERROR_PKCS12_UNABLE_TO_LOCATE_OBJECT_BY_NAME:
    case SEC_ERROR_PKCS12_UNABLE_TO_EXPORT_KEY:
    case SEC_ERROR_PKCS12_UNABLE_TO_WRITE:
    case SEC_ERROR_PKCS12_UNABLE_TO_READ:
    case SEC_ERROR_PKCS12_KEY_DATABASE_NOT_INITIALIZED:
        result = ResultInternalLogicError();
        break;
    // --------------------------------------------------------------------------------------------
    // Errors considered as internal logic error (should not happen by design)
    // --------------------------------------------------------------------------------------------
    case SEC_ERROR_NOT_A_RECIPIENT: //pkcs7
    case SEC_ERROR_PKCS7_KEYALG_MISMATCH:
    case SEC_ERROR_PKCS7_BAD_SIGNATURE:
    case SEC_ERROR_DECRYPTION_DISALLOWED:
    case SEC_ERROR_ADDING_CERT: // when PK11_ImportCert, PK11_ImportCertForKey or __CERT_AddTempCertToPerm fails
    case SEC_ERROR_NO_KEY: // in PK11_MatchItem
    case SEC_ERROR_CERT_NOT_VALID:  // when PKIX_ProcessingParams_Create or PKIX_PL_NssContext_Create in CERT_PKIXVerifyCert
    case SEC_ERROR_OLD_CRL: // in crl_storeCRL
    case SEC_ERROR_TOKEN_NOT_LOGGED_IN:
    case SEC_ERROR_OUT_OF_SEARCH_LIMITS:
    case SEC_ERROR_INVALID_POLICY_MAPPING:
    case SEC_ERROR_NOT_INITIALIZED:
    case SEC_ERROR_CRL_ALREADY_EXISTS:
    case SEC_ERROR_UNKNOWN_OBJECT_TYPE:
    case SEC_ERROR_INCOMPATIBLE_PKCS11:
    case SEC_ERROR_NO_EVENT:
    case SEC_ERROR_REVOKED_CERTIFICATE_CRL: // not used
    case SEC_ERROR_CRL_INVALID_VERSION: // such the one shuold not be passed
    case SEC_ERROR_CRL_V1_CRITICAL_EXTENSION: // such the one shuold not be passed
    case SEC_ERROR_CRL_UNKNOWN_CRITICAL_EXTENSION: // such the one shuold not be passed
    case SEC_ERROR_BAD_KEY:
    case SEC_ERROR_BAD_PASSWORD:
    case SEC_ERROR_RETRY_PASSWORD: // not used
    case SEC_ERROR_NO_NODELOCK: // not used
    case SEC_ERROR_BAD_DATABASE:
    case SEC_ERROR_CERT_VALID: // used only in vfyutil
    case SEC_ERROR_CERT_NO_RESPONSE: // not used
    case SEC_ERROR_PATH_LEN_CONSTRAINT_INVALID: // in cert_VerifyCertChainOld
    case SEC_ERROR_CERT_USAGES_INVALID: // in CERT_CheckCertUsage
    case SEC_INTERNAL_ONLY: // not used
    case SEC_ERROR_NO_EMAIL_CERT: // not used
    case SEC_ERROR_NO_RECIPIENT_CERTS_QUERY: // not used
    case XP_SEC_FORTEZZA_BAD_CARD:          // not used
    case XP_SEC_FORTEZZA_NO_CARD:           // not used
    case XP_SEC_FORTEZZA_NONE_SELECTED:     // not used
    case XP_SEC_FORTEZZA_MORE_INFO:         // not used
    case XP_SEC_FORTEZZA_PERSON_NOT_FOUND:  // not used
    case XP_SEC_FORTEZZA_NO_MORE_INFO:      // not used
    case XP_SEC_FORTEZZA_BAD_PIN:           // not used
    case XP_SEC_FORTEZZA_PERSON_ERROR:      // not used
    case SEC_ERROR_NO_KRL:
    case SEC_ERROR_KRL_EXPIRED:
    case SEC_ERROR_KRL_BAD_SIGNATURE:
    case SEC_ERROR_REVOKED_KEY:
    case SEC_ERROR_KRL_INVALID:
    case SEC_ERROR_NEED_RANDOM:
    case SEC_ERROR_NO_MODULE:
    case SEC_ERROR_NO_TOKEN:
    case SEC_ERROR_READ_ONLY:
    case SEC_ERROR_NO_SLOT_SELECTED:
    case SEC_ERROR_CERT_NICKNAME_COLLISION:
    case SEC_ERROR_KEY_NICKNAME_COLLISION:
    case SEC_ERROR_SAFE_NOT_CREATED:
    case SEC_ERROR_BAGGAGE_NOT_CREATED:
    case XP_JAVA_REMOVE_PRINCIPAL_ERROR:
    case XP_JAVA_DELETE_PRIVILEGE_ERROR:
    case XP_JAVA_CERT_NOT_EXISTS_ERROR:
    case SEC_ERROR_MESSAGE_SEND_ABORTED: // not used
    case SEC_ERROR_KEYGEN_FAIL:
    case SEC_ERROR_INVALID_PASSWORD:
    case SEC_ERROR_RETRY_OLD_PASSWORD:
    case SEC_ERROR_BAD_NICKNAME:
    case SEC_ERROR_NOT_FORTEZZA_ISSUER:
    case SEC_ERROR_CANNOT_MOVE_SENSITIVE_KEY:
    case SEC_ERROR_JS_INVALID_MODULE_NAME:
    case SEC_ERROR_JS_INVALID_DLL:
    case SEC_ERROR_JS_ADD_MOD_FAILURE:
    case SEC_ERROR_JS_DEL_MOD_FAILURE:
    case SEC_ERROR_OLD_KRL: // not used
    case SEC_ERROR_CKL_CONFLICT: // not used
    case SEC_ERROR_KRL_NOT_YET_VALID: // not used
    case SEC_ERROR_CRL_NOT_YET_VALID: // not used
    /* smime stuff */
    case SEC_ERROR_DIGEST_NOT_FOUND:
    case SEC_ERROR_UNSUPPORTED_MESSAGE_TYPE:
    case SEC_ERROR_MODULE_STUCK:
    case SEC_ERROR_BAD_TEMPLATE:
    case SEC_ERROR_CRL_NOT_FOUND:
    case SEC_ERROR_REUSED_ISSUER_AND_SERIAL:
    case SEC_ERROR_BUSY:
    case SEC_ERROR_EXTRA_INPUT:
    /* error codes used by elliptic curve code */
    case SEC_ERROR_UNSUPPORTED_ELLIPTIC_CURVE:
    case SEC_ERROR_UNSUPPORTED_EC_POINT_FORM:
    case SEC_ERROR_OCSP_INVALID_SIGNING_CERT:
    /* No longer used. Unknown AIA location types are now silently ignored. */
    case SEC_ERROR_UNKNOWN_AIA_LOCATION_TYPE:
    case SEC_ERROR_BAD_HTTP_RESPONSE:
    case SEC_ERROR_BAD_LDAP_RESPONSE:
    case SEC_ERROR_FAILED_TO_ENCODE_DATA:
    case SEC_ERROR_BAD_INFO_ACCESS_LOCATION:
    case SEC_ERROR_LIBPKIX_INTERNAL:
    case SEC_ERROR_PKCS11_GENERAL_ERROR:
    case SEC_ERROR_PKCS11_FUNCTION_FAILED:
    case SEC_ERROR_PKCS11_DEVICE_ERROR:
    case SEC_ERROR_BAD_INFO_ACCESS_METHOD:
    case SEC_ERROR_CRL_IMPORT_FAILED:
    case SEC_ERROR_EXPIRED_PASSWORD:
    case SEC_ERROR_LOCKED_PASSWORD:
    case SEC_ERROR_UNKNOWN_PKCS11_ERROR:
    case SEC_ERROR_BAD_CRL_DP_URL:
    case SEC_ERROR_CERT_SIGNATURE_ALGORITHM_DISABLED:
    case SEC_ERROR_LEGACY_DATABASE:
        result = ResultInternalLogicError();
    default:
        break; // result remains in ResultInternalLogicError
    }

    return result;
} // NOLINT(impl/function_size)

nn::Result NssUtil::ConvertSslErrorToResult(PRErrorCode sslError)
{
    if((sslError >= SSL_ERROR_END_OF_LIST) || (sslError < SSL_ERROR_BASE))
    {
        return ResultInternalErrorCodeNotInRange();
    }

    nn::Result result = ResultInternalLogicError();
    switch(sslError)
    {
    case SSL_ERROR_NO_CERTIFICATE:                      // when SSL_PeerCertificateChain fails to obtain peer cert
        result = ResultSslErrorNoCertificate();
        break;
    case SSL_ERROR_DECOMPRESSION_FAILURE:               // when sending DECOMPRESSION_FAILURE or BAD_RECORD_MAC
        result = ResultSslErrorDecompressionFailed();
        break;
    case SSL_ERROR_WEAK_SERVER_EPHEMERAL_DH_KEY:        // when SECKEY_BigIntegerBitLength fails in ssl3_HandleServerKeyExchange
    case SSL_ERROR_WEAK_SERVER_CERT_KEY:                // when ssl3_AuthCertificate fails
        result = ResultSslErrorInsufficientSecurity();
        break;
    case SSL_ERROR_BAD_CERT_DOMAIN:                     // when host name validation fails
        result = ResultSslErrorInvalidCertDomain();
        break;
    case SSL_ERROR_RX_UNEXPECTED_CERT_STATUS:           // when ssl3_HandleCertificateStatus fails if wait_certificate_status is not available
        result = ResultSslErrorUnexpectedCertificateStatusResponse();
        break;
    case SSL_ERROR_INCORRECT_SIGNATURE_ALGORITHM:       // when ssl3_CheckSignatureAndHashAlgorithmConsistency fails fonr inconsistent algorithm
        result = ResultSslErrorInconsistentSignAlgorithm();
        break;
    case SSL_ERROR_RX_RECORD_TOO_LONG:                  // returned when sending RECORD_OVERFLOW alert
        result = ResultSslErrorReceivedTooLongRecord();
        break;
    case SSL_ERROR_RX_UNEXPECTED_HELLO_VERIFY_REQUEST:  // when dtls_HandleHelloVerifyRequest fails
        result = ResultSslErrorUnexpectedHelloVerifyRequest();
        break;
    case SSL_ERROR_MD5_DIGEST_FAILURE:                  // when the process fails for MD5
    case SSL_ERROR_SHA_DIGEST_FAILURE:                  // when the process fails for SHA
    case SSL_ERROR_DIGEST_FAILURE:                      // when hash calculation feature fails
        result = ResultSslErrorDigestCalculationFailed();
        break;
    case SSL_ERROR_BAD_MAC_READ:                        // returned when sending BAD_RECORD_MAC alert
        result = ResultSslErrorDecryptionWithInvalidMac();
        break;
    case SSL_ERROR_MAC_COMPUTATION_FAILURE:             // when MAC calculation fails
        result = ResultSslErrorMacCalculationFailed();
        break;
    // --------------------------------------------------------------------------------------------
    // Received malformed data
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_RX_MALFORMED_HELLO_REQUEST:          // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CLIENT_HELLO:           // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_SERVER_HELLO:           // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CERTIFICATE:            // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_SERVER_KEY_EXCH:        // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CERT_REQUEST:           // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_HELLO_DONE:             // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CERT_VERIFY:            // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CLIENT_KEY_EXCH:        // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_FINISHED:               // received malformed SSL handshake
    case SSL_ERROR_RX_MALFORMED_CHANGE_CIPHER:          // received malformed SSL record
    case SSL_ERROR_RX_MALFORMED_ALERT:                  // received malformed SSL record
    case SSL_ERROR_RX_MALFORMED_HANDSHAKE:              // received malformed SSL record
    case SSL_ERROR_RX_MALFORMED_APPLICATION_DATA:       // received malformed SSL record
    case SSL_ERROR_RX_MALFORMED_HELLO_VERIFY_REQUEST:   // when dtls_HandleHelloVerifyRequest fails
        result = ResultSslErrorReceivedMalformedData();
        break;
    // --------------------------------------------------------------------------------------------
    // Received unexpected handshake data
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_RX_UNEXPECTED_HELLO_REQUEST:     // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_CLIENT_HELLO:      // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_SERVER_HELLO:      // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_CERTIFICATE:       // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_SERVER_KEY_EXCH:   // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_CERT_REQUEST:      // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_HELLO_DONE:        // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_CERT_VERIFY:       // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_CLIENT_KEY_EXCH:   // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_FINISHED:          // received unexpected SSL handshake in inappropriate state
    case SSL_ERROR_SSL2_DISABLED:                   // SSLv2, when SSL2 is in Server Hello while SSL v2 is disabled
        result = ResultSslErrorReceivedUnexpectedHandshakeData();
        break;
    // --------------------------------------------------------------------------------------------
    // Received unexpected record data
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_RX_UNEXPECTED_CHANGE_CIPHER:     // received unexpected SSL record in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_ALERT:             // received unexpected SSL record in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_HANDSHAKE:         // received unexpected SSL record in inappropriate state
    case SSL_ERROR_RX_UNEXPECTED_APPLICATION_DATA:  // received unexpected SSL record in inappropriate state
        result = ResultSslErrorReceivedUnexpectedRecordData();
        break;
    // --------------------------------------------------------------------------------------------
    // Received record/message with unknown discriminant
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_RX_UNKNOWN_RECORD_TYPE:          // received record/message with unknown discriminant
    case SSL_ERROR_RX_UNKNOWN_HANDSHAKE:            // received record/message with unknown discriminant
    case SSL_ERROR_RX_UNKNOWN_ALERT:                // received record/message with unknown discriminant
        result = ResultSslErrorReceivedUnkownData();
        break;
    // --------------------------------------------------------------------------------------------
    // SSL alerts
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_BAD_MAC_ALERT:
        result = ResultSslAlertBadRecordMac();
        break;
    case SSL_ERROR_BAD_CERT_ALERT:
        result = ResultSslAlertBadCertificate();
        break;
    case SSL_ERROR_REVOKED_CERT_ALERT:
        result = ResultSslAlertCertificateRevoked();
        break;
    case SSL_ERROR_EXPIRED_CERT_ALERT:
        result = ResultSslAlertCertificateExpired();
        break;
    case SSL_ERROR_CLOSE_NOTIFY_ALERT:
        result = ResultSslAlertCloseNotify();
        break;
    case SSL_ERROR_HANDSHAKE_UNEXPECTED_ALERT:
        result = ResultSslAlertUnexpectedMessage();
        break;
    case SSL_ERROR_DECRYPTION_FAILED_ALERT:
        result = ResultSslAlertDecriptionFailed();
        break;
    case SSL_ERROR_RECORD_OVERFLOW_ALERT:
        result = ResultSslAlertRecordOverfrlow();
        break;
    case SSL_ERROR_DECOMPRESSION_FAILURE_ALERT:
        result = ResultSslAlertDecompressionFailure();
        break;
    case SSL_ERROR_HANDSHAKE_FAILURE_ALERT:
        result = ResultSslAlertHandshakeFailure();
        break;
    case SSL_ERROR_ILLEGAL_PARAMETER_ALERT:
        result = ResultSslAlertIllegalParameter();
        break;
    case SSL_ERROR_UNSUPPORTED_CERT_ALERT:
        result = ResultSslAlertUnsupportedCertificate();
        break;
    case SSL_ERROR_CERTIFICATE_UNKNOWN_ALERT:
        result = ResultSslAlertCertificateUnkown();
        break;
    case SSL_ERROR_UNKNOWN_CA_ALERT:
        result = ResultSslAlertUnknownCa();
        break;
    case SSL_ERROR_ACCESS_DENIED_ALERT:
        result = ResultSslAlertAccessDenied();
        break;
    case SSL_ERROR_DECODE_ERROR_ALERT:
        result = ResultSslAlertDecodeError();
        break;
    case SSL_ERROR_DECRYPT_ERROR_ALERT:
        result = ResultSslAlertDecryptError();
        break;
    case SSL_ERROR_EXPORT_RESTRICTION_ALERT:
        result = ResultSslAlertExportRestriction();
        break;
    case SSL_ERROR_PROTOCOL_VERSION_ALERT:
        result = ResultSslAlertProtocolVersion();
        break;
    case SSL_ERROR_INSUFFICIENT_SECURITY_ALERT:
        result = ResultSslAlertInsufficientSecurity();
        break;
    case SSL_ERROR_INTERNAL_ERROR_ALERT:
        result = ResultSslAlertInternalError();
        break;
    case SSL_ERROR_USER_CANCELED_ALERT:
        result = ResultSslAlertUserCancelled();
        break;
    case SSL_ERROR_NO_RENEGOTIATION_ALERT:
        result = ResultSslAlertNoRenegotiation();
        break;
    case SSL_ERROR_UNSUPPORTED_EXTENSION_ALERT:
        result = ResultSslAlertUnsupportedExtention();
        break;
    case SSL_ERROR_CERTIFICATE_UNOBTAINABLE_ALERT:
        result = ResultSslAlertCertificateUnobtainable();
        break;
    case SSL_ERROR_UNRECOGNIZED_NAME_ALERT:
        result = ResultSslAlertUnrecognizedName();
        break;
    case SSL_ERROR_BAD_CERT_STATUS_RESPONSE_ALERT:
        result = ResultSslAlertBadCertificateStatusResponse();
        break;
    case SSL_ERROR_BAD_CERT_HASH_VALUE_ALERT:
        result = ResultSslAlertBadCertificateHashValue();
        break;
    case SSL_ERROR_INAPPROPRIATE_FALLBACK_ALERT:
        result = ResultSslAlertInappropriateFallback();
        break;
    // --------------------------------------------------------------------------------------------
    // Errors considered as system level error (processes should not fail)
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_GENERATE_RANDOM_FAILURE:           //when random number generation fails
    case SSL_ERROR_ENCRYPTION_FAILURE:                // when encryption fails
    case SSL_ERROR_DECRYPTION_FAILURE:                // when decryption fails
    case SSL_ERROR_SIGN_HASHES_FAILURE:               // when PK11_Digest or SGN_Digest fails
    case SSL_ERROR_EXTRACT_PUBLIC_KEY_FAILURE:        // when fails to extract public key from the certificate
    case SSL_ERROR_CLIENT_KEY_EXCHANGE_FAILURE:       // when cliant key exchange fails
    case SSL_ERROR_SOCKET_WRITE_FAILURE:              // when send fails upon socket
    case SSL_ERROR_SYM_KEY_CONTEXT_FAILURE:           // when creating context for symetric key fails
    case SSL_ERROR_IV_PARAM_FAILURE:                  // when ssl3_ParamFromIV fails
    case SSL_ERROR_SESSION_KEY_GEN_FAILURE:           // when session key generation fails
    case SSL_ERROR_NO_COMPRESSION_OVERLAP:            // when handshake fails due to not found suite
    case SSL_ERROR_HANDSHAKE_NOT_COMPLETED:           // when ssl3_RedoHandshake fails
    case SSL_ERROR_BAD_HANDSHAKE_HASH_VALUE:          // when ssl3_VerifySignedHashes or ssl3_HandleFinished fails
    case SSL_ERROR_NEXT_PROTOCOL_DATA_INVALID:        // when ssl3_ClientHandleNextProtoNegoXtn, ssl3_SelectAppPtorocol or ssl3_ClientHandleAppPtoroXtn fails
        result = ResultErrorLower();
        break;
    // --------------------------------------------------------------------------------------------
    // Errors considered as internal logic error (should not happen by design)
    // --------------------------------------------------------------------------------------------
    case SSL_ERROR_BAD_CERTIFICATE:                   // SSLv2, when failed to handle server cert
    case SSL_ERROR_BAD_CLIENT:                        // when encounters bad configuration for SSL client
    case SSL_ERROR_BAD_SERVER:                        // when encounters bad configuration for SSL server
    case SSL_ERROR_UNSUPPORTED_CERTIFICATE_TYPE:      // SSLv2, when cert type is not SSL_CT_X509_CERTIFICATE
    case SSL_ERROR_UNSUPPORTED_VERSION:               // for the server, when receives unsupported version during negotiation
    case SSL_ERROR_WRONG_CERTIFICATE:                 // not used in the code
    case SSL_ERROR_POST_WARNING:                      // not used in the code
    case SSL_ERROR_SSL_DISABLED:                      // when SSL is not enabled where it is supposed to be enabled
    case SSL_ERROR_UNKNOWN_CIPHER_SUITE:              // when failed to find appropriate ciipher suite config
    case SSL_ERROR_FORTEZZA_PQG:                      // not used in the code
    case SSL_ERROR_NO_CIPHERS_SUPPORTED:              // when ssl3_config_match_init fails to find anything in supported ciphers
    case SSL_ERROR_BAD_BLOCK_PADDING:                 // SSLv2, in ssl2_GatherData
    case SSL_ERROR_TX_RECORD_TOO_LONG:                // not used in the code
    case SSL_ERROR_SERVER_KEY_EXCHANGE_FAILURE:       // when ssl3_SendServerKeyExchange or ssl3_HandleServerKeyExchange fails
    case SSL_ERROR_SYM_KEY_UNWRAP_FAILURE:            // not used in the code
    case SSL_ERROR_PUB_KEY_SIZE_LIMIT_EXCEEDED:       // when key_size_limit is invalid
    case SSL_ERROR_INIT_CIPHER_SUITE_FAILURE:         // not used in the code
    case SSL_ERROR_NO_SERVER_KEY_FOR_ALG:             // when no server key is found in ssl3_HandleClientKeyExchange
    case SSL_ERROR_TOKEN_INSERTION_REMOVAL:           // when corresponding token is not found
    case SSL_ERROR_TOKEN_SLOT_NOT_FOUND:              // when appropriate key slot is not found
    case SSL_ERROR_CERT_KEA_MISMATCH:                 // when NSS_FindCertKEAType fails in SSL_ConfigSecureServerWithCertChain
    case SSL_ERROR_NO_TRUSTED_SSL_CLIENT_CA:          // SSL_ERROR_NO_TRUSTED_SSL_CLIENT_CA became obsolete in NSS 3.14
    case SSL_ERROR_SERVER_CACHE_NOT_CONFIGURED:       // server error
    case SSL_ERROR_RX_UNEXPECTED_NEW_SESSION_TICKET:  // when sending UNEXPECTED_MESSAGE alert (server error)
    case SSL_ERROR_RX_MALFORMED_NEW_SESSION_TICKET:   // when sending DECODE_ERROR (server error)
    case SSL_ERROR_RX_UNEXPECTED_UNCOMPRESSED_RECORD: // when ssl3_HandleRecord fails
    case SSL_ERROR_FEATURE_NOT_SUPPORTED_FOR_SSL2:    // when function not for SSLv2 is called
    case SSL_ERROR_FEATURE_NOT_SUPPORTED_FOR_SERVERS: // when client function is called for the server
    case SSL_ERROR_FEATURE_NOT_SUPPORTED_FOR_CLIENTS: // not used in the code
    case SSL_ERROR_INVALID_VERSION_RANGE:             // when SSL socket is configured with invalid version range
    case SSL_ERROR_CIPHER_DISALLOWED_FOR_VERSION:     // when selected cipher is not valid for the protocol version in ssl3_HandleServerHello
    case SSL_ERROR_FEATURE_NOT_SUPPORTED_FOR_VERSION: // used only in sslinfo.c by SSL_ExportKeyingMaterial
    case SSL_ERROR_UNSUPPORTED_HASH_ALGORITHM:        // when appropriate algorithm for hash calculation is not found
    case SSL_ERROR_NEXT_PROTOCOL_NO_CALLBACK:         // when ssl3_SelectAppProtocol fails, when sending NO_APPLICATION_PROTOCOL
    case SSL_ERROR_NEXT_PROTOCOL_NO_PROTOCOL:         // when ssl3_ClientHandleNextProtoNegoXtn fails, when sending INTERNAL_ERROR
    case SSL_ERROR_RENEGOTIATION_NOT_ALLOWED:         // when sending NO_RENEGOTIATION
    case SSL_ERROR_UNSAFE_NEGOTIATION:                // when ssl3_HandleServerHello fails when renegotiation is not allowed
    case SSL_ERROR_SESSION_NOT_FOUND:                 // when fails to find session, server error
        result = ResultInternalLogicError();
        break;
    default:
        break; // result remains in ResultInternalLogicError
    }

    return result;
} // NOLINT(impl/function_size)

nn::Result NssUtil::GetDecryptedPendingBytes(PRFileDesc* pInDescriptor, uint32_t* pOutPendingBytes)
{
    int32_t rval = SSL_DataPending(pInDescriptor);
    if (rval >= 0)
    {
        *pOutPendingBytes = rval;
        return nn::ResultSuccess();
    }

    return NssUtil::ConvertNssErrorToResult(PR_GetError());
}

//  Get the version range supported by the SSL engine
void NssUtil::DumpSupportedSslVersions()
{
    SSLVersionRange enabledVersions;
    SECStatus       status = SSL_VersionRangeGetSupported(ssl_variant_stream, &enabledVersions);
    if (status != SECSuccess)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::DumpSupportedSslVersions]Failed to get SSL version range: %x\n", status);
    }
    else
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::DumpSupportedSslVersions]Supported SSL versions: %X\n", enabledVersions);
    }
}

void NssUtil::DumpPublicKeyInCert(CERTCertificate* pInCert)
{
    NN_SDK_REQUIRES_NOT_NULL(pInCert);
    if(pInCert == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[DumpPublicKeyInCert]Got invalid pointer.\n");
        return;
    }

    SECKEYPublicKey* pPublicKey = CERT_ExtractPublicKey(pInCert);
    NN_SDK_REQUIRES_NOT_NULL(pPublicKey);
    if(pPublicKey == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[DumpPublicKeyInCert]pPublicKey got invalid pointer.\n");
        return;
    }

    SECItem* pDerPublicKey = SECKEY_EncodeDERSubjectPublicKeyInfo(pPublicKey);
    NN_SDK_REQUIRES_NOT_NULL(pDerPublicKey);
    if(pDerPublicKey == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[DumpPublicKeyInCert]pDerPublicKey got invalid pointer.\n");
        return;
    }

    NN_DETAIL_SSL_DBG_PRINT_RAW("**** DER public key in peer certificate (len:%d) *************\n",
        pDerPublicKey->len);
    NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(pDerPublicKey->data, pDerPublicKey->len);
    NN_DETAIL_SSL_DBG_PRINT_RAW("\n***********************************************************\n");
}

void NssUtil::DumpDerCertificate(CERTCertificate* pInCert)
{
    NN_SDK_REQUIRES_NOT_NULL(pInCert);
    if(pInCert == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[DumpPublicKeyInCert]Got invalid pointer.\n");
        return;
    }

    NN_DETAIL_SSL_DBG_PRINT_RAW("**** Der certificate (len:%d) ********************************\n",
        pInCert->derCert.len);
    NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(pInCert->derCert.data, pInCert->derCert.len);
    NN_DETAIL_SSL_DBG_PRINT_RAW("\n***********************************************************\n");
}


char* NssUtil::MakeIdString(uint64_t id)
{
    char                        *ret = nullptr;
    uint32_t                    _idHiBits = ((id & 0xFFFFFFFF00000000LL) >> 32);
    uint32_t                    _idLoBits = (id & 0x00000000FFFFFFFFLL);

    do
    {
        ret = new char[Util::g_NumCharsUint64Hex + 1];
        if (ret == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[Util::MakeIdString] failed to alloc memory\n");
            break;
        }

        PR_snprintf(ret,
                    Util::g_NumCharsUint64Hex,
                    "%08X%08X",
                    _idHiBits,
                    _idLoBits);
        ret[Util::g_NumCharsUint64Hex] = 0;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


char *NssUtil::CreateRandIdName(const char *pPrefix, uint32_t minBufSize)
{
    char                        *pRet = nullptr;
    PRUint32                    len;
    uint64_t                    randVal = 0;

    do
    {
        if (pPrefix == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CreateRandIdName] no prefix supplied!\n");
            break;
        }

        //  Length needs to total:
        //  prefix_rand\0
        len = PL_strlen(pPrefix) + Util::g_NumCharsUint64Hex + 2;
        len = len > minBufSize ? len : minBufSize;
        pRet = reinterpret_cast<char *>(SslMemoryManager::AllocateChunk(len, 0));
        if (pRet == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CreateRandIdName] failed to alloc chunk\n");
            break;
        }

        PR_GetRandomNoise(&randVal, sizeof(randVal));
        PR_snprintf(pRet, len, "%s_%llx", pPrefix, randVal);
        pRet[len - 1] = '\0';
    } while (NN_STATIC_CONDITION(false));

    return pRet;
}


void NssUtil::DestroyRandIdName(char *pId)
{
    if (pId != nullptr)
    {
        nn::Result result = SslMemoryManager::Free(pId, 0);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::DestroyRandIdName] failed to free chunk: %d-%d\n",
                                    result.GetModule(),
                                    result.GetDescription());
        }
    }
}


// ------------------------------------------------------------------------------------------------
// NssUtil::CertTool::VerifyLogger
// ------------------------------------------------------------------------------------------------
nn::Result NssUtil::CertTool::VerifyLogger::Initialize()
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        m_pArena = PORT_NewArena(m_sArenaSize);
        if(m_pArena == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyLogger::Initialize]Failed to allocate arena.\n");
            result = ResultInsufficientMemory();
            break;
        }

        m_pLogger = static_cast<CERTVerifyLog*>(PORT_ArenaZAlloc(m_pArena, sizeof(CERTVerifyLog)));
        if(m_pLogger == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyLogger::Initialize]Failed to allocate logger.\n");
            result = ResultInsufficientMemory();
            break;
        }

        m_pLogger->arena = m_pArena;
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyLogger::Initialize] Initialized.\n");
    } while(NN_STATIC_CONDITION(false));

    if(result.IsFailure())
    {
        Finalize();
    }

    return result;
}

nn::Result NssUtil::CertTool::VerifyLogger::Finalize()
{
    nn::Result result = nn::ResultSuccess();

    if (m_pArena != nullptr)
    {
        CERTVerifyLogNode       *pLogNode = nullptr;

        for (pLogNode = m_pLogger->head; pLogNode != nullptr; pLogNode = pLogNode->next)
        {
            if (pLogNode->cert != nullptr)
            {
                CERT_DestroyCertificate(pLogNode->cert);
                pLogNode->cert = nullptr;
            }
        }

        PORT_FreeArena(m_pArena, PR_FALSE);
    }

    m_pArena  = nullptr;
    m_pLogger = nullptr;

    return result;
}

CERTVerifyLog* NssUtil::CertTool::VerifyLogger::GetCERTVerifyLog()
{
    return m_pLogger;
}

void NssUtil::CertTool::VerifyLogger::Dump(CertTool* pInCertTool)
{
    if(m_pLogger == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyLogger::Dump]ERROR: VerfifyLogger is not initialized yet.\n");
        return;
    }

    NN_DETAIL_SSL_DBG_PRINT_RAW("\n DUMP VerifyLog *******************************************\n");
    NN_DETAIL_SSL_DBG_PRINT(" Entry count:%d\n", m_pLogger->count);
    CERTVerifyLogNodeStr* pCurrentNode = m_pLogger->head;
    int                   counter = 0;
    while(pCurrentNode != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT_RAW(" ~~~~~~~~~~~\n");
        NN_DETAIL_SSL_DBG_PRINT(" Entry %d: cert(%p) error(%d:%s) depth(%d) arg(%p)\n",
            counter++,
            pCurrentNode->cert,
            pCurrentNode->error,
            PR_ErrorToName(pCurrentNode->error),
            pCurrentNode->depth,
            pCurrentNode->arg);
        pInCertTool->DumpCertificateInformation(pCurrentNode->cert);
        NN_DETAIL_SSL_DBG_PRINT_RAW(" ~~~~~~~~~~~\n");

        pCurrentNode = pCurrentNode->next;
    }
    NN_DETAIL_SSL_DBG_PRINT_RAW(" DUMP VerifyLog done***************************************\n\n");
    NN_UNUSED(counter);
}

// ------------------------------------------------------------------------------------------------
//  NssUtil::CertTool
// ------------------------------------------------------------------------------------------------
void NssUtil::CertTool::DumpCertificateInformation(CERTCertificate* pInCert)
{
    NN_SDK_REQUIRES_NOT_NULL(pInCert);
    if(pInCert == nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::DumpCertificateInformation]Got invalid pointer.\n");
        return;
    }

    PRTime         notBefore;
    PRTime         notAfter;
    PRExplodedTime timeInPrintableFormat;
    char           timeString[m_sTimeStrLength] = {0};

    char* pSubject    = CERT_NameToAscii(&pInCert->subject);
    char* pIssuer     = CERT_NameToAscii(&pInCert->issuer);
    char* pCommonName = CERT_GetCommonName(&pInCert->subject);

    do
    {
        if(CERT_GetCertTimes(pInCert, &notBefore, &notAfter) != SECSuccess)
        {
            NN_DETAIL_SSL_PRINT_CERT_INFO("[NssUtil::CertTool::DumpCertificateInformation]Failed to bet time.\n");
            break;
        }
        NN_DETAIL_SSL_DBG_PRINT_RAW("\n Dump certificate start ***********************************\n");
        NN_DETAIL_SSL_PRINT_CERT_INFO(" subject: %s\n", pSubject);

        PR_ExplodeTime(notBefore, PR_GMTParameters, &timeInPrintableFormat);
        PR_FormatTime(timeString, sizeof(timeString), " %b %d %H:%M:%S %Y GMT", &timeInPrintableFormat);
        NN_DETAIL_SSL_PRINT_CERT_INFO(" start date: %s\n", timeString);

        PR_ExplodeTime(notAfter, PR_GMTParameters, &timeInPrintableFormat);
        PR_FormatTime(timeString, sizeof(timeString), " %b %d %H:%M:%S %Y GMT", &timeInPrintableFormat);
        NN_DETAIL_SSL_PRINT_CERT_INFO(" expire date: %s\n", timeString);

        NN_DETAIL_SSL_PRINT_CERT_INFO(" common name: %s\n", (pCommonName)?pCommonName:"nullptr");
        NN_DETAIL_SSL_PRINT_CERT_INFO(" issuer: %s\n", pIssuer);
        NN_DETAIL_SSL_DBG_PRINT_RAW(" Dump certificate end *************************************\n\n");
    } while (NN_STATIC_CONDITION(false));

    if(pSubject)    PORT_Free(pSubject);
    if(pIssuer)     PORT_Free(pIssuer);
    if(pCommonName) PORT_Free(pCommonName);
}


nn::Result NssUtil::CertTool::SetupRevocation(PLArenaPool         *pArena,
                                              CERTRevocationFlags **pOutRevFlags)
{
    nn::Result                  ret = nn::ResultSuccess();
    CERTRevocationFlags         *pRevFlags;

    do
    {
        //  Create a revocation flag object, init to 0
        pRevFlags = PORT_ArenaZNew(pArena, CERTRevocationFlags);
        if (pRevFlags == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetupRevocation] failed to alloc mem for rev flag obj\n");
            ret = ResultInsufficientMemory();
            break;
        }

        //  Enable CRL checking support for chain and leaf tests
        //  TODO: Add OCSP support
        ret = SetupCrl(pArena, &pRevFlags->leafTests);
        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetupRevocation] failed to setup CRL for leaf tests\n");
            break;
        }

        ret = SetupCrl(pArena, &pRevFlags->chainTests);
        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetupRevocation] failed to setup CRL for chain tests\n");
            break;
        }

        *pOutRevFlags = pRevFlags;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result NssUtil::CertTool::SetupCrl(PLArenaPool         *pArena,
                                       CERTRevocationTests *pTests)
{
    nn::Result                  ret = ResultSuccess();
    void                        *tmp = nullptr;
    PRUint64                    *pFlagsPerMethod = nullptr;
    CERTRevocationMethodIndex   *pPrefMethods = nullptr;

    do
    {
        if (pTests->cert_rev_flags_per_method == nullptr)
        {
            tmp = PORT_ArenaZAlloc(pArena,
                                   sizeof(PRUint64) * cert_revocation_method_count);
            if (tmp == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetupCrl] failed to alloc mem for rev flags per method\n");
                ret = ResultInsufficientMemory();
                break;
            }

            pFlagsPerMethod = reinterpret_cast<PRUint64 *>(tmp);
            pTests->cert_rev_flags_per_method = pFlagsPerMethod;
        }

        if (pTests->preferred_methods == nullptr)
        {
            tmp = PORT_ArenaZAlloc(pArena,
                                   (sizeof(CERTRevocationMethodIndex) *
                                    cert_revocation_method_count));
            if (tmp == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetupCrl] failed to alloc mem for pref methods\n");
                ret = ResultInsufficientMemory();
                break;
            }

            pPrefMethods = reinterpret_cast<CERTRevocationMethodIndex *>(tmp);
            pTests->preferred_methods = pPrefMethods;
        }

        pFlagsPerMethod[cert_revocation_method_crl] =
            (CERT_REV_M_TEST_USING_THIS_METHOD         |
             CERT_REV_M_FORBID_NETWORK_FETCHING);
        pTests->number_of_defined_methods++;

        pPrefMethods[pTests->number_of_preferred_methods] =
            cert_revocation_method_crl;
        pTests->number_of_preferred_methods++;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::os::Mutex                   NssUtil::CertTool::g_VerifyLock(false);
NssCore::BgTask                 NssUtil::CertTool::g_CleanPkixHashTableTask = 0;

void NssUtil::CertTool::ScheduleCleanPkixHashTables()
{
    nn::Result                  result;

    do
    {
        //  Lazy init/alloc of the bg task.  This does not actually "alloc"
        //  any memory, so no need to "shutdown" or "finalize" later.
        if (NssUtil::CertTool::g_CleanPkixHashTableTask == 0)
        {
            result =
                NssCore::AllocBgTask(&NssUtil::CertTool::g_CleanPkixHashTableTask,
                                     NssUtil::CertTool::CleanPkixHashTables,
                                     nullptr);
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_WARN_PRINT("[NssUtil::CertTool::ScheduleCleanPkixHashTables] failed to alloc bg task\n");
                break;
            }
        }

        //  Schedule the cleaner task to run at the next wake interval of
        //  the bg task thread.
        NssCore::ScheduleBgTask(NssUtil::CertTool::g_CleanPkixHashTableTask);
    } while (NN_STATIC_CONDITION(false));
}


void NssUtil::CertTool::CleanPkixHashTables(void *pItemDetails)
{
    PKIX_Error                  *pErr = nullptr;

    //  Protect the PKIX tables from VerifyCertificate
    NssUtil::CertTool::g_VerifyLock.Lock();

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge cachedCrlSigTable...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(cachedCrlSigTable, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging crl sig table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge cachedCertSigTable...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(cachedCertSigTable, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging cached cert sig table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge cachedCertChainTable...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(cachedCertChainTable, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging cached cert chain table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge cachedCertTable...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(cachedCertTable, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging cached cert table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge cachedCrlEntryTable...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(cachedCrlEntryTable, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging cached crl entry table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge aiaConnectionCache...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(aiaConnectionCache, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging aia connection cache table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] purge httpSocketCache...\n");
    pErr = nnsdkNssPKIX_PL_HashTable_Purge(httpSocketCache, nullptr);
    if (pErr != nullptr)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool:CleanPkixHashTables] warning: err purging http socket cache table: %d, %d\n",
                                pErr->errCode,
                                pErr->plErr);
        PKIX_PL_Object_DecRef(reinterpret_cast<PKIX_PL_Object *>(pErr), nullptr);
        pErr = nullptr;
    }

    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::CleanPkixHashTables] DONE\n");

    //  Done, release the lock
    NssUtil::CertTool::g_VerifyLock.Unlock();
}


SECStatus NssUtil::CertTool::SslChainVerifyCb(void               *pArg,
                                              const CERTCertList *pCurrentChain,
                                              PRBool             *pChainOk)
{
    SECStatus                   status = SECSuccess;
    SslChainVerifyArgs          *pChainVerifyArgs = reinterpret_cast<SslChainVerifyArgs *>(pArg);
    SslConnectionImpl           *pConn = pChainVerifyArgs->pConn;
    CertStore                   *pCertStore = pConn->GetParentSslContext()->GetCertStore();
    bool                        revoked = false;
    CERTCRLEntryReasonCode      reason;
    CERTCertificate             *pRevokedCert = nullptr;

    do
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::SslChainVerifyCb] checking imported CRL for context %p\n",
                                pConn->GetParentSslContext());

        //  Default to the chain being OK.
        *pChainOk = PR_TRUE;

        //  PKIX will automatically handle app imported trusts, but not app
        //  imported CRL since we have no way to specify it.  So, we check it
        //  here and if we find a revoked cert, we ABORT completely rather
        //  than rejecting the build chain.
        revoked =
            pCertStore->IsAnyCertInChainRevokedFromCrl(&reason,
                                                       &pRevokedCert,
                                                       pCurrentChain);
        if (!revoked)
        {
            break;
        }

        //  Either the cert or the chain has been revoked by an imported CRL.
        //  Save off some information about the revocation and return a
        //  failure here so PKIX aborts the verify completely.
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::SslChainVerifyCb] found revoked cert %p\n",
                                pRevokedCert);
        status = SECFailure;

        //  The error we return is based on whether the server cert was revoked
        //  or one of the chain of trust.  To verify this, we check the revoked
        //  cert against the head of the list.  If it is the server cert, report
        //  a revoked cert error, otherwise it's an untrusted issuer.
        CERTCertListNode        *pCertNode = CERT_LIST_HEAD(pCurrentChain);
        CERTCertificate         *pServerCert = pCertNode->cert;
        if (pServerCert == pRevokedCert)
        {
            pChainVerifyArgs->chainVerifyErr = SEC_ERROR_REVOKED_CERTIFICATE;
        }
        else
        {
            pChainVerifyArgs->chainVerifyErr = SEC_ERROR_UNTRUSTED_ISSUER;
        }

    } while (NN_STATIC_CONDITION(false));

    return status;
}


nn::Result NssUtil::CertTool::VerifyCertificate(CERTCertificate*         pInServerCert,
                                                PRInt64                  dateToCheck,
                                                SslConnectionImpl*       pInConnection)
{
    nn::Result                ret = nn::ResultSuccess();
    CERTValInParam            certInParams[cert_pi_max];
    CERTValOutParam           certOutParams[cert_po_max];
    int                       certInIndex = 0;
    int                       certOutIndex = 0;
    VerifyLogger              logger;
    CERTVerifyLog             *pVlog = nullptr;
    CERTCertList              *pTrustedRoots = nullptr;
    CERTRevocationFlags       *pRev = nullptr;
    PLArenaPool               *pArena = nullptr;
    EvCertUtil::PolicyOidInfo *pOidInfo = nullptr;
    CertStore                 *pCertStore = pInConnection->GetParentSslContext()->GetCertStore();
    ErrorList                 *pErrList = pInConnection->GetErrorList();
    CERTCertList              *pChain = nullptr;
    SECStatus                 status;
    bool                      needChain = false;
    CERTChainVerifyCallback   verifyCbInfo;
    SslChainVerifyArgs        chainVerifyArgs = { nullptr, 0 };

    logger.Initialize();
    pVlog = logger.GetCERTVerifyLog();

    memset(certInParams, 0, sizeof(certInParams));
    memset(certOutParams, 0, sizeof(certOutParams));

    do
    {
        //  Ask the connection if it needs the full chain
        needChain = pInConnection->IsServerChainNeeded();

        //  See if the server cert is trusted because it was imported
        //  into the cert store.
        if (pCertStore->IsServerCertTrusted(pInServerCert))
        {
            //  Before we green light this, see if we need to verify the date.
            //  If dateToCheck is -1 then we don't care.  If it is 0 then
            //  check against right now, otherwise use the provided time.
            if (dateToCheck >= 0)
            {
                if (dateToCheck == 0)
                {
                    dateToCheck = PR_Now();
                }

                SECCertTimeValidity valid =
                    CERT_CheckCertValidTimes(pInServerCert,
                                             dateToCheck,
                                             PR_FALSE);
                if (valid != secCertTimeValid)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] FAILED: server cert failed time/date check: %d\n",
                                            valid);
                    ret = ResultInternalCertVerifyFailed();
                    if (!pErrList->AddError(SEC_ERROR_EXPIRED_CERTIFICATE))
                    {
                        ret = ResultInsufficientMemory();
                        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] WARNING: Failed to add error to list - %d\n",
                                                SEC_ERROR_EXPIRED_CERTIFICATE);
                    }

                    break;
                }

                NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] SUCCESS: (date valid) server cert directly imported.\n");
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] SUCCESS: (date ignored) server cert directly imported.\n");
            }

            if (needChain)
            {
                ret = pInConnection->SaveServerCertChain(pInServerCert);
            }
            break;
        }

        //  Configure the date check parameter.
        //  < 0 is "no date verify"
        //  0 is "now"
        //  > 0 is a time stamp in PRTime (microseconds since the epoch)
        certInParams[certInIndex].type              = cert_pi_date;
        certInParams[certInIndex].value.scalar.time = dateToCheck;
        certInIndex++;

        //  Attach any trusted roots which were installed in the CertStore.
        //  If we cannot create the list or none were installed then proceed,
        //  the auth will likely fail later.
        pTrustedRoots = CERT_NewCertList();
        if (pTrustedRoots == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: failed to create list for trusted roots\n");
            ret = ResultInsufficientMemory();
            break;
        }

        ret = pCertStore->GetTrustedCertList(pTrustedRoots);
        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: failed to get list for trusted roots\n");
            break;
        }

        certInParams[certInIndex].type                = cert_pi_trustAnchors;
        certInParams[certInIndex].value.pointer.chain = pTrustedRoots;
        certInIndex++;

        //  Tell PKIX that it can use the built-in CAs as well as the
        //  trust anchors we have provided (if any.)
        certInParams[certInIndex].type = cert_pi_useOnlyTrustAnchors;
        certInParams[certInIndex].value.scalar.b = PR_FALSE;
        certInIndex++;

        //  Enable revocation checking during verification
        //  TODO: Add OCSP based revocation support
        pArena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
        if (pArena == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: failed to create arena for revocation tests\n");
            ret = ResultInsufficientMemory();
            break;
        }

        ret = SetupRevocation(pArena, &pRev);
        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: failed to setup revocation tests\n");
            break;
        }

        certInParams[certInIndex].type                     = cert_pi_revocationFlags;
        certInParams[certInIndex].value.pointer.revocation = pRev;
        certInIndex++;

        //  TODO: Add Authority Infomration Access (AIA) support for
        //        automatically pulling intermediate certs not presented
        //        by the server.
        certInParams[certInIndex].type           = cert_pi_useAIACertFetch;
        certInParams[certInIndex].value.scalar.b = PR_FALSE;
        certInIndex++;

        //  TODO: To perform full EV certificate verification, we must perform revocation checking
        //        too which may involve enabling OCSP. We don't care what verify option is set if
        //        VerifyOption_EvCertPartial is set
        //        VerifyOption_EvCert which may be added in the future should check if OCSP is also
        //        enabled for revocation check.

        //  Configure params for verifying EV certificate
        if ((pInConnection->GetVerifyOptionMask() & Connection::VerifyOption::VerifyOption_EvCertPartial)
            == Connection::VerifyOption::VerifyOption_EvCertPartial)
        {
            pOidInfo = new EvCertUtil::PolicyOidInfo();
            if (pOidInfo == nullptr)
            {
                ret = ResultInsufficientMemory();
                NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] Cannot allocate memory for BuiltInPolicyOids.\n");
                break;
            }


            EvCertUtil::PolicyOidInfo* pUserOidInfo = pInConnection->GetParentSslContext()->GetPolicyOidInfo();
            size_t totalOidCount = EvCertUtil::BuiltInPolicyOids::GetCount() + pUserOidInfo->GetCount();

            //  There's no reason to proceed setup if there's no OID to add
            if (totalOidCount > 0)
            {
                ret = pOidInfo->Setup(totalOidCount);
                if (ret.IsFailure())
                {
                    break;
                }

                //  Add policy OIDs in built-in
                ret = EvCertUtil::BuiltInPolicyOids::GetOidInfo(pOidInfo);
                if (ret.IsFailure())
                {
                    break;
                }

                //  Add policy OIDs added by the user
                if (pUserOidInfo->GetCount() > 0)
                {
                    ret = pOidInfo->AddOids(pUserOidInfo->GetHead(), pUserOidInfo->GetCount());
                    if (ret.IsFailure())
                    {
                        break;
                    }
                }
                NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] Added %d OIDs (built-in:%d)(user:%d)\n",
                    totalOidCount, EvCertUtil::BuiltInPolicyOids::GetCount(), pUserOidInfo->GetCount());
                certInParams[certInIndex].type             = cert_pi_policyOID;
                certInParams[certInIndex].value.arraySize  = static_cast<int>(pOidInfo->GetCount());
                certInParams[certInIndex].value.array.oids = pOidInfo->GetHead();
                certInIndex++;
            }
        }

        //  Add our own custom verify hook, used to check imported CRL
        //  for the connection's context.
        if (pCertStore->HasImportedCrl())
        {
            chainVerifyArgs.pConn          = pInConnection;
            chainVerifyArgs.chainVerifyErr = 0;
            verifyCbInfo.isChainValid      = SslChainVerifyCb;
            verifyCbInfo.isChainValidArg   = &chainVerifyArgs;
            certInParams[certInIndex].type = cert_pi_chainVerifyCallback;
            certInParams[certInIndex].value.pointer.chainVerifyCallback = &verifyCbInfo;
            certInIndex++;
        }

        //  No more input parameters, mark the end
        certInParams[certInIndex].type = cert_pi_end;

        //  The outputs we are interested in receiving back:
        //  * The trust anchor (the trusted root for the chain)
        //  * The trust chain
        //  * The error log
        if (needChain)
        {
            certOutParams[certOutIndex].type               = cert_po_trustAnchor;
            certOutParams[certOutIndex].value.pointer.cert = nullptr;
            certOutIndex++;

            certOutParams[certOutIndex].type                = cert_po_certList;
            certOutParams[certOutIndex].value.pointer.chain = nullptr;
            certOutIndex++;
        }

        certOutParams[certOutIndex].type              = cert_po_errorLog;
        certOutParams[certOutIndex].value.pointer.log = pVlog;
        certOutIndex++;

        certOutParams[certOutIndex].type = cert_po_end;

        //  Take the verify lock, also used by our cleaner to
        //  purge the PKIX caches.  We do not want to encounter
        //  any race conditions while using the caches.
        NssUtil::CertTool::g_VerifyLock.Lock();

        status = CERT_PKIXVerifyCert(pInServerCert,
                                     certificateUsageSSLServer,
                                     certInParams,
                                     certOutParams,
                                     nullptr);

        //  Regardless of the status, it is possible for the PKIX routines
        //  to cache certs, CAs, CRLs and signatures which will really eat
        //  into our memory.  After unlocking, queue up a polled (lazy)
        //  cleanup item to purge the PKIX caches.
        NssUtil::CertTool::g_VerifyLock.Unlock();
        NssUtil::CertTool::ScheduleCleanPkixHashTables();

        if ((status != SECSuccess) || (pVlog->count > 0))
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: Certificate validation failed.\n");

            //  If imported CRL was used, any error details will not be present
            //  in the CERTVerifyLog.  Instead, it will come back in the chain
            //  callback args.  But, do not rely on calling through to the
            //  CertStore (race condition), use the args themselves as pConn
            //  is only set if CRL was imported.
            if ((chainVerifyArgs.pConn != nullptr) &&
                (chainVerifyArgs.chainVerifyErr != 0))
            {
                if (!pErrList->AddError(chainVerifyArgs.chainVerifyErr))
                {
                    ret = ResultInsufficientMemory();
                    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] WARNING: Failed to add error to list - %d (%s)\n",
                                            chainVerifyArgs.chainVerifyErr,
                                            PR_ErrorToName(chainVerifyArgs.chainVerifyErr));
                    break;
                }
            }

            for (CERTVerifyLogNode *pLogNode = pVlog->head;
                 pLogNode != nullptr;
                 pLogNode = pLogNode->next)
            {
                NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] FAILED: CERT_VerifyCertificate - %d (%s)\n",
                    pLogNode->error, PR_ErrorToName(pLogNode->error));

                //  Get detailed error here because NSS doesn't tell why SEC_ERROR_POLICY_VALIDATION_FAILED
                //  is set. SEC_ERROR_EXTENSION_NOT_FOUND would be set if no certificate extension field
                //  is found in the certificate. SEC_ERROR_UNRECOGNIZED_OID would be set if policy OID found
                //  in the cert is not the one we have in pOidInfo
                if (pLogNode->error == SEC_ERROR_POLICY_VALIDATION_FAILED && pOidInfo != nullptr)
                {
                    SECErrorCodes tmpError;
                    if (EvCertUtil::GetErrorDetail(&tmpError, pOidInfo, pInServerCert).IsSuccess())
                    {
                        pLogNode->error = tmpError;
                    }
                }

                if (!pErrList->AddError(pLogNode->error))
                {
                    ret = ResultInsufficientMemory();
                    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] WARNING: Failed to add error to list - %d (%s)\n",
                        pLogNode->error, PR_ErrorToName(pLogNode->error));
                    break;
                }
            }

#if defined(NN_DETAIL_SSL_DBG_PRINT_CERT_CHAIN_UPON_VERIFY_FAILURE)
            logger.Dump(this);
#endif // NN_DETAIL_SSL_DBG_PRINT_CERT_CHAIN_UPON_VERIFY_FAILURE

            if (ret.IsSuccess())
            {
                ret = ResultInternalCertVerifyFailed();
            }
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] SUCCESS: CERT_PKIXVerifyCert succeeded.\n");

            //  Make sure we got both the chain and the anchor from PKIX.
            if (needChain)
            {
                if ((certOutParams[0].value.pointer.cert == nullptr) ||
                    (certOutParams[1].value.pointer.chain == nullptr))
                {
                    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] WARNING: anchor or chain missing (%p, %p)\n",
                                            certOutParams[0].value.pointer.cert,
                                            certOutParams[1].value.pointer.chain);
                    break;
                }

                //  Retain the chain temporarily so it can be added to the
                //  caller's provided buffer (if requested).
                pChain = certOutParams[1].value.pointer.chain;

                //  Add the trust anchor to the end of the chain
                status =
                    CERT_AddCertToListTail(pChain,
                                           certOutParams[0].value.pointer.cert);
                if (status != SECSuccess)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] ERROR: failed to add trust anchor to chain\n");

                    //  Report the error.  No need to null out pChain as the output
                    //  parameters are still set and will get purged below in
                    //  the error handling.
                    ret = ResultInsufficientMemory();
                    break;
                }

                //  Null out the output parameters so they are not accidentally purged 2x.
                certOutParams[1].value.pointer.chain = nullptr;
                certOutParams[0].value.pointer.cert = nullptr;

                //  Fall through, the chain will get added to the caller's
                //  buffer below, if return value is success.
            }
        }
    } while (NN_STATIC_CONDITION(false));

    if ((ret.IsSuccess()) && (pChain != nullptr))
    {
        //  Ask the connection to save the server chain for the caller.
        ret = pInConnection->SaveServerCertChain(pChain);

        //  Regardless of the outcome, destroy the chain - we are done with it.
        CERT_DestroyCertList(pChain);
        pChain = nullptr;
    }

    //  Destroy the trust anchor and trust chain, if available
    if (needChain)
    {
        if (certOutParams[0].value.pointer.cert != nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] release trust anchor\n");
            CERT_DestroyCertificate(certOutParams[0].value.pointer.cert);
        }

        if (certOutParams[1].value.pointer.chain != nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyCertificate] release trust trust chain\n");
            CERT_DestroyCertList(certOutParams[1].value.pointer.chain);
        }
    }

    if (pArena != nullptr)
    {
        PORT_FreeArena(pArena, PR_FALSE);
        pArena = nullptr;
    }

    if (pTrustedRoots != nullptr)
    {
        CERT_DestroyCertList(pTrustedRoots);
        pTrustedRoots = nullptr;
    }

    if (pOidInfo != nullptr)
    {
        pOidInfo->Cleanup();
        delete pOidInfo;
    }

    // NSS doesn't reset PR error which is set in low level even when it's handles as success in
    // the upper layer. Because we log all errors by logger and ErrorList, we reset PR error here.
    PORT_SetError(SECSuccess);

    logger.Finalize();
    return ret;
}    //  NOLINT(impl/function_size)


nn::Result NssUtil::CertTool::VerifyName(
    CERTCertificate* pInServerCert,
    PRFileDesc*      pInSocket,
    char*            pInHostName,
    PRErrorCode*     pOutErrorCode)
{
    NN_SDK_REQUIRES_NOT_NULL(pInServerCert);
    if(pInServerCert == nullptr)
    {
        return ResultInvalidReference();
    }

    char*      pHostName = nullptr;
    SECStatus  status    = SECSuccess;
    nn::Result result    = nn::ResultSuccess();

    if((pInHostName == nullptr) && (pInSocket != nullptr))
    {
        pHostName = SSL_RevealURL(pInSocket);
    }
    else if((pInSocket == nullptr) && (pInHostName != nullptr))
    {
        pHostName = pInHostName;
    }
    else
    {
        return ResultInternalLogicError();
    }

    status = CERT_VerifyCertName(pInServerCert, pHostName);
    if(status != SECSuccess)
    {
        PRErrorCode prError = PR_GetError();
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyName]FAILED: CERT_VerifyCertName - %d (%s)\n",
            prError, PR_ErrorToName(prError));

        *pOutErrorCode = prError;
        result = ResultInternalHostNameVerifyFailed();
    }
    else
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyName]SUCCESS: Verified host name (%s)\n",
            pHostName);

        // NSS sets SSL_ERROR_BAD_CERT_DOMAIN when name validation fails with one of the subject
        // alternative names, and never reset it. Let's reset it here to report the coorect status.
        PORT_SetError(SECSuccess);
    }

    if((pInHostName == nullptr) && (pInSocket != nullptr))
    {
        PR_Free(pHostName);
    }

    return result;
}

nn::Result NssUtil::CertTool::VerifyDate(CERTCertificate* pInServerCert)
{
    NN_SDK_REQUIRES_NOT_NULL(pInServerCert);
    if(pInServerCert == nullptr)
    {
        return ResultInvalidReference();
    }

    SECCertTimeValidity status;
    nn::Result          result = nn::ResultSuccess();

    status = CERT_CheckCertValidTimes(pInServerCert, PR_Now(), PR_FALSE);
    if(status == secCertTimeValid)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyDate]SUCCESS: Verified date\n");
    }
    else if(status == secCertTimeExpired)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyDate]FAILED: CERT_CheckCertValidTimes - secCertTimeExpired\n");
        // TODO: Set appropriate Result here
    }
    else if(status == secCertTimeNotValidYet)
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyDate]FAILED: CERT_CheckCertValidTimes - secCertTimeNotValidYet\n");
        // TODO: Set appropriate Result here
    }
    else
    {
        NN_DETAIL_SSL_DBG_PRINT("[NssUtil::CertTool::VerifyDate]FAILED: CERT_CheckCertValidTimes - Unknown (%d)\n", status);
        // TODO: Set appropriate Result here
    }

    return result;
}

// ------------------------------------------------------------------------------------------------
// Converter functions
// ------------------------------------------------------------------------------------------------
// http://stackoverflow.com/questions/24908261/converting-certcertificate-dercert-to-seckeypublickey
nn::Result NssUtil::NssConverter::DerDataToCertificate(char* pInDerCert, uint32_t dataLength)
{
    SECItem derCertItem;
    derCertItem.type = siBuffer;
    derCertItem.data = reinterpret_cast<unsigned char*>(pInDerCert);
    derCertItem.len  = dataLength;

    CERTCertificate* pCert = CERT_NewTempCertificate(
        CERT_GetDefaultCertDB(),
        &derCertItem,
        nullptr,
        PR_FALSE,
        PR_FALSE);

    SECKEYPublicKey* pPublicKey = CERT_ExtractPublicKey(pCert);
    NN_UNUSED(pPublicKey);

    // TODO: Add actual implementation here
    CERT_DestroyCertificate(pCert);
    return nn::ResultSuccess();
}

nn::Result NssUtil::NssConverter::DerDataToPublicKey(char* pInDerData, uint32_t dataLength)
{
    SECItem derKeyItem;
    derKeyItem.type = siBuffer;
    derKeyItem.data = reinterpret_cast<unsigned char*>(pInDerData);
    derKeyItem.len  = dataLength;

    CERTSubjectPublicKeyInfo* pPublicKeyInfo = SECKEY_DecodeDERSubjectPublicKeyInfo(&derKeyItem);
    SECKEYPublicKey*          pPublicKey     = SECKEY_ExtractPublicKey(pPublicKeyInfo);

    NN_UNUSED(pPublicKeyInfo);
    NN_UNUSED(pPublicKey);

    // TODO: Add actual implementation here

    return nn::ResultSuccess();
}

NssUtil::ErrorList::ErrorList() NN_NOEXCEPT
    : m_pHead(nullptr)
{}

NssUtil::ErrorList::~ErrorList() NN_NOEXCEPT
{
    ClearErrors();
}

bool NssUtil::ErrorList::AddError(PRErrorCode error) NN_NOEXCEPT
{
    ErrorNode* pNewError = new ErrorNode;
    if( !pNewError )
    {
        return false;
    }

    pNewError->error = error;
    pNewError->pNext = m_pHead;
    m_pHead = pNewError;

    return true;
}

void NssUtil::ErrorList::ClearErrors() NN_NOEXCEPT
{
    ErrorNode* pError = m_pHead;
    while(pError)
    {
        ErrorNode* pNext = pError->pNext;
        delete pError;
        pError = pNext;
    }

    m_pHead = nullptr;
}

const NssUtil::ErrorList::ErrorNode* NssUtil::ErrorList::GetHead() const NN_NOEXCEPT
{
    return m_pHead;
}

NssUtil::ErrorList::ErrorNode::ErrorNode() NN_NOEXCEPT
    : error(SECSuccess),
      pNext(nullptr)
{}

}}} // namespace nn { namespace ssl { namespace detail {
