﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_NssCommon.h"
#include "server/ssl_Util.h"
#include "server/ssl_NssConfigFsManagerBase.h"

namespace nn { namespace ssl { namespace detail {

const char *NssConfigFsManagerBase::g_MountPrefix = NN_DETAIL_SSL_NNSDK_FS_MOUNT;

nn::Result NssConfigFsManagerBase::MountHostFs(const char *pMountPathStr)
{
    nn::Result                  ret = nn::ResultSuccess();

    do
    {
        const char *mountPath =
            nnsdkSetupHostSystemFs(g_MountPrefix,
                                   pMountPathStr,
                                   PR_TRUE);
        if (mountPath == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerBase] Failed to mount host fs (%s)\n",
                                    pMountPathStr);
            ret = ResultInternalNssFailedFileMount();
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerBase] Host fs (%s) mounted at: %s\n",
                                pMountPathStr,
                                mountPath);
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result NssConfigFsManagerBase::UnmountHostFs()
{
    NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerBase] Shutdown host fs mount\n");
    nnsdkShutdownHostFs();
    return nn::ResultSuccess();
}


nn::Result NssConfigFsManagerBase::SetupNssCfgTree(const char *pInPathStr, size_t pathLength)
{
    nn::Result                  result = ResultSuccess();

    NN_SDK_REQUIRES_NOT_NULL(pInPathStr);
    if (pInPathStr == nullptr)
    {
        return ResultInvalidReference();
    }

    char        tmpString[g_MountPathStringLength];
    const char* pCurrent    = strchr(pInPathStr, '/');
    const char* pNext       = nullptr;
    const char* pCurrentDir = nullptr;

    NN_DETAIL_SSL_DBG_PRINT("[SetupNssCfgTree] setting up: \'%s\', len %d\n", pInPathStr, pathLength);
    while ((pCurrent != nullptr) && (static_cast<size_t>(pCurrent - pInPathStr) <= pathLength))
    {
        pNext = strchr(pCurrent + 1, '/');
        if (pNext == nullptr)
        {
            pCurrentDir = pInPathStr;
        }
        else
        {
            size_t currentPathLength = static_cast<size_t>(pNext - pInPathStr);
            if (currentPathLength >= sizeof(tmpString))
            {
                currentPathLength = sizeof(tmpString) - 1;
            }
            strncpy(tmpString, pInPathStr, currentPathLength);
            tmpString[currentPathLength] = '\0';
            pCurrentDir = tmpString;
        }

        NN_DETAIL_SSL_DBG_PRINT("[SetupNssCfgTree] Creating sub folder: %s\n", pCurrentDir);
        PRStatus status = PR_MkDir(pCurrentDir, PR_IRWXU);
        if (status != PR_SUCCESS)
        {
            PRErrorCode prError = PR_GetError();
            if(prError != PR_FILE_EXISTS_ERROR)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetupNssCfgTree] PR_MkDir failed: %d (%s)\n",
                    prError, PR_ErrorToName(prError));
                result = ResultErrorLower();
                break;
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetupNssCfgTree] Folder already exists.\n");
            }
        }
        pCurrent = pNext;
    }

    return result;
}

} } }
