﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_SaveDataManagement.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_NssCommon.h"
#include "server/ssl_NssConfigFsManager.h"
#include "server/ssl_NssConfigurator.h"
#include "server/ssl_Util.h"
#include "debug/ssl_DebugUtil.h"

namespace nn { namespace ssl { namespace detail {

nn::Result NssConfigFsManager::Initialize()
{
    NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER();

    nn::Result                  ret = nn::ResultSuccess();
    uint64_t                    retry = 0;

    do
    {
        nn::fs::DisableAutoSaveDataCreation();

        //  Always delete existing system save data to make sure that SSL process starts
        //  with fresh database when the system starts, since there's nothing the SS
        //  process can do, the error from this function is not handled
        ret = nn::fs::DeleteSaveData(g_SslSystemSaveId);
        NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManager] nn::fs::DeleteSaveData %s (desc:%d)\n",
            (ret.IsSuccess())?("SUCCEEDED"):("FAILED"), ret.GetDescription());

        do
        {
            ret = nn::fs::MountSystemSaveData(g_MountPrefix, g_SslSystemSaveId);
            if (ret.IsSuccess())
            {
                break;
            }
            else if (nn::fs::ResultTargetNotFound::Includes(ret))
            {
                ret = nn::fs::CreateSystemSaveData(g_SslSystemSaveId,
                                                   g_MaxSaveDataSize,
                                                   g_SaveDataJournalSize,
                                                   g_SaveDataFlag);
                if (ret.IsSuccess())
                {
                    //  Needs to create a mount point once system save data is created
                    continue;
                }
            }

            //  Comes here when something fails
            if (nn::fs::ResultAllocationMemoryFailed::Includes(ret))
            {
                continue; //  Just retry to see if it succeeds next time
            }
            else
            {
                //  Unexpected situation
                NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManager] FS cannot be initialized (desc:%d)",
                    ret.GetDescription());
                ret = ResultErrorLower();
                break;
            }
        } while (retry++ < g_FsInitMaxRetry);

        NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManager] mount of save data %s\n",
                                ret.IsSuccess() ? "SUCCEEDED" : "FAILED");
    } while (NN_STATIC_CONDITION(false));

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result NssConfigFsManager::Finalize()
{
    nn::fs::Unmount(g_MountPrefix);
    return nn::ResultSuccess();
}

} } }
