﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>
#include "ssl_MemoryManager.h"
#include "ssl_NssCommon.h"
#include "ssl_Util.h"

namespace nn { namespace ssl { namespace detail {

//  Forward declaration since there are some circular dependencies
class CertStore;
class ServerCertEntry;
class ClientCertEntry;


class ServerCertEntry
{
private:
    CertStore                               *m_CertStore;
    PRCList                                 m_CertList;
    char                                    *m_NicknameBase;
    nn::Result                              m_ImportStatus;

    nn::Result CertListAdd(CERTCertificate *pCert);

public:
    static const char*                      NicknamePrefix;
    static const int                        NicknamePrefixLen = 8;

    ServerCertEntry(CertStore *certStore, char *nicknameBase);
    ~ServerCertEntry();

    CertStore *GetCertStore();
    nn::Result GetImportStatus();

    bool IsServerCertTrusted(CERTCertificate *pCert);
    nn::Result GetTrustedCertList(CERTCertList *pList);

    void RefreshCert();

    static SECStatus ImportServerCertificateCallbackHook(void    *pArg,
                                                         SECItem **pCertItems,
                                                         int     certsCount);

    //  TODO: APIs lookup of CERT based on Subject ID, Email Address, etc.
};


class ClientCertEntry
{
public:
    static const char                       *g_NicknameBase;
    static const int                        g_NicknameBaseLen = 7;

private:
    CERTCertificate                         *m_pCert;
    SECKEYPrivateKey                        *m_pPrivKey;

public:
    ClientCertEntry(CERTCertificate   *pCert,
                    SECKEYPrivateKey  *pPrivKey);
    ~ClientCertEntry();

    CERTCertificate *GetCert();
    SECKEYPrivateKey *GetKey();

    static uint32_t GetMaxNameSize();
};


class CrlEntry
{
private:
    CERTSignedCrl               *m_pCrl;

    bool IsCertRevoked(CERTCRLEntryReasonCode  *pOutReason,
                       CERTCertificate         *pInCert);

public:
    explicit CrlEntry(CERTSignedCrl *pCrl);
    ~CrlEntry();

    bool IsAnyCertInChainRevoked(CERTCRLEntryReasonCode  *pOutReason,
                                 CERTCertificate         **pOutRevokedCert,
                                 const CERTCertList      *pTrustChain);
};


class CertStore
{
private:
    nn::os::Mutex                          m_Lock;
    uint64_t                               m_OwnerId;
    PRCList                                *m_pServerEntries;
    PRCList                                *m_pCrlEntries;
    ClientCertEntry                        *m_ClientEntry;    //  Only 1 allowed!

    struct GetCrlCountCbArgs
    {
        CertStore               *pCertStore;
        int                     total;
    };

    nn::Result ServerEntryAdd(ServerCertEntry *pNewEntry);
    nn::Result ServerEntryRemove(ServerCertEntry *pEntry);
    bool IsServerEntryPresent(ServerCertEntry *pEntry);

    static nn::Result DecodeAndValidateCrl(CERTSignedCrl **pOutNewCrl,
                                           const uint8_t  *pInCrlDerData,
                                           uint32_t       crlDerDataSize,
                                           bool           enableDateCheck);

    static nn::Result GetCrlCountForApp(PRCList *pCtxList, void *pCbArg);

    nn::Result WalkLockAndGetCrlCount(CertStore *pReqStore,
                                      PRCList   *pList,
                                      PRCList   *pThisEntry,
                                      int       *pCount);
    nn::Result WalkAndUnlock(CertStore *pReqStore,
                             PRCList   *pList,
                             PRCList   *pThisEntry);

public:
    CertStore();
    explicit CertStore(uint64_t ownerId);
    ~CertStore();

    nn::Result SetOwnerId(uint64_t id);
    uint64_t GetOwnerId();

    ClientCertEntry *GetClientCert();
    uint64_t GetServerCertEntryCount();

    nn::Result ImportServerPki(uint64_t                   *pOutNewId,
                               const char                 *pInCertData,
                               uint32_t                   certDataSize,
                               nn::ssl::CertificateFormat fmt);

    nn::Result RemoveServerPki(uint64_t certId);

    bool IsServerCertTrusted(CERTCertificate *pCert);
    nn::Result GetTrustedCertList(CERTCertList *pList);
    nn::Result GetTrustedCerts(CERTCertList *pOutList, uint64_t certId);

    nn::Result ImportClientPki(uint64_t                   *pOutNewId,
                               const char                 *pInP12Data,
                               const char                 *pInPwData,
                               uint32_t                   p12DataLen,
                               uint32_t                   pwDataLen);

    nn::Result ImportDeviceUniqueClientPki(uint64_t    *pOutNewId,
                                           const char  *pInNickname);

    nn::Result RemoveClientPki(uint64_t certId);

    nn::Result ImportCrl(uint64_t       *pOutNewId,
                         const uint8_t  *pInCrlDerData,
                         uint32_t       crlDerDataSize);
    nn::Result RemoveCrl(uint64_t crlId);
    bool HasImportedCrl();
    bool IsAnyCertInChainRevokedFromCrl(CERTCRLEntryReasonCode  *pOutReason,
                                        CERTCertificate         **pOutRevokedCert,
                                        const CERTCertList      *pTrustChain);

    static char *ExtractNickname(CERTCertificate *pCert);
};


}}}
