﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ssl_BdfReader.h"
#include "ssl_StaticDataReader.h"
#include "ssl_Util.h"

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include "ssl_TrustedCerts.win32.cpp"
#include "ssl_CrlBuiltin.win32.cpp"
#include "ssl_CrlNtdServer1.cpp"
#endif

namespace nn { namespace ssl { namespace detail {

//////////////////////////////////////////////////////////////////////////////
//  BdfReader - The reader factory and interface for getting built-in data.
//////////////////////////////////////////////////////////////////////////////
nn::Result BdfReaderFactory::Initialize() NN_NOEXCEPT
{
    return ResultSuccess();
}


nn::Result BdfReaderFactory::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


BdfReader *BdfReaderFactory::Create(BdfReader::BdfId id) NN_NOEXCEPT
{
    BdfReader                   *pNewReader = nullptr;
    nn::Result                  result = ResultSuccess();
    uint8_t                     *pData = nullptr;
    uint32_t                    dataLen = 0;

    do
    {
        switch (id)
        {
            case BdfReader::BdfId_TrustedCerts:
                pData   = g_SystemDataSslTrustedCertsBdf;
                dataLen = g_SystemDataSslTrustedCertsBdfLen;
                break;

            case BdfReader::BdfId_Crl:
                pData   = g_SystemDataSslCrlBdf;
                dataLen = g_SystemDataSslCrlBdfLen;
                break;

            case BdfReader::BdfId_TestCrl:
            {
                if(Util::IsInitializeForTestEnabled())
                {
                    pData    = g_SslCrlNtdServer1Bdf;
                    dataLen  = g_SslCrlNtdServer1BdfLen;
                }
                else
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] Please enable initialize_for_test for ssl.test\n", id);
                }

                break;
            }

            default:
                NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] invalid id: %8.8X\n", id);
                break;
        }

        if (pData == nullptr)
        {
            break;
        }

        pNewReader = new StaticDataReader(pData, dataLen);
        if (pNewReader == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to create new instance of StaticDataReader for id %8.8X\n",
                                    id);
            result = ResultErrorLower();
            break;
        }

        result = pNewReader->Open();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[BdfReaderFactory::Create] failed to open BdfReader for id %8.8X: %d-%d\n",
                                    id,
                                    result.GetModule(),
                                    result.GetDescription());
            delete pNewReader;
            pNewReader = nullptr;
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return pNewReader;
}

} } }    //  nn::ssl::detail
