﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/spsm/spsm_PowerStateTypes.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  PowerButton 監視用クラス
//  TORIAEZU:
//      単一スレッドでの利用を前提としているため、
//      m_State 変数の排他などは行なわない。
//
class PowerButton final : public IEventHandler
{
public:
    static TimeSpan GetTimeSpanShortPressing() NN_NOEXCEPT
    {
        return TimeSpan::FromMilliSeconds(30);
    }
    static TimeSpan GetTimeSpanMiddlePressing() NN_NOEXCEPT
    {
        return TimeSpan::FromSeconds(3);
    }
    static TimeSpan GetTimeSpanLongPressing() NN_NOEXCEPT
    {
        return TimeSpan::FromSeconds(7);
    }

public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_PowerButtonEventHolder);
        os::LinkMultiWaitHolder(p, &m_TimerEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_PowerButtonEventHolder);
        os::UnlinkMultiWaitHolder(&m_TimerEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT
    {
        if (holder == &m_TimerEventHolder)
        {
            TimerEventHandler();
            return true;
        }
        else if (holder == &m_PowerButtonEventHolder)
        {
            PowerButtonEventHandler();
            return true;
        }
        return false;
    }
    void OnPowerStateEntry(PowerState enteredState) NN_NOEXCEPT
    {
        switch ( enteredState )
        {
            case PowerState_SleepReady:
            {
                // スリープしようとしている時点で電源ボタンのタイマー計測を止める（3秒押し・7秒押しイベントはこれ以降発生しない）
                DisablePowerButtonCounterEventHandler();
                break;
            }
            default:
            {
                break;
            }
        }
    }
    void OnPowerStateExit(PowerState exitedState) NN_NOEXCEPT
    {
        switch ( exitedState )
        {
            case PowerState_SleepReady:
            {
                // 電源ボタンのタイマー計測を再開する（押下開始時刻も現時刻でリセットする）
                // NOTE: この関数内で現在の電源ボタンの押下状態を取り直すため、
                //       先に bpc の EssentialServicesAwake 処理が終わっていないといけない
                EnablePowerButtonCounterEventHandler();
                break;
            }
            default:
            {
                break;
            }
        }
    }

private:
    void TimerEventHandler() NN_NOEXCEPT;
    void PowerButtonEventHandler() NN_NOEXCEPT;
    void EnablePowerButtonCounterEventHandler() NN_NOEXCEPT;
    void DisablePowerButtonCounterEventHandler() NN_NOEXCEPT;

    void NotifyStartPressingPowerButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("%s\n", __FUNCTION__);
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonStartedPressing));
    }

    void NotifyShortPressingPowerButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("%s\n", __FUNCTION__);
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonPressedBriefly));
    }

    void NotifyMiddlePressingPowerButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("%s\n", __FUNCTION__);
        m_pPowerStateMessageQueue->Cancel(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonPressedBriefly)); // 本来同時に成立しないので、まだ処理されていないならキャンセルしておく
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonPressedFor3Sec));
    }

    void NotifyLongPressingPowerButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("%s\n", __FUNCTION__);
        m_pPowerStateMessageQueue->Cancel(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonPressedBriefly)); // 本来同時に成立しないので、まだ処理されていないならキャンセルしておく
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerButtonPressedFor7Sec));
    }

private:
    os::Mutex m_Mutex{ false };

    nn::spsm::server::PowerStateMessageQueue*     m_pPowerStateMessageQueue;

    os::TimerEvent              m_TimerEvent{ os::EventClearMode_AutoClear };
    os::SystemEvent             m_PowerButtonEvent{ os::EventClearMode_AutoClear, true };

    os::MultiWaitHolderType     m_TimerEventHolder;
    os::MultiWaitHolderType     m_PowerButtonEventHolder;

    os::Tick    m_TickShortPressing;
    os::Tick    m_TickMiddlePressing;
    os::Tick    m_TickLongPressing;
    os::Tick    m_StartTick;

    bool        m_State{ false };
    bool        m_IsCountEnabled{ true };
};

}}} // namespace nn::spsm::observer

