﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  InternalTimer 監視用クラス
//  TORIAEZU:
//      単一スレッドでの利用を前提としているため、
//      m_State 変数の排他などは行なわない。
//
class InternalTimer final : public IEventHandler
{
public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_TimerEventHolder);
        os::LinkMultiWaitHolder(p, &m_StartTimerEventHolder);
        os::LinkMultiWaitHolder(p, &m_CancelTimerEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_TimerEventHolder);
        os::UnlinkMultiWaitHolder(&m_StartTimerEventHolder);
        os::UnlinkMultiWaitHolder(&m_CancelTimerEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT
    {
        if (holder == &m_TimerEventHolder)
        {
            ProcessTimerEvent();
            return true;
        }
        if (holder == &m_StartTimerEventHolder)
        {
            ProcessStartTimerEvent();
            return true;
        }
        if (holder == &m_CancelTimerEventHolder)
        {
            ProcessCancelTimerEvent();
            return true;
        }
        return false;
    }

private:
    void ProcessTimerEvent() NN_NOEXCEPT;
    void ProcessStartTimerEvent() NN_NOEXCEPT;
    void ProcessCancelTimerEvent() NN_NOEXCEPT;
    void StartTimer() NN_NOEXCEPT;
    void CancelTimer() NN_NOEXCEPT;

    void NotifyInternalTimerExpired() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("internal timer expired\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventInternalTimerExpired));
    }

private:
    nn::spsm::server::PowerStateMessageQueue*     m_pPowerStateMessageQueue;

    os::TimerEvent              m_TimerEvent{ os::EventClearMode_AutoClear };
    os::MultiWaitHolderType     m_TimerEventHolder;

    os::MultiWaitHolderType     m_StartTimerEventHolder; // 抱えるイベント実体は cpp 側参照
    os::MultiWaitHolderType     m_CancelTimerEventHolder; // 抱えるイベント実体は cpp 側参照

    bool                        m_State{ false };
};

//-----------------------------------------------------------------------------

void StartInternalTimer(nn::TimeSpan duration) NN_NOEXCEPT;
void CancelInternalTimer() NN_NOEXCEPT;

}}} // namespace nn::spsm::observer

