﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>

#include "settings_SystemSaveData-os.win.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< ファイル名のプレフィックス
const char FileNamePrefix[] = "Nintendo_SystemSaveData_";

} // namespace

SystemSaveData::SystemSaveData() NN_NOEXCEPT
    : m_IsInMemoryMode(false)
    , m_MemoryAddress(nullptr)
    , m_MemorySize(0)
{
    m_SystemDataFile.SetPrefix(FileNamePrefix);
}

SystemSaveData::SystemSaveData(const SystemDataFile& file) NN_NOEXCEPT
    : m_IsInMemoryMode(false)
    , m_MemoryAddress(nullptr)
    , m_MemorySize(0)
    , m_SystemDataFile(file)
{
    // 何もしない
}

void SystemSaveData::SetSystemSaveDataId(
    ::nn::fs::SystemSaveDataId value) NN_NOEXCEPT
{
    NN_UNUSED(value);
}

void SystemSaveData::SetTotalSize(int64_t value) NN_NOEXCEPT
{
    NN_UNUSED(value);
}

void SystemSaveData::SetJournalSize(int64_t value) NN_NOEXCEPT
{
    NN_UNUSED(value);
}

void SystemSaveData::SetFlags(uint32_t value) NN_NOEXCEPT
{
    NN_UNUSED(value);
}

void SystemSaveData::SetMountName(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);
    m_SystemDataFile.SetMountName(name);
}

::nn::Result SystemSaveData::EnableInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    m_IsInMemoryMode = true;

    m_MemoryAddress = reinterpret_cast<::nn::Bit8*>(buffer);
    m_MemorySize = size;

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Mount(bool creates) NN_NOEXCEPT
{
    NN_RESULT_DO(m_SystemDataFile.Mount(creates));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Commit(bool synchronizes) NN_NOEXCEPT
{
    NN_UNUSED(synchronizes);
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Create(int64_t size) NN_NOEXCEPT
{
    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.Create(size));
    }
    else
    {
        NN_RESULT_THROW(::nn::fs::ResultPathAlreadyExists());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::OpenToRead() NN_NOEXCEPT
{
    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.OpenToRead());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::OpenToWrite() NN_NOEXCEPT
{
    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.OpenToWrite());
    }

    NN_RESULT_SUCCESS;
}

void SystemSaveData::Close() NN_NOEXCEPT
{
    if (!m_IsInMemoryMode)
    {
        m_SystemDataFile.Close();
    }
}

::nn::Result SystemSaveData::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.Read(offset, buffer, size));
    }
    else
    {
        NN_RESULT_THROW_UNLESS(
            0 <= offset && offset + size <= m_MemorySize,
            ::nn::fs::ResultOutOfRange());

        ::std::memcpy(buffer, &m_MemoryAddress[offset], size);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Write(
    int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.Write(offset, buffer, size));
    }
    else
    {
        NN_RESULT_THROW_UNLESS(
            0 <= offset && offset + size <= m_MemorySize,
            ::nn::fs::ResultUsableSpaceNotEnough());

        ::std::memcpy(&m_MemoryAddress[offset], buffer, size);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Flush() NN_NOEXCEPT
{
    if (!m_IsInMemoryMode)
    {
        NN_RESULT_DO(m_SystemDataFile.Flush());
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::settings::detail
