﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <functional>
#include <mutex>
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_Uuid.h>

#include "settings_LockableMutexType.h"
#include "settings_Platform.h"
#include "settings_PrivateSettingsDatabase.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemSaveData-os.win.h"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2017) && defined(NN_SDK_BUILD_DEBUG)
#undef offsetof
#define offsetof(type, member) \
    (reinterpret_cast<size_t>(&(reinterpret_cast<type*>(0)->member)))
#endif

//!< プライベート設定の警告ログを出力します。
#define NN_SETTINGS_PRIVATE_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[private settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< システムセーブデータの識別子
const ::nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000052;

//!< システムセーブデータの合計サイズ
const int64_t SystemSaveDataTotalSize = (256 + 32) << 10;

//!< システムセーブデータのジャーナルサイズ
const int64_t SystemSaveDataJournalSize = (256 + 32) << 10;

//!< システムセーブデータのフラグ集合
const uint32_t SystemSaveDataFlags =
    ::nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveData;

//!< マウント名
const char MountName[] = "PrivateSettings";

//!< 初回起動設定のデフォルト値
const ::nn::settings::system::InitialLaunchSettings
    DefaultInitialLaunchSettings =
{
    {},
    {
        0LL,
        ::nn::util::InvalidUuid,
    }
};

//!< ExternalSteadyClock のソース ID のデフォルト値
const ::nn::settings::system::ClockSourceId DefaultExternalSteadyClockSourceId =
    ::nn::util::InvalidUuid;

//!< シャットダウン時の RTC 値のデフォルト値
const int64_t DefaultShutdownRtcValue = 0LL;

//!< ExternalSteadyClock の内部オフセットのデフォルト値
const int64_t DefaultExternalSteadyClockInternalOffset = 0LL;

//!< 時計補正ツールの実行が必要か否かを表す値のデフォルト値
const bool DefaultRequiresRunRepairTimeReviser = false;

//!< システムセーブデータ操作排他用ミューテックス
LockableMutexType g_SystemSaveDataMutex =
{
    NN_OS_MUTEX_INITIALIZER(false)
};

//!< システムセーブデータから値を取得します。
template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータから値を取得します。
template<typename T, typename U>
::nn::Result GetSystemSaveDataValue(
    U* pOutCount, T outValues[], U count,
    ::std::function<::nn::Result(U*, T*, U, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータの値を変更します。
template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT;

//!< プライベート設定から値を取得します。
template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< プライベート設定から値を取得します。
template<typename T, typename U>
::nn::Result GetSystemSettingsValue(
    U* pOutCount, T outValues[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(U*, T*, U) NN_NOEXCEPT> getter,
    ::std::function<void(U*, T*, U) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< プライベート設定の値を変更します。
template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT;

//!< システムセーブデータから初回起動設定を取得します。
::nn::Result GetSystemSaveDataInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの初回起動設定を変更します。
::nn::Result SetSystemSaveDataInitialLaunchSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから ExternalSteadyClock のソース ID を取得します。
::nn::Result GetSystemSaveDataExternalSteadyClockSourceId(
    ::nn::settings::system::ClockSourceId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの ExternalSteadyClock のソース ID を変更します。
::nn::Result SetSystemSaveDataExternalSteadyClockSourceId(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ClockSourceId& value) NN_NOEXCEPT;

//!< システムセーブデータからシャットダウン時の RTC 値を取得します。
::nn::Result GetSystemSaveDataShutdownRtcValue(
    int64_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのシャットダウン時の RTC 値を変更します。
::nn::Result SetSystemSaveDataShutdownRtcValue(
    SystemSaveData* pSystemSaveData,
    int64_t pOutValue) NN_NOEXCEPT;

//!< システムセーブデータからExternalSteadyClock の内部オフセットを取得します。
::nn::Result GetSystemSaveDataExternalSteadyClockInternalOffset(
    int64_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの ExternalSteadyClock の内部オフセットを変更します。
::nn::Result SetSystemSaveDataExternalSteadyClockInternalOffset(
    SystemSaveData* pSystemSaveData,
    int64_t pOutValue) NN_NOEXCEPT;

//!< システムセーブデータから時計補正ツールの実行が必要か否かを取得します。
::nn::Result GetSystemSaveDataRequiresRunRepairTimeReviser(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの時計補正ツールの実行が必要か否かを変更します。
::nn::Result SetSystemSaveDataRequiresRunRepairTimeReviser(
    SystemSaveData* pSystemSaveData,
    bool pOutValue) NN_NOEXCEPT;

//!< システムセーブデータ上のプライベート設定を読み出します。
::nn::Result ReadSystemSaveDataPrivateSettings(
    uint64_t* pOutCount, char outBuffer[], uint64_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータ上のプライベート設定をリセットします。
::nn::Result ResetSystemSaveDataPrivateSettings() NN_NOEXCEPT;

//!< システムセーブデータのインメモリモードを有効にします。（デバッグ用）
::nn::Result EnableSystemSaveDataInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT;

} // namespace

::nn::Result GetPrivateSettingsInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::InitialLaunchSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetInitialLaunchSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullInitialLaunchSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataInitialLaunchSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultInitialLaunchSettings;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetPrivateSettingsInitialLaunchSettings(
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::InitialLaunchSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetInitialLaunchSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataInitialLaunchSettings, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetPrivateSettingsExternalSteadyClockSourceId(
    ::nn::settings::system::ClockSourceId* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ClockSourceId Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetExternalSteadyClockSourceId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullExternalSteadyClockSourceIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataExternalSteadyClockSourceId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultExternalSteadyClockSourceId;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetPrivateSettingsExternalSteadyClockSourceId(
    const ::nn::settings::system::ClockSourceId& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ClockSourceId Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetExternalSteadyClockSourceId()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataExternalSteadyClockSourceId,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetPrivateSettingsShutdownRtcValue(int64_t* pOutValue) NN_NOEXCEPT
{
    typedef int64_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetShutdownRtcValue()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullShutdownRtcValueBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataShutdownRtcValue));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultShutdownRtcValue;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetPrivateSettingsShutdownRtcValue(int64_t value) NN_NOEXCEPT
{
    typedef int64_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetShutdownRtcValue()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataShutdownRtcValue,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetPrivateSettingsExternalSteadyClockInternalOffset(
    int64_t* pOutValue) NN_NOEXCEPT
{
    typedef int64_t Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetExternalSteadyClockInternalOffset()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullExternalSteadyClockInternalOffsetBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataExternalSteadyClockInternalOffset));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultExternalSteadyClockInternalOffset;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetPrivateSettingsExternalSteadyClockInternalOffset(
    int64_t value) NN_NOEXCEPT
{
    typedef int64_t Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetExternalSteadyClockInternalOffset()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataExternalSteadyClockInternalOffset,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetPrivateSettingsRequiresRunRepairTimeReviser(
    bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetRequiresRunRepairTimeReviser()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullRequiresRunRepairTimeReviser());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataRequiresRunRepairTimeReviser));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultRequiresRunRepairTimeReviser;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetPrivateSettingsRequiresRunRepairTimeReviser(
    bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetRequiresRunRepairTimeReviser()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataRequiresRunRepairTimeReviser,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadPrivateSettings(
    uint64_t* pOutCount, char outBuffer[], size_t count) NN_NOEXCEPT
{
    typedef char ValueType;
    typedef uint64_t CountType;
    NN_RESULT_DO(
        (GetSystemSettingsValue<ValueType, CountType>)(
            pOutCount, outBuffer, static_cast<CountType>(count),
            "ReadSettings()",
            [] (CountType* pOutCount, ValueType outBuffer[], CountType count
                ) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutCount != nullptr, ResultNullSettingsCountBuffer());
                NN_RESULT_THROW_UNLESS(
                    outBuffer != nullptr, ResultNullSettingsBuffer());
                NN_RESULT_DO(
                    (GetSystemSaveDataValue<ValueType, CountType>)(
                        pOutCount, outBuffer, count,
                        ReadSystemSaveDataPrivateSettings));
                NN_RESULT_SUCCESS;
            },
            [] (CountType* pOutCount, ValueType outValues[], CountType count
                ) NN_NOEXCEPT
            {
                NN_UNUSED(outValues);
                NN_UNUSED(count);

                if (pOutCount != nullptr)
                {
                    *pOutCount = 0;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result EnablePrivateSettingsInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    NN_RESULT_DO(EnableSystemSaveDataInMemoryModeForDebug(buffer, size));

    NN_RESULT_SUCCESS;
}

namespace {

//!< システムアプリケーション設定のデータベースを表す構造体です。
struct SystemApplicationSettingsDatabase final
{
    uint32_t initialLaunchSettingsFlags;
    ::nn::Bit8 _padding[4]; // 書き換え禁止
    ::nn::time::SteadyClockTimePoint initialLaunchSettingsTimeStamp;

    ::nn::Bit8 margin[32];
};

NN_STATIC_ASSERT(sizeof(SystemApplicationSettingsDatabase) == 64);

//!< クロック設定のデータベースを表す構造体です。
struct ClockSettingsDatabase final
{
    ::nn::settings::system::ClockSourceId externalSteadyClockSourceId;
    int64_t shutdownRtcValue;
    int64_t externalSteadyClockInternalOffset;

    ::nn::Bit8 margin[32];
};

NN_STATIC_ASSERT(sizeof(ClockSettingsDatabase) == 64);

//!< 修理工程で設定するフラグのデータベースを表す構造体です。
struct RepairSettingsDatabase final
{
    uint32_t requiresRunRepairTimeReviser;

    ::nn::Bit8 margin[60];
};

NN_STATIC_ASSERT(sizeof(RepairSettingsDatabase) == 64);

//!< プライベート設定データベースのヘッダを表す構造体です。
struct PrivateSettingsDatabaseHeader final
{
    ::nn::Bit8 margin[16];
};

NN_STATIC_ASSERT(sizeof(PrivateSettingsDatabaseHeader) == 16);

//!< プライベート設定データベースを表す構造体です。
struct PrivateSettingsDatabase final
{
    PrivateSettingsDatabaseHeader header;
    SystemApplicationSettingsDatabase systemApplicationSettingsDatabase;
    ClockSettingsDatabase clockSettingsDatabase;
    RepairSettingsDatabase repairSettingsDatabase;

    ::nn::Bit8 margin[261936];
};

NN_STATIC_ASSERT(sizeof(PrivateSettingsDatabase) == 262144);

//!< システムセーブデータを返します。
::nn::Result GetSystemSaveData(SystemSaveData** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemSaveData,
                             sizeof(SystemSaveData),
                         NN_ALIGNOF(SystemSaveData)> s_Storage;

    SystemSaveData& systemSaveData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemSaveData) SystemSaveData();
        systemSaveData.SetSystemSaveDataId(SystemSaveDataId);
        systemSaveData.SetTotalSize(SystemSaveDataTotalSize);
        systemSaveData.SetJournalSize(SystemSaveDataJournalSize);
        systemSaveData.SetFlags(SystemSaveDataFlags);
        systemSaveData.SetMountName(MountName);

        NN_RESULT_DO(systemSaveData.Mount(true));

        s_IsInitialized = true;
    }

    *ppOutValue = &systemSaveData;

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result = getter(pOutValue, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result GetSystemSaveDataValue(
    U* pOutValue,  T outValues[], U count,
    ::std::function<::nn::Result(U*, T*, U, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result =
        getter(pOutValue, outValues, count, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(setter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    const ::nn::Result result = setter(pSystemSaveData, value);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(synchronizes));
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = getter(pOutValue);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataPrivateSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutValue);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_PRIVATE_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutValue);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T, typename U>
::nn::Result GetSystemSettingsValue(
    U* pOutCount, T outValues[], U count,
    const char* const functionName,
    ::std::function<::nn::Result(U*, T*, U) NN_NOEXCEPT> getter,
    ::std::function<void(U*, T*, U) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = getter(pOutCount, outValues, count);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataPrivateSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutCount, outValues, count);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_PRIVATE_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutCount, outValues, count);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(setter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = setter(value);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataPrivateSettings();

        if (result.IsSuccess())
        {
            result = setter(value);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_PRIVATE_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataInitialLaunchSettings(
    ::nn::settings::system::InitialLaunchSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsFlags)),
        &pOutValue->flags, sizeof(pOutValue->flags)));
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsTimeStamp)),
        &pOutValue->timeStamp, sizeof(pOutValue->timeStamp)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataInitialLaunchSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::InitialLaunchSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsFlags)),
        &value.flags, sizeof(value.flags)));
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase,
                     systemApplicationSettingsDatabase) +
            offsetof(SystemApplicationSettingsDatabase,
                     initialLaunchSettingsTimeStamp)),
        &value.timeStamp, sizeof(value.timeStamp)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataExternalSteadyClockSourceId(
    ::nn::settings::system::ClockSourceId* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, externalSteadyClockSourceId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataExternalSteadyClockSourceId(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ClockSourceId& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, externalSteadyClockSourceId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataShutdownRtcValue(
    int64_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, shutdownRtcValue)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataShutdownRtcValue(
    SystemSaveData* pSystemSaveData,
    int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, shutdownRtcValue)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataExternalSteadyClockInternalOffset(
    int64_t* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, externalSteadyClockInternalOffset)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataExternalSteadyClockInternalOffset(
    SystemSaveData* pSystemSaveData,
    int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, clockSettingsDatabase) +
            offsetof(ClockSettingsDatabase, externalSteadyClockInternalOffset)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataRequiresRunRepairTimeReviser(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, repairSettingsDatabase) +
            offsetof(RepairSettingsDatabase, requiresRunRepairTimeReviser)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataRequiresRunRepairTimeReviser(
    SystemSaveData* pSystemSaveData,
    bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(PrivateSettingsDatabase, repairSettingsDatabase) +
            offsetof(RepairSettingsDatabase, requiresRunRepairTimeReviser)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemSaveDataPrivateSettings(
    uint64_t* pOutCount, char outBuffer[], uint64_t count,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto size = ::std::min(
        static_cast<size_t>(count), sizeof(PrivateSettingsDatabase));
    NN_RESULT_DO(pSystemSaveData->Read(0, outBuffer, size));
    *pOutCount = size;
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のプライベート設定をゼロ初期化します。
::nn::Result ClearSystemSaveDataPrivateSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    const auto fileSize = static_cast<int32_t>(sizeof(PrivateSettingsDatabase));
    const ::nn::Bit8 zeros[512] = {};
    for (int32_t restSize = fileSize; 0 < restSize; )
    {
        const auto size = ::std::min(
            static_cast<int32_t>(NN_ARRAY_SIZE(zeros)), restSize);
        NN_RESULT_DO(pSystemSaveData->Write(fileSize - restSize, zeros, size));
        restSize -= size;
    }

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のプライベート設定をリセットします。
::nn::Result ResetSystemSaveDataPrivateSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(ClearSystemSaveDataPrivateSettings(pSystemSaveData));
    NN_RESULT_DO(SetSystemSaveDataInitialLaunchSettings(
                     pSystemSaveData, DefaultInitialLaunchSettings));
    NN_RESULT_DO(SetSystemSaveDataExternalSteadyClockSourceId(
                     pSystemSaveData, DefaultExternalSteadyClockSourceId));
    NN_RESULT_DO(SetSystemSaveDataShutdownRtcValue(
                     pSystemSaveData, DefaultShutdownRtcValue));
    NN_RESULT_DO(SetSystemSaveDataExternalSteadyClockInternalOffset(
                     pSystemSaveData,
                     DefaultExternalSteadyClockInternalOffset));

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSaveDataPrivateSettings() NN_NOEXCEPT
{
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    ::nn::Result result = pSystemSaveData->Create(
        static_cast<int64_t>(sizeof(PrivateSettingsDatabase)));
    NN_RESULT_THROW_UNLESS(
        result.IsSuccess() ||
        ::nn::fs::ResultPathAlreadyExists::Includes(result),
        result);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    result = ResetSystemSaveDataPrivateSettings(pSystemSaveData);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(true));
    NN_RESULT_SUCCESS;
}

::nn::Result EnableSystemSaveDataInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->EnableInMemoryModeForDebug(buffer, size));
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
