﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_KeyboardLayout.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Keyboard.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_BitPack.h>

#include "settings_KeyboardImpl.h"
#include "settings_SystemSettingsServer.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< キーボード配列のフィールド定義です。
struct KeyboardLayoutField
{
    typedef ::nn::util::BitPack<
                ::nn::Bit32, KeyboardLayoutField>::Field<0, 8, uint8_t>
            Layout; //!< キーボード配列の種別
};

//!< キーボード配列のビットパックを扱う型です。
typedef ::nn::util::BitPack<::nn::Bit32, KeyboardLayoutField>
        KeyboardLayoutBitPack;

//!< 自プログラムに要望されているキーボード配列を取得します。
bool GetDesirableKeyboardLayout(::nn::Bit32* pOutValue) NN_NOEXCEPT;

//!< 自プログラムが要望するキーボード配列を指定します。
void SetDesirableKeyboardLayout(::nn::Bit32 value) NN_NOEXCEPT;

} // namespace

::nn::Result GetKeyboardLayout(KeyboardLayout* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullKeyboardLayoutBuffer());
    auto layout = int32_t();
    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->GetKeyboardLayout(&layout));
    *pOutValue = static_cast<KeyboardLayout>(layout);
    NN_RESULT_SUCCESS;
}

::nn::Result SetKeyboardLayout(KeyboardLayout value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->SetKeyboardLayout(static_cast<int32_t>(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetApplicationOwnKeyboardLayout(
    bool* pOutFlag, KeyboardLayout* pOutLayout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFlag);

    NN_RESULT_THROW_UNLESS(
        pOutLayout != nullptr, ResultNullKeyboardLayoutBuffer());

    KeyboardLayoutBitPack bitPack = {};

    if (!GetDesirableKeyboardLayout(&bitPack.storage))
    {
        *pOutFlag = false;
    }
    else
    {
        const auto layout =
            static_cast<KeyboardLayout>(
                bitPack.Get<KeyboardLayoutField::Layout>());

        switch (layout)
        {
        case KeyboardLayout_Japanese:
        case KeyboardLayout_EnglishUs:
        case KeyboardLayout_EnglishUsInternational:
        case KeyboardLayout_EnglishUk:
        case KeyboardLayout_French:
        case KeyboardLayout_FrenchCa:
        case KeyboardLayout_Spanish:
        case KeyboardLayout_SpanishLatin:
        case KeyboardLayout_German:
        case KeyboardLayout_Italian:
        case KeyboardLayout_Portuguese:
        case KeyboardLayout_Russian:
        case KeyboardLayout_Korean:
        case KeyboardLayout_ChineseSimplified:
        case KeyboardLayout_ChineseTraditional:
            *pOutFlag = true;
            *pOutLayout = layout;
            break;

        default:
            // 意図しない値が設定されていた場合は無視
            *pOutFlag = false;
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetApplicationOwnKeyboardLayout(KeyboardLayout value) NN_NOEXCEPT
{
    KeyboardLayoutBitPack bitPack = {};
    bitPack.Set<KeyboardLayoutField::Layout>(static_cast<uint8_t>(value));
    SetDesirableKeyboardLayout(bitPack.storage);
    NN_RESULT_SUCCESS;
}

namespace {

#ifdef NN_BUILD_CONFIG_OS_HORIZON

bool GetDesirableKeyboardLayout(::nn::Bit32* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    return ::nn::applet::GetDesirableKeyboardLayout(pOutValue);
}

void SetDesirableKeyboardLayout(::nn::Bit32 value) NN_NOEXCEPT
{
    ::nn::applet::SetDesirableKeyboardLayout(value);
}

#else

class DesirableKeyboardLayoutHolder final
{
    NN_DISALLOW_COPY(DesirableKeyboardLayoutHolder);
    NN_DISALLOW_MOVE(DesirableKeyboardLayoutHolder);

private:
    ::nn::os::SdkMutex m_Mutex;
    bool m_HasValue;
    ::nn::Bit32 m_Value;

public:
    DesirableKeyboardLayoutHolder() NN_NOEXCEPT
        : m_Mutex()
        , m_HasValue(false)
        , m_Value()
    {
        // 何もしない
    }

    bool Get(::nn::Bit32* pOutValue) NN_NOEXCEPT
    {
        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
        *pOutValue = m_Value;
        return m_HasValue;
    }

    void Set(::nn::Bit32 value) NN_NOEXCEPT
    {
        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
        m_HasValue = true;
        m_Value = value;
    }
};

DesirableKeyboardLayoutHolder& GetDesirableKeyboardLayoutHolder() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(DesirableKeyboardLayoutHolder, s_Holder);

    return s_Holder;
}

bool GetDesirableKeyboardLayout(::nn::Bit32* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    return GetDesirableKeyboardLayoutHolder().Get(pOutValue);
}

void SetDesirableKeyboardLayout(::nn::Bit32 value) NN_NOEXCEPT
{
    GetDesirableKeyboardLayoutHolder().Set(value);
}

#endif

} // namespace

}}} // namespace nn::settings::detail
