﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/settings/factory/settings_BatteryLot.h>
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/factory/settings_GameCard.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_Speaker.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace settings { namespace detail {

//!< 生産時較正情報のマジックナンバー
const uint32_t CalibrationInfoMagicNumber =
    'C' << 0 | 'A' << 8 | 'L' << 16 | '0' << 24;

//!< SHA-256 ハッシュを表す構造体です。
struct Sha256Hash final
{
    ::nn::Bit8 data[32];
};

//!< 生産時較正情報のヘッダーを表す構造体です。
struct CalibrationInfoHeader final
{
    uint32_t magicNumber;   //!< 生産時較正情報のマジックナンバー
    uint32_t version;       //!< 生産時較正情報のフォーマットバージョン
    uint32_t bodySize;      //!< 生産時較正情報の本体のバイト数
    uint16_t model;         //!< 生産時較正情報の機種情報
    uint16_t updateCount;   //!< 生産時較正情報の書き換え回数
    NN_PADDING8;
    NN_PADDING8;
    Sha256Hash bodyHash;    //!< 生産時較正情報の本体の SHA-256 ハッシュ
};

NN_STATIC_ASSERT(sizeof(CalibrationInfoHeader) == 16 * 4);

//!< 構成識別子 1 のブロックを表す構造体です。
struct ConfigurationId1Block final
{
    //!< 構成識別子 1
    ::nn::settings::factory::ConfigurationId1 configurationId1;

    //!< 構成識別子 1 の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ConfigurationId1Block) == 16 * 2);

//!< 無線 LAN に対して設定されたカントリーコードのブロックを表す構造体です。
struct WirelessLanCountryCodesBlock final
{
    //!< 無線 LAN に対して設定されたカントリーコードの数
    uint32_t count;

    NN_PADDING4;

    //!< 無線 LAN に対して設定されたカントリーコード
    ::nn::settings::factory::CountryCode countryCodes[128];

    NN_PADDING6;

    //!< 無線 LAN に対して設定されたカントリーコードの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(WirelessLanCountryCodesBlock) == 16 * 25);

//!< 無線 LAN に対して設定された MAC アドレスのブロックを表す構造体です。
struct WirelessLanMacAddressBlock final
{
    //!< 無線 LAN に対して設定された MAC アドレス
    ::nn::settings::factory::MacAddress macAddress;

    //!< 無線 LAN に対して設定された MAC アドレスの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(WirelessLanMacAddressBlock) == 8);

//!< Bluetooth に対して設定された BD アドレスのブロックを表す構造体です。
struct BluetoothBdAddressBlock final
{
    //!< Bluetooth に対して設定された BD アドレス
    ::nn::settings::factory::BdAddress bdAddress;

    //!< Bluetooth に対して設定された BD アドレスの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(BluetoothBdAddressBlock) == 8);

//!< 加速度センサのオフセット値のブロックを表す構造体です。
struct AccelerometerOffsetBlock final
{
    //!< 加速度センサのオフセット値
    ::nn::settings::factory::AccelerometerOffset accelerometerOffset;

    //!< 加速度センサのオフセット値の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(AccelerometerOffsetBlock) == 8);

//!< 加速度センサのスケール値のブロックを表す構造体です。
struct AccelerometerScaleBlock final
{
    //!< 加速度センサのスケール値
    ::nn::settings::factory::AccelerometerScale accelerometerScale;

    //!< 加速度センサのスケール値の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(AccelerometerScaleBlock) == 8);

//!< ジャイロスコープのオフセット値のブロックを表す構造体です。
struct GyroscopeOffsetBlock final
{
    //!< ジャイロスコープのオフセット値
    ::nn::settings::factory::GyroscopeOffset gyroscopeOffset;

    //!< ジャイロスコープのオフセット値の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(GyroscopeOffsetBlock) == 8);

//!< ジャイロスコープのスケール値のブロックを表す構造体です。
struct GyroscopeScaleBlock final
{
    //!< ジャイロスコープのスケール値
    ::nn::settings::factory::GyroscopeScale gyroscopeScale;

    //!< ジャイロスコープのスケール値の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(GyroscopeScaleBlock) == 8);

//!< 製品シリアル番号のブロックを表す構造体です。
struct SerialNumberBlock final
{
    //!< 製品シリアル番号
    nn::settings::factory::SerialNumber serialNumber;

    NN_PADDING6;

    //!< 製品シリアル番号の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(SerialNumberBlock) == 16 * 2);

//!< ECC-P256 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct EccP256DeviceKeyBlock final
{
    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵
    ::nn::Bit8 deviceKey[48];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccP256DeviceKeyBlock) == 16 * 4);

//!< ECC-P256 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct EccP256DeviceCertificateBlock final
{
    //!< ECC-P256 版のデバイス登録用デバイス証明書
    ::nn::settings::factory::EccP256DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-P256 版のデバイス登録用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccP256DeviceCertificateBlock) == 16 * 25);

//!< ECC-B233 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct EccB233DeviceKeyBlock final
{
    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵
    ::nn::Bit8 deviceKey[48];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccB233DeviceKeyBlock) == 16 * 4);

//!< ECC-B233 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct EccB233DeviceCertificateBlock final
{
    //!< ECC-B233 版のデバイス登録用デバイス証明書
    ::nn::settings::factory::EccB233DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-B233 版のデバイス登録用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccB233DeviceCertificateBlock) == 16 * 25);

//!< ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct EccP256ETicketKeyBlock final
{
    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵
    ::nn::Bit8 deviceKey[48];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccP256ETicketKeyBlock) == 16 * 4);

//!< ECC-P256 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct EccP256ETicketCertificateBlock final
{
    //!< ECC-P256 版の eTicket 発行用デバイス証明書
    ::nn::settings::factory::EccP256DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-P256 版の eTicket 発行用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccP256ETicketCertificateBlock) == 16 * 25);

//!< ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct EccB233ETicketKeyBlock final
{
    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵
    ::nn::Bit8 deviceKey[48];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccB233ETicketKeyBlock) == 16 * 4);

//!< ECC-B233 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct EccB233ETicketCertificateBlock final
{
    //!< ECC-B233 版の eTicket 発行用デバイス証明書
    ::nn::settings::factory::EccB233DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-B233 版の eTicket 発行用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EccB233ETicketCertificateBlock) == 16 * 25);

//!< SSL クライアント秘密鍵のブロックを表す構造体です。
struct SslKeyBlock final
{
    //!< SSL クライアント秘密鍵
    ::nn::Bit8 sslKey[272];

    NN_PADDING8;
    NN_PADDING6;

    //!< SSL クライアント秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(SslKeyBlock) == 16 * 18);

//!< SSL クライアント証明書のバイト数のブロックを表す構造体です。
struct SslCertificateSizeBlock final
{
    //!< SSL クライアント証明書のバイト数
    uint64_t sslCertificateSize;

    NN_PADDING6;

    //!< SSL クライアント証明書のバイト数の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(SslCertificateSizeBlock) == 16 * 1);

//!< SSL クライアント証明書のブロックを表す構造体です。
struct SslCertificateBlock final
{
    //!< SSL クライアント証明書
    ::nn::Bit8 sslCertificate[2048];

    //!< SSL クライアント証明書の SHA-256 ハッシュ
    Sha256Hash sslCertificateHash;
};

NN_STATIC_ASSERT(sizeof(SslCertificateBlock) == 16 * 130);

//!< 乱数のブロックを表す構造体です。
struct RandomNumberBlock final
{
    //!< 乱数
    ::nn::Bit8 randomNumber[4096];

    //!< 乱数の SHA-256 ハッシュ
    Sha256Hash randomNumberHash;
};

NN_STATIC_ASSERT(sizeof(RandomNumberBlock) == 16 * 258);

//!< ゲームカード相互認証秘密鍵のブロックを表す構造体です。
struct GameCardKeyBlock final
{
    //!< ゲームカード相互認証秘密鍵
    ::nn::Bit8 gameCardKey[272];

    NN_PADDING8;
    NN_PADDING6;

    //!< ゲームカード相互認証秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(GameCardKeyBlock) == 16 * 18);

//!< ゲームカード相互認証証明書のブロックを表す構造体です。
struct GameCardCertificateBlock final
{
    //!< ゲームカード相互認証証明書
    ::nn::settings::factory::GameCardCertificate gameCardCertificate;

    //!< ゲームカード相互認証証明書の SHA-256 ハッシュ
    Sha256Hash gameCardCertificateHash;
};

NN_STATIC_ASSERT(sizeof(GameCardCertificateBlock) == 16 * 66);

//!< RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct Rsa2048ETicketKeyBlock final
{
    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵
    ::nn::Bit8 deviceKey[544];

    NN_PADDING8;
    NN_PADDING6;

    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(Rsa2048ETicketKeyBlock) == 16 * 35);

//!< RSA-2048 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct Rsa2048ETicketCertificateBlock final
{
    //!< RSA-2048 版の eTicket 発行用デバイス証明書
    ::nn::settings::factory::Rsa2048DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< RSA-2048 版の eTicket 発行用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(Rsa2048ETicketCertificateBlock) == 16 * 37);

//!< 電池 LOT のブロックを表す構造体です。
struct BatteryLotBlock final
{
    //!< 電池 LOT
    ::nn::settings::factory::BatteryLot batteryLot;

    NN_PADDING6;

    //!< 電池 LOT の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(BatteryLotBlock) == 16 * 2);

//!< スピーカー補正用パラメータのブロックを表す構造体です。
struct CodecIcParameterBlock final
{
    //!< スピーカーのパラメータ
    ::nn::settings::factory::SpeakerParameter speakerParameter;

    //!< 予約領域
    ::nn::Bit8 reserved[1958];

    NN_PADDING8;
    NN_PADDING6;

    //!< スピーカー補正用パラメータの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(CodecIcParameterBlock) == 16 * 129);

//!< 製品リージョンコードのブロックを表す構造体です。
struct RegionCodeBlock final
{
    //!< 製品リージョンコード
    uint32_t regionCode;

    NN_PADDING8;
    NN_PADDING2;

    //!< 製品リージョンコードの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(RegionCodeBlock) == 16 * 1);

//!< ECQV-ECDSA 版の amiibo 用本体秘密鍵のブロックを表す構造体です。
struct EcqvEcdsaAmiiboKeyBlock final
{
    //!< ECQV-ECDSA 版の amiibo 用本体秘密鍵
    ::nn::Bit8 data[84];

    NN_PADDING4;
    NN_PADDING6;

    //!< ECQV-ECDSA 版の amiibo 用本体秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvEcdsaAmiiboKeyBlock) == 16 * 6);

//!< ECQV-ECDSA 版の amiibo 用証明書のブロックを表す構造体です。
struct EcqvEcdsaAmiiboCertificateBlock final
{
    //!< ECQV-ECDSA 版の amiibo 用証明書
    ::nn::Bit8 data[20];

    NN_PADDING8;
    NN_PADDING2;

    //!< ECQV-ECDSA 版の amiibo 用証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvEcdsaAmiiboCertificateBlock) == 16 * 2);

//!< ECQV-ECDSA 版の amiibo 用ルート証明書のブロックを表す構造体です。
struct EcqvEcdsaAmiiboRootCertificateBlock final
{
    //!< ECQV-ECDSA 版の amiibo 用ルート証明書
    ::nn::Bit8 data[112];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECQV-ECDSA 版の amiibo 用ルート証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvEcdsaAmiiboRootCertificateBlock) == 16 * 8);

//!< ECQV-BLS 版の amiibo 用本体秘密鍵のブロックを表す構造体です。
struct EcqvBlsAmiiboKeyBlock final
{
    //!< ECQV-BLS 版の amiibo 用本体秘密鍵
    ::nn::Bit8 data[68];

    NN_PADDING4;
    NN_PADDING6;

    //!< ECQV-BLS 版の amiibo 用本体秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvBlsAmiiboKeyBlock) == 16 * 5);

//!< ECQV-BLS 版の amiibo 用証明書のブロックを表す構造体です。
struct EcqvBlsAmiiboCertificateBlock final
{
    //!< ECQV-BLS 版の amiibo 用証明書
    ::nn::Bit8 data[32];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECQV-BLS 版の amiibo 用証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvBlsAmiiboCertificateBlock) == 16 * 3);

//!< ECQV-BLS 版の amiibo 用ルート証明書のブロックを表す構造体です。
struct EcqvBlsAmiiboRootCertificateBlock final
{
    //!< ECQV-BLS 版の amiibo 用ルート証明書
    ::nn::Bit8 data[144];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECQV-BLS 版の amiibo 用ルート証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(EcqvBlsAmiiboRootCertificateBlock) == 16 * 10);

//!< 本体モデルタイプのブロックを表す構造体です。
struct ProductModelBlock final
{
    //!< 本体モデルタイプ
    int32_t productModel;

    NN_PADDING8;
    NN_PADDING2;

    //!< 本体モデルタイプの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ProductModelBlock) == 16 * 1);

//!< 本体色情報のブロックを表す構造体です。
struct ProductColorBlock final
{
    //!< 本体色情報
    ::nn::Bit8 productColor[6];

    NN_PADDING8;

    //!< 本体色情報の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ProductColorBlock) == 16 * 1);

//!< 輝度レベルからバックライト出力への変換係数のブロックを表す構造体です。
struct BacklightBrightnessCoefficientsBlock final
{
    //!< 輝度レベルからバックライト出力への変換係数
    ::nn::util::Float3 coefficients;

    NN_PADDING2;

    //!< 輝度レベルからバックライト出力への変換係数の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(BacklightBrightnessCoefficientsBlock) == 16 * 1);

//!< ECC-B233 版のデバイス登録用デバイス秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedEccB233DeviceKeyBlock final
{
    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵（拡張版）
    ::nn::Bit8 deviceKey[84];

    NN_PADDING4;
    NN_PADDING6;

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedEccB233DeviceKeyBlock) == 16 * 6);

//!< ECC-P256 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedEccP256ETicketKeyBlock final
{
    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵（拡張版）
    ::nn::Bit8 deviceKey[80];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedEccP256ETicketKeyBlock) == 16 * 6);

//!< ECC-B233 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedEccB233ETicketKeyBlock final
{
    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵（拡張版）
    ::nn::Bit8 deviceKey[80];

    NN_PADDING8;
    NN_PADDING6;

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedEccB233ETicketKeyBlock) == 16 * 6);

//!< RSA-2048 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedRsa2048ETicketKeyBlock final
{
    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵（拡張版）
    ::nn::Bit8 deviceKey[580];

    NN_PADDING4;
    NN_PADDING6;

    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedRsa2048ETicketKeyBlock) == 16 * 37);

//!< SSL クライアント秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedSslKeyBlock final
{
    //!< SSL クライアント秘密鍵（拡張版）
    ::nn::Bit8 sslKey[308];

    NN_PADDING4;
    NN_PADDING6;

    //!< SSL クライアント秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedSslKeyBlock) == 16 * 20);

//!< ゲームカード相互認証秘密鍵（拡張版）のブロックを表す構造体です。
struct ExtendedGameCardKeyBlock final
{
    //!< ゲームカード相互認証秘密鍵（拡張版）
    ::nn::Bit8 gameCardKey[308];

    NN_PADDING4;
    NN_PADDING6;

    //!< ゲームカード相互認証秘密鍵（拡張版）の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(ExtendedGameCardKeyBlock) == 16 * 20);

//!< LCD ベンダ ID のブロックを表す構造体です。
struct LcdVendorIdBlock final
{
    //!< LCD ベンダ ID
    uint32_t lcdVendorId;

    NN_PADDING8;
    NN_PADDING2;

    //!< LCD ベンダ ID の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(LcdVendorIdBlock) == 16 * 1);

//!< RSA-2048 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct Rsa2048DeviceKeyBlock final
{
    //!< RSA-2048 版のデバイス登録用デバイス秘密鍵
    ::nn::Bit8 deviceKey[580];

    NN_PADDING4;
    NN_PADDING6;

    //!< RSA-2048 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(Rsa2048DeviceKeyBlock) == 16 * 37);

//!< RSA-2048 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct Rsa2048DeviceCertificateBlock final
{
    //!< RSA-2048 版のデバイス登録用デバイス証明書
    ::nn::settings::factory::Rsa2048DeviceCertificate deviceCertificate;

    NN_PADDING8;
    NN_PADDING6;

    //!< RSA-2048 版のデバイス登録用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(Rsa2048DeviceCertificateBlock) == 16 * 37);

//!< USB Type-C Power Source 回路バージョンのブロックを表す構造体です。
struct UsbTypeCPowerSourceCircuitBlock final
{
    //!< USB Type-C Power Source 回路バージョン
    ::nn::Bit8 version;

    ::nn::Bit8 _reserved[13];

    //!< USB Type-C Power Source 回路バージョンの CRC-16
    ::nn::Bit16 crc16;
};

NN_STATIC_ASSERT(sizeof(UsbTypeCPowerSourceCircuitBlock) == 16 * 1);

//!< 生産時較正情報の本体を表す構造体です。
struct CalibrationInfoBody final
{
    //!< 構成識別子 1 のブロック
    ConfigurationId1Block configurationId1Block;

    NN_PADDING8;
    NN_PADDING8;
    NN_PADDING8;
    NN_PADDING8;

    //!< 無線 LAN に対して設定されたカントリーコードのブロック
    WirelessLanCountryCodesBlock wirelessLanCountryCodesBlock;

    //!< 無線 LAN に対して設定された MAC アドレスのブロック
    WirelessLanMacAddressBlock wirelessLanMacAddressBlock;

    NN_PADDING8;

    //!< Bluetooth に対して設定された BD アドレスのブロック
    BluetoothBdAddressBlock bluetoothBdAddressBlock;

    NN_PADDING8;

    //!< 加速度センサのオフセット値のブロック
    AccelerometerOffsetBlock accelerometerOffsetBlock;

    //!< 加速度センサのスケール値のブロック
    AccelerometerScaleBlock accelerometerScaleBlock;

    //!< ジャイロスコープのオフセット値のブロック
    GyroscopeOffsetBlock gyroscopeOffsetBlock;

    //!< ジャイロスコープのスケール値のブロック
    GyroscopeScaleBlock gyroscopeScaleBlock;

    //!< 製品シリアル番号のブロック
    SerialNumberBlock serialNumberBlock;

    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵のブロック
    EccP256DeviceKeyBlock eccP256DeviceKeyBlock;

    //!< ECC-P256 版のデバイス登録用デバイス証明書のブロック
    EccP256DeviceCertificateBlock eccP256DeviceCertificateBlock;

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵のブロック
    EccB233DeviceKeyBlock eccB233DeviceKeyBlock;

    //!< ECC-B233 版のデバイス登録用デバイス証明書のブロック
    EccB233DeviceCertificateBlock eccB233DeviceCertificateBlock;

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロック
    EccP256ETicketKeyBlock eccP256ETicketKeyBlock;

    //!< ECC-P256 版の eTicket 発行用デバイス証明書のブロック
    EccP256ETicketCertificateBlock eccP256ETicketCertificateBlock;

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロック
    EccB233ETicketKeyBlock eccB233ETicketKeyBlock;

    //!< ECC-B233 版の eTicket 発行用デバイス証明書のブロック
    EccB233ETicketCertificateBlock eccB233ETicketCertificateBlock;

    //!< SSL クライアント秘密鍵のブロック
    SslKeyBlock sslKeyBlock;

    //!< SSL クライアント証明書のバイト数のブロック
    SslCertificateSizeBlock sslCertificateSizeBlock;

    //!< SSL クライアント証明書のブロック
    SslCertificateBlock sslCertificateBlock;

    //!< 乱数のブロック
    RandomNumberBlock randomNumberBlock;

    //!< ゲームカード相互認証秘密鍵のブロック
    GameCardKeyBlock gameCardKeyBlock;

    //!< ゲームカード相互認証証明書のブロック
    GameCardCertificateBlock gameCardCertificateBlock;

    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロック
    Rsa2048ETicketKeyBlock rsa2048ETicketKeyBlock;

    //!< RSA-2048 版の eTicket 発行用デバイス証明書のブロック
    Rsa2048ETicketCertificateBlock rsa2048ETicketCertificateBlock;

    //!< 電池 LOT のブロック
    BatteryLotBlock batteryLotBlock;

    //!< スピーカー補正用パラメータのブロック
    CodecIcParameterBlock codecIcParameterBlock;

    //!< 製品リージョンコードのブロック
    RegionCodeBlock regionCodeBlock;

    //!< ECQV-ECDSA 版の amiibo 用本体秘密鍵のブロック
    EcqvEcdsaAmiiboKeyBlock ecqvEcdsaAmiiboKeyBlock;

    //!< ECQV-ECDSA 版の amiibo 用証明書のブロック
    EcqvEcdsaAmiiboCertificateBlock ecqvEcdsaAmiiboCertificateBlock;

    //!< ECQV-ECDSA 版の amiibo 用ルート証明書のブロック
    EcqvEcdsaAmiiboRootCertificateBlock ecqvEcdsaAmiiboRootCertificateBlock;

    //!< ECQV-BLS 版の amiibo 用本体秘密鍵のブロック
    EcqvBlsAmiiboKeyBlock ecqvBlsAmiiboKeyBlock;

    //!< ECQV-BLS 版の amiibo 用証明書のブロック
    EcqvBlsAmiiboCertificateBlock ecqvBlsAmiiboCertificateBlock;

    //!< ECQV-BLS 版の amiibo 用ルート証明書のブロック
    EcqvBlsAmiiboRootCertificateBlock ecqvBlsAmiiboRootCertificateBlock;

    //!< 本体モデルタイプのブロック
    ProductModelBlock productModelBlock;

    //!< 本体色情報のブロック
    ProductColorBlock productColorBlock;

    //!< 輝度レベルからバックライト出力への変換係数のブロック
    BacklightBrightnessCoefficientsBlock backlightBrightnessCoefficientsBlock;

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵（拡張版）のブロック
    ExtendedEccB233DeviceKeyBlock extendedEccB233DeviceKeyBlock;

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロック
    ExtendedEccP256ETicketKeyBlock extendedEccP256ETicketKeyBlock;

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロック
    ExtendedEccB233ETicketKeyBlock extendedEccB233ETicketKeyBlock;

    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵（拡張版）のブロック
    ExtendedRsa2048ETicketKeyBlock extendedRsa2048ETicketKeyBlock;

    //!< SSL クライアント秘密鍵（拡張版）のブロック
    ExtendedSslKeyBlock extendedSslKeyBlock;

    //!< ゲームカード相互認証秘密鍵（拡張版）のブロック
    ExtendedGameCardKeyBlock extendedGameCardKeyBlock;

    //!< LCD ベンダ ID のブロック
    LcdVendorIdBlock lcdVendorIdBlock;

    //!< RSA-2048 版のデバイス登録用デバイス秘密鍵のブロック
    Rsa2048DeviceKeyBlock rsa2048DeviceKeyBlock;

    //!< RSA-2048 版のデバイス登録用デバイス証明書のブロック
    Rsa2048DeviceCertificateBlock rsa2048DeviceCertificateBlock;

    //!< USB Type-C Power Source 回路バージョンのブロック
    UsbTypeCPowerSourceCircuitBlock usbTypeCPowerSourceCircuitBlock;

    //!< 予約領域
    ::nn::Bit8 reserved[16 * 990];
};

NN_STATIC_ASSERT(sizeof(CalibrationInfoBody) == 16 * 2044);

//!< 生産時較正情報を表す構造体です。
struct CalibrationInfo final
{
    CalibrationInfoHeader header;   //!< 生産時較正情報のヘッダー
    CalibrationInfoBody body;       //!< 生産時較正情報の本体
};

NN_STATIC_ASSERT(sizeof(CalibrationInfo) == 16 * 2048);

}}}
