﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/TargetConfigs/build_Base.h>

#include "settings_CalibrationFileDatabase.h"
#include "settings_LockableMutexType.h"
#include "settings_Platform.h"

//!< 生産時設定の警告ログを出力します。
#define NN_SETTINGS_FACTORY_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[calibration file] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

#ifdef NN_BUILD_CONFIG_OS_HORIZON

namespace {

//!< マウント名
const char* const MountName = "@CalibFile";

//!< 無線認証画像のファイルパス
const char* const WirelessCertificationFilePath =
    "@CalibFile:/Certifications/WirelessCertification.png";

//!< ファイルを扱うクラスです。
class File final
{
    NN_DISALLOW_COPY(File);
    NN_DISALLOW_MOVE(File);

private:
    //!< ハンドルがオープンされているか否かを表す値
    bool m_IsOpened;

    //!< ファイルハンドル
    ::nn::fs::FileHandle m_Handle;

public:
    File() NN_NOEXCEPT;

    ~File() NN_NOEXCEPT;

    //!< ファイルをオープンします。
    ::nn::Result Open(const char* path, int mode) NN_NOEXCEPT;

    //!< ファイルの内容を読み込みます。
    ::nn::Result Read(size_t* outValue, int64_t offset, void* buffer,
                      size_t size) NN_NOEXCEPT;

    //!< ファイルのサイズを取得します。
    ::nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT;
};

} // namespace

#endif // NN_BUILD_CONFIG_OS_HORIZON

::nn::Result GetCalibFileWirelessCertificationFileSize(
    uint64_t* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullWirelessCertificationFileSizeBuffer());

    ::nn::Result result = ::nn::fs::ResultPartitionNotFound();

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    result = [] (uint64_t* pOutValue) -> ::nn::Result
    {
        File file;
        NN_RESULT_DO(
            file.Open(WirelessCertificationFilePath, ::nn::fs::OpenMode_Read));
        auto value = int64_t();
        NN_RESULT_DO(file.GetSize(&value));
        *pOutValue = static_cast<uint64_t>(value);
        NN_RESULT_SUCCESS;
    } (pOutValue);
#endif

    if (result.IsFailure())
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetCalibFileWirelessCertificationFileSize() failed. "
            "(%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        *pOutValue = 0;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetCalibFileWirelessCertificationFile(
    uint64_t* pOutValue, void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultNullWirelessCertificationFileSizeBuffer());
    NN_RESULT_THROW_UNLESS(outBuffer != nullptr,
                           ResultNullWirelessCertificationFileBuffer());

    ::nn::Result result = ::nn::fs::ResultPartitionNotFound();

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    result = [] (uint64_t* pOutValue, void* outBuffer,
                 size_t size) -> ::nn::Result
    {
        File file;
        NN_RESULT_DO(
            file.Open(WirelessCertificationFilePath, ::nn::fs::OpenMode_Read));
        auto value = size_t();
        NN_RESULT_DO(file.Read(&value, 0, outBuffer, size));
        *pOutValue = static_cast<uint64_t>(value);
        NN_RESULT_SUCCESS;
    } (pOutValue, outBuffer, size);
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
    NN_UNUSED(size);
#endif

    if (result.IsFailure())
    {
        NN_SETTINGS_FACTORY_WARN(
            "GetCalibFileWirelessCertificationFile() failed. "
            "(%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        *pOutValue = 0;
    }

    NN_RESULT_SUCCESS;
}

#ifdef NN_BUILD_CONFIG_OS_HORIZON

namespace {

File::File() NN_NOEXCEPT
    : m_IsOpened(false)
{
    // 何もしない
}

File::~File() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        ::nn::fs::CloseFile(m_Handle);
    }
}

::nn::Result File::Open(const char* path, int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES(!m_IsOpened);

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
            NN_RESULT_DO(
                ::nn::fs::MountBis(
                    ::nn::fs::BisPartitionId::CalibrationFile, MountName));
            s_IsInitialized = true;
        }
    }

    NN_RESULT_DO(::nn::fs::OpenFile(&m_Handle, path, mode));
    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

::nn::Result File::Read(size_t* outValue, int64_t offset, void* buffer,
                        size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(m_IsOpened);
    NN_RESULT_DO(::nn::fs::ReadFile(outValue, m_Handle, offset, buffer, size));
    NN_RESULT_SUCCESS;
}

::nn::Result File::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(m_IsOpened);
    NN_RESULT_DO(::nn::fs::GetFileSize(outValue, m_Handle));
    NN_RESULT_SUCCESS;
}

} // namespace

#endif // NN_BUILD_CONFIG_OS_HORIZON

}}} // namespace nn::settings::detail
