﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/psc.h>

#include <nn/sasbus/sasbus_IManager.sfdl.h>
#include <nn/sasbus/sasbus_ServiceType.h>
#include <nn/sasbus/driver/sasbus_Suspend.h>
#include <nn/sasbus/server/sasbus_ManagerImpl.h>

#include "sasbus_ServerUtil.h"

namespace nn { namespace sasbus { namespace server {

namespace {

    // TORIAEZU : 多めに確保
    const int SessionCountMax = 5;

    struct MyServerManagerOptions
    {
        static const size_t PointerTransferBufferSize = 4096;
    };

    // HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
    class SasbusServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, 1, MyServerManagerOptions>
    {};

    SasbusServerManager g_ServerManager;

    nn::sf::UnmanagedServiceObject<nn::sasbus::IManager, nn::sasbus::server::ManagerImpl>   g_ServiceObject;

    // スリープ等の通知を受け取るオブジェクト
    nn::psc::PmModule g_PmModule;

    bool g_IsHipcServerRegistered = false;
    bool g_IsHipcServerStarted = false;
}

nn::sf::SharedPointer<nn::sasbus::IManager> GetServiceObject() NN_NOEXCEPT
{
    return g_ServiceObject.GetShared();
}

void RegisterHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!g_IsHipcServerRegistered);

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ServerManager.RegisterObjectForPort(g_ServiceObject.GetShared(), SessionCountMax, nn::sasbus::SasbusServiceName));

    g_IsHipcServerRegistered = true;
}

void StartHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(!g_IsHipcServerStarted);

    // PSC へのモジュール登録
    auto result = g_PmModule.Initialize(
            nn::psc::PmModuleId_Sasbus,
            SasbusDependencies,
            sizeof(SasbusDependencies) / sizeof(SasbusDependencies[0]),
            nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    g_ServerManager.Start();

    g_IsHipcServerStarted = true;
}

void LoopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    nn::psc::PmState    state         = nn::psc::PmState_Unknown;
    nn::psc::PmState    previousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  flags;
    static const ServerThreadParam<SasbusServerManager> threadParam =
    {
        &g_PmModule,
        &g_ServerManager,
        StateEdge_MinimumAwakeToSleepReady
    };

    bool ipcEnable = true;

    while (NN_STATIC_CONDITION(true))
    {
        if (ipcEnable)
        {
            // IPC 許可状態
            IpcLoop<SasbusServerManager, nn::psc::PmModuleId_Sasbus>(&previousState, &state, &flags, &threadParam);
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            g_PmModule.GetEventPointer()->Wait();
            g_PmModule.GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));
        }

        auto stateEdge = GetStateEdge(previousState, state);
        switch (stateEdge)
        {
        case StateEdge_MinimumAwakeToSleepReady:
            NN_SDK_LOG("[bus] Sasbus suspend\n");
            nn::sasbus::driver::Suspend();
            ipcEnable = false;
            break;

        case StateEdge_EssentialServicesAwakeToMinimumAwake:
            NN_SDK_LOG("[bus] Sasbus resume\n");
            nn::sasbus::driver::Resume();
            ipcEnable = true;
            break;

        case StateEdge_SleepReadyToEssentialServicesSleepReady:
        case StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:
            ipcEnable = false;
            break;

        default:
            ipcEnable = true;
            break;
        }

        g_PmModule.Acknowledge(state, nn::ResultSuccess());
        previousState = state;
    }
}

void StopHipcServer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_IsHipcServerRegistered);
    NN_SDK_REQUIRES(g_IsHipcServerStarted);

    g_ServerManager.RequestStop();

    g_IsHipcServerStarted = false;
}

}}}
