﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#include <nn/sasbus/sasbus.h>
#include <nn/sasbus/sasbus_ServiceType.h>
#include <nn/sasbus/detail/sasbus_RingLifo.h>

#include <nn/sasbus/sasbus_IManager.sfdl.h>
#include <nn/sasbus/sasbus_ISession.sfdl.h>

namespace nn { namespace sasbus {

namespace {

// TORIAEZU : 1プロセスからは 1つのみセッションを発行する
const int NumberOfManagers = 1;
const int NumberOfSessions = 1;
const int NumberOfClients = NumberOfManagers + NumberOfSessions;

sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<nn::sasbus::IManager>       g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

inline ISession* GetInterface(const Session& session) NN_NOEXCEPT
{
    return static_cast<nn::sasbus::ISession*>(session._handle);
}

}

void Initialize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::sasbus::SasbusServiceName));
}

void InitializeWith(nn::sf::SharedPointer<nn::sasbus::IManager>&& manager) NN_NOEXCEPT
{
    g_Holder.InitializeHolderDirectly(std::move(manager));
}

// 以下は DFC と HIPC とで共通コード
void Finalize() NN_NOEXCEPT
{
    g_Holder.FinalizeHolder();
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSession(Session* pOutSession, SasbusDevice device) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::sasbus::ISession> session;
    auto result = g_Holder.GetObject()->OpenSession(&session, static_cast<std::int32_t>(device));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_handle = session.Detach();
}

// SessionHandle として外部に出したものを明示的に Close するための API
void CloseSession(Session* pSession) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::sasbus::ISession*>(pSession->_handle));
}

// Session がもつ各機能のラップ版
Result Send(Session* pSession, const void* pInData, size_t dataBytes, const char reg) NN_NOEXCEPT
{
    nn::sf::InBuffer inBuffer(static_cast<const char*>(pInData), dataBytes);

    return GetInterface(*pSession)->Send(inBuffer, reg);
}

Result Receive(void* pOutData, Session* pSession, size_t dataBytes, const char reg) NN_NOEXCEPT
{
    nn::sf::OutBuffer outBuffer(static_cast<char*>(pOutData), dataBytes);

    auto result = GetInterface(*pSession)->Receive(outBuffer, reg);
    pOutData = outBuffer.GetPointerUnsafe();

    return result;
}

Result StartPeriodicReceiveMode(Session* pSession, size_t dataSize, char reg, nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;
    pSession->_receiveBuffer = receiveBuffer;
    pSession->_receiveBufferSize = receiveBufferLength;

    nn::os::TransferMemory receiveBuf(receiveBuffer, receiveBufferLength, nn::os::MemoryPermission_ReadWrite);
    sfHandle = nn::sf::NativeHandle(receiveBuf.Detach(), true);

    int64_t microSecondsInterval =  interval.GetMicroSeconds();

    return GetInterface(*pSession)->StartPeriodicReceiveMode(std::move(sfHandle), receiveBufferLength, reg, dataSize, microSecondsInterval);
}

void StopPeriodicReceiveMode(Session* pSession) NN_NOEXCEPT
{
    GetInterface(*pSession)->StopPeriodicReceiveMode();
}

}}
