﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/usb/usb_Pm.h>
#include <nn/usb/pd/usb_Pd.h>

#include "psm_BatteryState.h"
#include "psm_ChargeArbiter.h"
#include "psm_IChargerDriver.h"
#include "psm_IUsbPowerManager.h"
#include "psm_OverlayNotificationSender.h"
#include "psm_SessionManager.h"
#include "psm_SettingsHolder-spec.nx.h"
#include "psm_SupplyRouteManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

//! NX における USB の電力関連のマネージャー
class UsbPowerManager final : public IUsbPowerManager
{
    NN_DISALLOW_COPY(UsbPowerManager);
    NN_DISALLOW_MOVE(UsbPowerManager);

public:
    UsbPowerManager() NN_NOEXCEPT;

    virtual ~UsbPowerManager() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Initialize(
        ChargeArbiter* pChargeArbiter,
        SupplyRouteManager* pSupplyRouteManager,
        BatteryState* pBatteryState,
        IChargerDriver* pChargerDriver,
        SessionManager* pSessionManager,
        OverlayNotificationSender* pNotificationSender,
        const SettingsHolder* pSettingsHolder) NN_NOEXCEPT NN_OVERRIDE;

    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    //! OTG を許可します。
    virtual ::nn::Result AllowOtg() NN_NOEXCEPT NN_OVERRIDE;

    //! OTG を禁止します。
    virtual ::nn::Result DisallowOtg() NN_NOEXCEPT NN_OVERRIDE;

    //! 給電エミュレーションを有効にします。
    virtual ::nn::Result EnableEnoughPowerChargeEmulation() NN_NOEXCEPT NN_OVERRIDE;

    //! 給電エミュレーションを無効にします。
    virtual ::nn::Result DisableEnoughPowerChargeEmulation() NN_NOEXCEPT NN_OVERRIDE;

    //! USB の給電状態の更新イベントを提供します。
    virtual ::nn::os::SystemEventType* GetPowerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //! USB Power Delivery からの通知イベントを提供します。
    virtual ::nn::os::SystemEventType* GetPowerDeliveryNoticeEvent() NN_NOEXCEPT NN_OVERRIDE;

    //! 給電エミュレーション変更後の給電状態の更新イベントを提供します。
    virtual ::nn::os::SystemEventType* GetEnoughPowerChargeEmulationEvent() NN_NOEXCEPT NN_OVERRIDE;

    //! USB の給電状態の更新を行います。
    virtual ::nn::Result Update() NN_NOEXCEPT NN_OVERRIDE;

    //! USB Power Delivery からの通知を処理します。
    virtual ::nn::Result ProcessPowerDeliveryNoticeEvent() NN_NOEXCEPT NN_OVERRIDE;

    //! 給電エミュレーション変更後の給電状態の更新を行います。
    virtual ::nn::Result ProcessEnoughPowerChargeEmulationEvent() NN_NOEXCEPT NN_OVERRIDE;

private:
    //! USB::PM から取得した状態に応じてチャージャと VDD50 の状態を更新します。
    void UpdateChargerAndVdd50() NN_NOEXCEPT;

    //! USB::PM から取得した状態に応じてチャージャの状態を更新します。
    void UpdateChargerType(const ::nn::usb::UsbPowerState& pmState, const ::nn::usb::pd::Status& pdState) NN_NOEXCEPT;

    //! USB::PM から取得した状態に応じて下限電圧を更新します。
    void UpdateInputVoltageLimit(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT;

    //! USB::PM から取得した状態に応じて最大入力電流を更新します。
    void UpdateInputCurrentLimit(const ::nn::usb::UsbPowerState& state) NN_NOEXCEPT;

    //! 給電状態を更新します。
    void UpdatePowerSupply() NN_NOEXCEPT;

    //! PowerRequest イベント発行の有効化状態を更新します。
    ::nn::Result UpdatePowerRequest() NN_NOEXCEPT;

private:
    //! 電池充電マネージャへのポインタ
    ChargeArbiter* m_pChargeArbiter;

    //! 給電経路ルール
    SupplyRouteManager* m_pSupplyRouteManager;

    //! 電池状態
    BatteryState* m_pBatteryState;

    //! チャージャドライバへのポインタ
    IChargerDriver* m_pChargerDriver;

    //! オーバーレイ通知管理
    OverlayNotificationSender* m_pNotificationSender;

    //! セッション管理
    SessionManager* m_pSessionManager;

    //! USB::PD セッション
    ::nn::usb::pd::Session m_UpdSession;

    //! USB::PD の割り込みを検出するイベント
    ::nn::os::SystemEventType m_UpdEvent;

    //! USB::PM クライアント
    ::nn::usb::PmClient m_UpmClient;

    //! OTG を許可するかどうか
    bool m_IsOtgAllowed;

    //! USB::PD の PowerRequest イベントの発行が有効かどうか
    ::std::atomic<bool> m_IsPowerRequestEnabled;

    //! 給電エミュレーションが有効かどうか
    ::std::atomic<bool> m_IsEnoughPowerChargeEmulationEnabled;

    //! 給電エミュレーションの設定変更を受信するイベント
    ::nn::os::SystemEventType m_EnoughPowerChargeEmulationEvent;

    //! VDD50B に給電する電圧の条件を満たすか
    ::std::atomic<bool> m_SufficientVoltageToSupplyToVdd50B;

    //! VDD50B に給電する方向の条件を満たすか
    ::std::atomic<bool> m_CorrectDirectionToSupplyToVdd50B;

    //! デバッグ・プラットフォーム設定
    const SettingsHolder* m_pSettingsHolder;
};

}}}} // namespace nn::psm::driver::detail
