﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Common.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/psm/driver/detail/psm_Constants.h>

#include "psm_BatteryTemperatureLevel.h"
#include "psm_BatteryVoltageLevelForCharge.h"
#include "psm_ChargeParameterRule.h"
#include "psm_ChargeParameterRule-spec.nx.h"
#include "psm_PowerState.h"

namespace nn { namespace psm { namespace driver { namespace detail {

ChargeParameterRule::ChargeParameterRule() NN_NOEXCEPT
    : m_BatteryTemperatureLevel(BatteryTemperatureLevel::Medium)
    , m_BatteryVoltageLevel(BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature)
    , m_PowerState(PowerState::FullAwake)
    , m_ChargeCurrentLimitMilliAmpere(FastChargeCurrentLimitMilliAmpereAwakeStateDefault)
    , m_ChargeVoltageLimitMilliVolt(ChargeVoltageLimitMilliVoltDefault)
    , m_ChargeAllowed(true)
{
    // 何もしない。
}

ChargeParameterRule::~ChargeParameterRule() NN_NOEXCEPT
{
    // 何もしない。
}

void ChargeParameterRule::SetBatteryTemperatureLevel(BatteryTemperatureLevel batteryTemperatureLevel) NN_NOEXCEPT
{
    if ( m_BatteryTemperatureLevel != batteryTemperatureLevel )
    {
        m_BatteryTemperatureLevel = batteryTemperatureLevel;
        Update();
    }
}

void ChargeParameterRule::SetBatteryVoltageLevel(BatteryVoltageLevelForCharge batteryVoltageLevel) NN_NOEXCEPT
{
    if ( m_BatteryVoltageLevel != batteryVoltageLevel )
    {
        m_BatteryVoltageLevel = batteryVoltageLevel;
        Update();
    }
}

void ChargeParameterRule::SetPowerState(PowerState powerState) NN_NOEXCEPT
{
    if ( m_PowerState != powerState )
    {
        m_PowerState = powerState;
        Update();
    }
}

// TODO: PlatformConfiguration に依存して切り替える予定ですが、別クラスにするか単純な内部実装分岐にするかは未定です。
void ChargeParameterRule::Update() NN_NOEXCEPT
{
    auto chargeCurrent = FastChargeCurrentLimitMilliAmpereAwakeStateDefault;
    auto chargeVoltage = ChargeVoltageLimitMilliVoltDefault;
    bool chargeAllowed = true;

    switch ( m_PowerState )
    {
    case PowerState::FullAwake:
        chargeCurrent = FastChargeCurrentLimitMilliAmpereAwakeStateDefault;
        break;
    case PowerState::MinimumAwake:
    case PowerState::SleepCharge:
    case PowerState::SleepDischarge:
        chargeCurrent = FastChargeCurrentLimitMilliAmpereSleepState;
        break;
    case PowerState::ShutdownChargeMain:
        chargeCurrent = FastChargeCurrentLimitMilliAmpereShutdownState;
        break;
    } // NOLINT(style/switch_default)

    switch ( m_BatteryTemperatureLevel )
    {
    case BatteryTemperatureLevel::TooLow:
        chargeCurrent = ::std::min(chargeCurrent, FastChargeCurrentLimitMilliAmpereLowTemperature);
        chargeAllowed = false;
        break;
    case BatteryTemperatureLevel::Low:
        chargeCurrent = ::std::min(chargeCurrent, FastChargeCurrentLimitMilliAmpereLowTemperature);
        chargeAllowed = true;
        break;
    case BatteryTemperatureLevel::Medium:
        chargeAllowed = true;
        break;
    case BatteryTemperatureLevel::High:
        if ( m_BatteryVoltageLevel == BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature )
        {
            chargeAllowed = false;
        }
        else
        {
            chargeVoltage = ChargeVoltageLimitMilliVoltHighTemperature;
            chargeAllowed = true;
        }
        break;
    case BatteryTemperatureLevel::TooHigh:
        if ( m_BatteryVoltageLevel == BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature )
        {
            chargeAllowed = false;
        }
        else
        {
            chargeVoltage = ChargeVoltageLimitMilliVoltHighTemperature;
            chargeAllowed = false;
        }
        break;
    } // NOLINT(style/switch_default)

    if ( m_BatteryVoltageLevel == BatteryVoltageLevelForCharge::DisallowFastSpeedCharge )
    {
        chargeCurrent = ::std::min(chargeCurrent, FastChargeCurrentLimitMilliAmpereAwakeStateLow);
    }

    m_ChargeCurrentLimitMilliAmpere = chargeCurrent;
    m_ChargeVoltageLimitMilliVolt = chargeVoltage;
    m_ChargeAllowed = chargeAllowed;
}

int ChargeParameterRule::GetChargeCurrentLimitMilliAmpere() NN_NOEXCEPT
{
    return m_ChargeCurrentLimitMilliAmpere;
}

int ChargeParameterRule::GetChargeVoltageLimitMilliVolt() NN_NOEXCEPT
{
    return m_ChargeVoltageLimitMilliVolt;
}

bool ChargeParameterRule::GetChargeAllowed() NN_NOEXCEPT
{
    return m_ChargeAllowed;
}

}}}} // namespace nn::psm::driver::detail
