﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include <nn/os/os_Mutex.h>

#include "psm_IChargerDriver.h"
#include "psm_ChargeParameterRule.h"
#include "psm_BatteryTemperatureLevel.h"
#include "psm_BatteryVoltageLevelForCharge.h"
#include "psm_PowerState.h"
#include "psm_SettingsHolder-spec.nx.h"

namespace nn { namespace psm { namespace driver { namespace detail {

//! 調停をして充電状態を決定するクラスです。
class ChargeArbiter final
{
    NN_DISALLOW_COPY(ChargeArbiter);
    NN_DISALLOW_MOVE(ChargeArbiter);

public:
    ChargeArbiter() NN_NOEXCEPT;

    ~ChargeArbiter() NN_NOEXCEPT;

    //! 初期化します。
    ::nn::Result Initialize(IChargerDriver* pChargerDriver, const SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    //! 終了します。
    void Finalize() NN_NOEXCEPT;

    //! 充電中かどうかを取得します。
    bool IsBatteryCharging() NN_NOEXCEPT;

    //! 充電が要求されているかどうかを取得します。
    bool IsChargeRequested() NN_NOEXCEPT;

    //! OTG モード要求中かどうかを取得します。
    bool IsOtgRequested() NN_NOEXCEPT;

    //! 電池温度レベルを設定します。
    ::nn::Result SetBatteryTemperatureLevel(BatteryTemperatureLevel batteryTemperatureLevel) NN_NOEXCEPT;

    //! 電池電圧レベルを設定します。
    ::nn::Result SetBatteryVoltageLevel(BatteryVoltageLevelForCharge batteryVoltageLevel) NN_NOEXCEPT;

    //! 電源ステートを設定してチャージャを適切な設定にします。
    ::nn::Result SetPowerState(PowerState state) NN_NOEXCEPT;

    //! 充電要求を開始します。充電が許可されている場合、充電が開始します。
    ::nn::Result StartChargeRequest() NN_NOEXCEPT;

    //! 充電要求を取り下げて、充電を停止します。
    ::nn::Result StopChargeRequest() NN_NOEXCEPT;

    //! OTG モード要求を開始します。
    ::nn::Result StartOtgRequest() NN_NOEXCEPT;

    //! OTG モード要求を取り下げて、元に戻します。
    ::nn::Result StopOtgRequest() NN_NOEXCEPT;

    //! 満充電通知を受けて充電可否を更新します。
    ::nn::Result NotifyFullCharge() NN_NOEXCEPT;

    //! 満充電通知の取り下げを受けて充電可否を更新します。
    ::nn::Result NotifyNotFullCharge() NN_NOEXCEPT;

    //! 高速充電を有効にします。
    ::nn::Result EnableFastBatteryCharging() NN_NOEXCEPT;

    //! 高速充電を無効にします。
    ::nn::Result DisableFastBatteryCharging() NN_NOEXCEPT;

private:
    //! 充電状態をチャージャに反映します。
    ::nn::Result UpdateCharger() NN_NOEXCEPT;

    //! 充電中かどうかを取得します。
    bool IsBatteryChargingInternal() NN_NOEXCEPT;

private:
    //! システム電源状態
    PowerState m_PowerState;

    //! 充電を要求されているか
    bool m_IsChargeRequested;

    //! OTG モードを要求されているか
    bool m_IsOtgRequested;

    //! チャージャドライバ
    IChargerDriver* m_pChargerDriver;

    //! 充電状態の更新を排他するミューテックス
    ::nn::os::Mutex m_Mutex;

    //! 電池が満充電
    bool m_IsFullCharge;

    //! 充電パラメータの導出ルール
    ChargeParameterRule m_ChargeParameterRule;

    //! 充電電流の上限
    int m_ChargeCurrentLimitCapMilliAmpere;

    //! デバッグ・プラットフォーム設定
    const SettingsHolder* m_pSettingsHolder;
};

}}}} // namespace nn::psm::driver::detail
