﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/result/result_HandlingUtility.h>

#include "../psm_IUsbPowerManager.h"
#include "psm_UsbPowerEventHandler.h"

namespace nn { namespace psm { namespace driver { namespace detail {

UsbPowerEventHandler::UsbPowerEventHandler() NN_NOEXCEPT
    : m_pUsbPowerManager(nullptr)
    , m_UpmMultiWaitHolder()
    , m_EnoughPowerChargeEmulationMultiWaitHolder()
    , m_UsbPmEvent()
    , m_EnoughPowerChargeEmulationEventForTest()
{
    // 何もしない。
}

UsbPowerEventHandler::~UsbPowerEventHandler() NN_NOEXCEPT
{
    // 何もしない。
}

void UsbPowerEventHandler::Initialize(IUsbPowerManager* pUsbPowerManager) NN_NOEXCEPT
{
    m_pUsbPowerManager = pUsbPowerManager;

    ::nn::os::InitializeEvent(&m_UsbPmEvent, false, ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeEvent(&m_EnoughPowerChargeEmulationEventForTest, false, ::nn::os::EventClearMode_ManualClear);

    ::nn::os::InitializeMultiWaitHolder(&m_UpmMultiWaitHolder, m_pUsbPowerManager->GetPowerEvent());
    ::nn::os::InitializeMultiWaitHolder(&m_EnoughPowerChargeEmulationMultiWaitHolder, m_pUsbPowerManager->GetEnoughPowerChargeEmulationEvent());
}

void UsbPowerEventHandler::Finalize() NN_NOEXCEPT
{
    ::nn::os::FinalizeMultiWaitHolder(&m_EnoughPowerChargeEmulationMultiWaitHolder);
    ::nn::os::FinalizeMultiWaitHolder(&m_UpmMultiWaitHolder);

    ::nn::os::FinalizeEvent(&m_EnoughPowerChargeEmulationEventForTest);
    ::nn::os::FinalizeEvent(&m_UsbPmEvent);
}

void UsbPowerEventHandler::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_UpmMultiWaitHolder);
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_EnoughPowerChargeEmulationMultiWaitHolder);
}

void UsbPowerEventHandler::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_EnoughPowerChargeEmulationMultiWaitHolder);
    ::nn::os::UnlinkMultiWaitHolder(&m_UpmMultiWaitHolder);
}

bool UsbPowerEventHandler::HandleEvent() NN_NOEXCEPT
{
    if ( ::nn::os::TryWaitSystemEvent(m_pUsbPowerManager->GetPowerEvent()) )
    {
        // 次回の割り込みを受け付けられるように最初にクリア
        ::nn::os::ClearSystemEvent(m_pUsbPowerManager->GetPowerEvent());

        NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessUsbPmEvent());

        ::nn::os::SignalEvent(&m_UsbPmEvent);

        return true;
    }
    else if ( ::nn::os::TryWaitSystemEvent(m_pUsbPowerManager->GetEnoughPowerChargeEmulationEvent()) )
    {
        // 次回の割り込みを受け付けられるように最初にクリア
        ::nn::os::ClearSystemEvent(m_pUsbPowerManager->GetEnoughPowerChargeEmulationEvent());

        NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessEnoughPowerChargeEmulationEvent());

        ::nn::os::SignalEvent(&m_EnoughPowerChargeEmulationEventForTest);

        return true;
    }

    return false;
}

::nn::Result UsbPowerEventHandler::GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = &m_UsbPmEvent;

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerEventHandler::GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = &m_EnoughPowerChargeEmulationEventForTest;

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerEventHandler::ProcessUsbPmEvent() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pUsbPowerManager->Update());

    NN_RESULT_SUCCESS;
}

::nn::Result UsbPowerEventHandler::ProcessEnoughPowerChargeEmulationEvent() NN_NOEXCEPT
{
    // TODO: 一旦、処理内容を変えないために ProcessEnoughPowerChargeEmulationEvent を呼びますが、
    // ProcessEnoughPowerChargeEmulationEvent は Update と統合して簡略化できます。
    NN_RESULT_DO(m_pUsbPowerManager->ProcessEnoughPowerChargeEmulationEvent());

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
