﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "../psm_BatteryTemperatureManager.h"
#include "../psm_IFuelGaugeDriver.h"
#include "psm_FuelGaugeEventHandler.h"

namespace nn { namespace psm { namespace driver { namespace detail {

FuelGaugeEventHandler::FuelGaugeEventHandler() NN_NOEXCEPT
    : m_pTemperatureManager(nullptr)
    , m_pFuelGaugeDriver(nullptr)
    , m_FuelGaugeInterruptEvent()
    , m_FuelGaugeInterruptMultiWaitHolder()
    , m_FuelGaugeEvent()
{
    // 何もしない。
}

FuelGaugeEventHandler::~FuelGaugeEventHandler() NN_NOEXCEPT
{
    // 何もしない。
}

void FuelGaugeEventHandler::Initialize(BatteryTemperatureManager* pTemperatureManager, IFuelGaugeDriver* pFuelGaugeDriver) NN_NOEXCEPT
{
    m_pTemperatureManager = pTemperatureManager;
    m_pFuelGaugeDriver = pFuelGaugeDriver;

    ::nn::os::InitializeEvent(&m_FuelGaugeEvent, false, ::nn::os::EventClearMode_ManualClear);

    NN_ABORT_UNLESS_RESULT_SUCCESS(pFuelGaugeDriver->BindInterrupt(&m_FuelGaugeInterruptEvent));
    ::nn::os::InitializeMultiWaitHolder(&m_FuelGaugeInterruptMultiWaitHolder, &m_FuelGaugeInterruptEvent);
}

void FuelGaugeEventHandler::Finalize() NN_NOEXCEPT
{
    ::nn::os::FinalizeMultiWaitHolder(&m_FuelGaugeInterruptMultiWaitHolder);

    ::nn::os::FinalizeEvent(&m_FuelGaugeEvent);
}

void FuelGaugeEventHandler::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_FuelGaugeInterruptMultiWaitHolder);
}

void FuelGaugeEventHandler::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_FuelGaugeInterruptMultiWaitHolder);
}

bool FuelGaugeEventHandler::HandleEvent() NN_NOEXCEPT
{
    if ( ::nn::os::TryWaitSystemEvent(&m_FuelGaugeInterruptEvent) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessFuelGaugeInterrupt());
        return true;
    }

    return false;
}

::nn::Result FuelGaugeEventHandler::GetFuelGaugeInterruptForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = &m_FuelGaugeEvent;

    NN_RESULT_SUCCESS;
}

::nn::Result FuelGaugeEventHandler::ProcessFuelGaugeInterrupt() NN_NOEXCEPT
{
    IFuelGaugeDriver::Status status;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeDriver->GetStatus(&status));

    if ( status.IsTemperatureInterruptGenerated() )
    {
        m_pTemperatureManager->Update();
    }

    if ( status.IsChargeInterruptGenerated() )
    {
        // MinimumAwake のとき連続で割り込みが起きないようにする
        NN_RESULT_DO(m_pFuelGaugeDriver->DisableHighChargeInterrupt());
        NN_RESULT_DO(m_pFuelGaugeDriver->DisableLowChargeInterrupt());
        NN_RESULT_DO(m_pFuelGaugeDriver->ClearChargeInterrupt());
        NN_DETAIL_PSM_TRACE("Charge interrupt generated.\n");
    }

    if ( status.IsVoltageInterruptGenerated() )
    {
        // MinimumAwake のとき連続で割り込みが起きないようにする
        NN_RESULT_DO(m_pFuelGaugeDriver->SetLowVoltageThreshold(0));
        NN_RESULT_DO(m_pFuelGaugeDriver->ClearVoltageInterrupt());
        NN_DETAIL_PSM_TRACE("Voltage interrupt generated.\n");
    }

    // 非対応の割り込みについてはログ出力だけしておく
    if ( status.IsBatteryStatusInterruptGenerated() )
    {
        NN_DETAIL_PSM_TRACE("Battery status interrupt generated.\n");
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeDriver->ClearInterrupt(&m_FuelGaugeInterruptEvent));

    ::nn::os::SignalEvent(&m_FuelGaugeEvent);

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
