﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>

#include <nn/psc/detail/psc_Log.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmModuleId.h>
#include <nn/psc/util/psc_Util.h>

#include <algorithm>

namespace nn {
namespace psc {
namespace util {

const char* GetStateNameString(nn::psc::PmState state) NN_NOEXCEPT
{
    switch (state)
    {
        #define NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(name) case nn::psc::PmState_##name: return #name;

        // Support NX spec only to dump with module name
        #if defined(NN_BUILD_CONFIG_SPEC_NX)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(FullAwake)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(MinimumAwake)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(SleepReady)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(EssentialServicesSleepReady)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(EssentialServicesAwake)
        NN_DETAIL_PSC_CASE_RETURN_STATE_NAME(ShutdownReady)
        #endif

        #undef NN_DETAIL_PSC_CASE_RETURN_STATE_NAME

        default: return "???";
    }
}

const char* GetModuleNameString(nn::psc::PmModuleId id) NN_NOEXCEPT
{
    switch (id)
    {
        #define NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(name) case nn::psc::PmModuleId_##name: return #name;

        // Support NX spec only to dump with module name
        #if defined(NN_BUILD_CONFIG_SPEC_NX)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Reserved0)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Reserved1)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Reserved2)

        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Socket)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Usb)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Eth)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Fgm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(PcvClock)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(PcvVoltage)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Gpio)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Pinmux)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Uart)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(I2c)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(I2cPowerBus)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Spi)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Pwm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Psm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Tc)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Omm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Pcie)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Lbl)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Display)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Multimedia)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Hid)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(SocketWlan)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Wlan)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Fs)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Audio)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Tm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(TmHostIo)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Bluetooth)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Bpc)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Fan)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Pcm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Nfc)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Apm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Btm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Nifm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(GpioLow)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Npns)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Lm)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Bcat)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Time)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Pctl)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Erpt)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Eupld)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Friends)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Bgtc)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Account)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Sasbus)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Ntc)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Idle)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Tcap)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(PsmLow)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Ndd)

        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(NvHost)
        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(NvDbgSvc)

        NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME(Spsm)

        // Intentionally not writing default case will invoke build warning if new module id was defined but not added here
        // default: return "???";

        #else
        default: return "???";
        #endif

        #undef NN_DETAIL_PSC_CASE_RETURN_MODULE_NAME

    } // NOLINT(style/switch_default)

    return "???";
}

int32_t GetMaxLevel(
    int moduleCount,
    ModuleTransitionInformation moduleInfoList[]
) NN_NOEXCEPT
{
    int32_t maxLevel = 0;
    for (int i = 0; i < moduleCount; i++)
    {
        auto& info = moduleInfoList[i];
        NN_UNUSED(info); // For release build

        if (info.level > maxLevel)
        {
            maxLevel = info.level;
        }
    }
    return maxLevel;
}

// Dump logs in dot format
void PrintModuleInformation(
    TransitionInformation& transitionInfo,
    int moduleCount,
    ModuleTransitionInformation moduleInfoList[],
    int dependencyCount,
    DependencyInformation dependencyInfoList[]
) NN_NOEXCEPT
{
    const int moduleWidth = 11;
    const int stateWidth  = std::strlen(nn::psc::util::GetStateNameString(transitionInfo.currentState));
    NN_UNUSED(stateWidth);

    // Print dot header
    NN_DETAIL_PSC_INFO("-- dot format analysis graph (copy from here) --\n");
    NN_DETAIL_PSC_INFO("digraph psc_transition_report {\n");

    // Print transition information
    NN_DETAIL_PSC_INFO("  graph [label = \"Last or current transition result: %s To %s result 0x%08x\", ranksep = 3.5];\n",
                nn::psc::util::GetStateNameString(transitionInfo.previousState),
                nn::psc::util::GetStateNameString(transitionInfo.currentState),
                transitionInfo.result
                );
    NN_DETAIL_PSC_INFO("  Total [label = \"Total\\n% 6d us\", style = \"bold,filled\", fillcolor = \"#CCFFCC\"];\n",
                static_cast<uint32_t>(ConvertToTimeSpan(static_cast<nn::os::Tick>(transitionInfo.endTick - transitionInfo.startTick)).GetMicroSeconds())
                );

    int32_t maxLevel = GetMaxLevel(moduleCount, moduleInfoList);
    for (int32_t level = 1; level <= maxLevel; ++level)
    {
        NN_DETAIL_PSC_INFO("  subgraph cluster_level%d { label=\"Level %d\"; rank = same; style=filled; color=lightgrey;\n", level, level);

        // Print module transition information
        for (int i = 0; i < moduleCount; i++)
        {
            auto& info = moduleInfoList[i];
            NN_UNUSED(info); // For release build

            if (info.level != level)
            {
                continue;
            }

            auto moduleName = nn::psc::util::GetModuleNameString(info.id);
            const int moduleSpaces = std::max<int>(moduleWidth - std::strlen(moduleName), 0);
            NN_UNUSED(moduleName); NN_UNUSED(moduleSpaces);// For release build

            if (!info.initialized)
            {
            NN_DETAIL_PSC_INFO("  % *s%03d [label = % *s\"%s (%03d)\\nNot initialized\", fillcolor=gray, style=filled];\n",
                moduleWidth,
                    moduleName,
                    info.id,
                moduleSpaces, "",
                    moduleName,
                    info.id
                );
            }
            else if (info.currentState == info.nextState)
            {
                auto time = static_cast<int32_t>(ConvertToTimeSpan(static_cast<nn::os::Tick>(info.endTick - info.startTick)).GetMicroSeconds());
                if (time < 100000)
                {
                    NN_DETAIL_PSC_INFO("  % *s%03d [label = % *s\"%s (%03d)\\n% *s\\n% 6d us\"];\n",
                        moduleWidth,
                        moduleName,
                        info.id,
                        moduleSpaces, "",
                        moduleName,
                        info.id,
                        stateWidth,
                        GetStateNameString(info.currentState),
                        time
                    );
                }
                else
                {
                    NN_DETAIL_PSC_WARN("  % *s%03d [label = % *s\"%s (%03d)\\n% *s\\n% 6d us\", fillcolor=red, style=filled];\n",
                        moduleWidth,
                        moduleName,
                        info.id,
                        moduleSpaces, "",
                        moduleName,
                        info.id,
                        stateWidth,
                        GetStateNameString(info.currentState),
                        time
                    );
                }
            }
            else
            {
                NN_DETAIL_PSC_WARN("  % *s%03d [label = % *s\"%s (%03d)\\nOn transition %12s->%s\", fillcolor=yellow, style=filled];\n",
                    moduleWidth,
                    moduleName,
                    info.id,
                    moduleSpaces, "",
                    moduleName,
                    info.id,
                    GetStateNameString(info.currentState),
                    GetStateNameString(info.nextState)
                );
            }
        }
        NN_DETAIL_PSC_INFO("  }\n");
    }

    // Print dependencies
    for (int i = 0; i < dependencyCount; i++)
    {
        auto& info = dependencyInfoList[i];
        NN_UNUSED(info); // For release build

        NN_DETAIL_PSC_TRACE("  % *s%03d -> %s%03d;\n",
            moduleWidth,
            nn::psc::util::GetModuleNameString(info.sourceId),
            info.sourceId,
            nn::psc::util::GetModuleNameString(info.destinationId),
            info.destinationId
        );
    }

    // Print dot footer
    NN_DETAIL_PSC_INFO("}\n");
    NN_DETAIL_PSC_INFO("-- dot format analysis graph end --\n");
} // NOLINT(impl/function_size)

}}}
