﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_HipcServer.h>

#include <nn/psc/detail/psc_Log.h>
#include <nn/psc/psc_Result.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmModuleId.h>

#include "psc_PmModuleSfObject.h"
#include "psc_PmModule.h"

namespace nn     {
namespace psc    {
namespace server {

nn::Result PmModuleSfObject::Initialize(
    nn::sf::Out<nn::sf::NativeHandle> handleOut,
    const PmModuleId                  moduleId,
    const nn::sf::InBuffer&           children)
NN_NOEXCEPT
{
    nn::Result           result;
    nn::os::SystemEvent* pEvent;

    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    if ((m_pModule = PmModule::GetModuleById(moduleId)) == nullptr)
    {
        return nn::psc::ResultOutOfMemory();
    }

    m_pModule->AddReference();

    result = m_pModule->Initialize(
                 &pEvent,
                 reinterpret_cast<const PmModuleId*>(children.GetPointerUnsafe()),
                 children.GetSize() / sizeof(PmModuleId));

    if (result.IsFailure())
    {
        if (m_pModule->RemoveReference())
        {
            delete m_pModule;
        }

        m_pModule = nullptr;
    }
    else
    {
        handleOut.Set(nn::sf::NativeHandle(pEvent->GetReadableHandle(), false));
    }

    return result;
}

nn::Result PmModuleSfObject::Finalize()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    return m_pModule != nullptr ?
           m_pModule->Finalize() :
           nn::psc::ResultNotInitialized();
}

nn::Result PmModuleSfObject::GetRequest(
    nn::sf::Out<PmState>   stateSf,
    nn::sf::Out<PmFlagSet> flagsSf)
NN_NOEXCEPT
{
    PmState    state;
    PmFlagSet  flags;
    nn::Result result;

    result = m_pModule != nullptr ?
             m_pModule->GetRequest(&state, &flags) :
             nn::psc::ResultNotInitialized();

    if (result.IsSuccess())
    {
        stateSf.Set(state);
        flagsSf.Set(flags);
    }

    return result;
}

nn::Result PmModuleSfObject::Acknowledge()
NN_NOEXCEPT
{
    NN_DETAIL_PSC_FATAL("Deprecated Acknowledge() was called for module %d. Please rebuild the client system process.\n", m_pModule->Id());
    NN_ABORT();
}

nn::Result PmModuleSfObject::AcknowledgeEx(PmState state)
NN_NOEXCEPT
{
    return m_pModule != nullptr ?
           m_pModule->Acknowledge(state) :
           nn::psc::ResultNotInitialized();
}

PmModuleSfObject::PmModuleSfObject()
NN_NOEXCEPT :
    m_pModule(nullptr)
{

}

PmModuleSfObject::~PmModuleSfObject()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(PmModule::m_ModuleListLock);

    if (m_pModule != nullptr)
    {
        // If this fails, this module can't be removed without breaking dependencies.
        // Skipping through this error here will likely cause problems elsewhere.
        if ( m_pModule->Finalize().IsSuccess() )
        {
            if (m_pModule->RemoveReference())
            {
                delete m_pModule;
            }
        }
        else
        {
            NN_DETAIL_PSC_ERROR("Failed to finalize module %d.\n", m_pModule->Id());
        }
    }
}

}}}
