﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/psc/detail/psc_Log.h>
#include <nn/psc/server/psc_UnimplementedIdList.h>
#include <nn/psc/util/psc_Util.h>

#include "psc_PmModule.h"
#include "psc_PmDependent.h"

namespace nn     {
namespace psc    {
namespace server {

nn::os::Mutex PmModule::m_ModuleListLock(false);
nn::util::IntrusiveList<PmModule, nn::util::IntrusiveListBaseNodeTraits<PmModule>> PmModule::m_ModuleList;

PmModule* PmModule::GetModuleById(PmModuleId id)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ModuleListLock.IsLockedByCurrentThread());

    for (auto pModule  = m_ModuleList.begin();
              pModule != m_ModuleList.end();
              pModule++)
    {
        if (pModule->Id() == id)
        {
            return &(*pModule);
        }
    }

    return new PmModule(id);
}

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
void PmModule::CalculateLevel(int32_t baseLevel) NN_NOEXCEPT
{
    const int32_t currentLevel = baseLevel + 1;

    if (m_Level >= currentLevel)
    {
        return;
    }
    else
    {
        m_Level = currentLevel;
    }

    for (auto pChild  = m_ChildList.begin();
              pChild != m_ChildList.end();
              pChild++)
    {
        pChild->GetModulePointer()->CalculateLevel(currentLevel);
    }
}
#endif

nn::Result PmModule::AddParent(PmModule* pModule)
NN_NOEXCEPT
{
    for (auto pParent  = m_ParentList.begin();
              pParent != m_ParentList.end();
              pParent++)
    {
        if (pModule == pParent->GetModulePointer())
        {
            return ResultSuccess();
        }
    }

    PmDependent* pParent;

    if ((pParent = new PmDependent(pModule)) == nullptr)
    {
        return nn::psc::ResultOutOfMemory();
    }

    m_ParentList.push_front(*pParent);

    return ResultSuccess();
}

nn::Result PmModule::AddChild(PmModule* pModule)
NN_NOEXCEPT
{
    for (auto pChild  = m_ChildList.begin();
              pChild != m_ChildList.end();
              pChild++)
    {
        if (pModule == pChild->GetModulePointer())
        {
            return ResultSuccess();
        }
    }

    PmDependent* pChild;

    if ((pChild = new PmDependent(pModule)) == nullptr)
    {
        return nn::psc::ResultOutOfMemory();
    }

    m_ChildList.push_front(*pChild);

    return ResultSuccess();
}

void PmModule::RemoveParent(PmModule* pModule)
NN_NOEXCEPT
{
    for (auto pParent  = m_ParentList.begin();
              pParent != m_ParentList.end();
              pParent++)
    {
        if (pModule == pParent->GetModulePointer())
        {
            m_ParentList.erase(pParent);
            delete &(*pParent);
            break;
        }
    }
}

void PmModule::RemoveChild(PmModule* pModule)
NN_NOEXCEPT
{
    for (auto pChild  = m_ChildList.begin();
              pChild != m_ChildList.end();
              pChild++)
    {
        if (pModule == pChild->GetModulePointer())
        {
            m_ChildList.erase(pChild);
            delete &(*pChild);
            break;
        }
    }
}

nn::Result PmModule::ValidateDependency(PmModule* pParent, uint32_t iterationCounter)
NN_NOEXCEPT
{
    nn::Result result;

    if (iterationCounter >= MaximumDependencyLevels)
    {
        return nn::psc::ResultMaxDependencyLevels();
    }

    for (auto pDependent  = m_ChildList.begin();
              pDependent != m_ChildList.end();
              pDependent++)
    {
        if (pParent == pDependent->GetModulePointer())
        {
            return nn::psc::ResultCircularDependency();
        }

        if ((result = pDependent->ValidateDependency(pParent, iterationCounter + 1)).IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

uint32_t PmModule::ChildDependencyLevels(uint32_t* pLevelsOut, uint32_t iterationCounter)
NN_NOEXCEPT
{
    if (m_ChildList.empty() || iterationCounter >= MaximumDependencyLevels)
    {
        if (*pLevelsOut < iterationCounter)
        {
            *pLevelsOut = iterationCounter;
        }
    }
    else
    {
        for (auto pChild  = m_ChildList.begin();
                  pChild != m_ChildList.end();
                  pChild++)
        {
            pChild->ChildDependencyLevels(pLevelsOut, iterationCounter + 1);
        }
    }

    return *pLevelsOut;
}

uint32_t PmModule::ParentDependencyLevels(uint32_t* pLevelsOut, uint32_t iterationCounter)
NN_NOEXCEPT
{
    if (m_ParentList.empty() || iterationCounter >= MaximumDependencyLevels)
    {
        if (*pLevelsOut < iterationCounter)
        {
            *pLevelsOut = iterationCounter;
        }
    }
    else
    {
        for (auto pParent  = m_ParentList.begin();
                  pParent != m_ParentList.end();
                  pParent++)
        {
            pParent->ParentDependencyLevels(pLevelsOut, iterationCounter + 1);
        }
    }

    return *pLevelsOut;
}

bool PmModule::IsUnimplementedModule()
NN_NOEXCEPT
{
    for (auto i = 0;
              i < sizeof(UnimplementedModules) / sizeof(UnimplementedModules[0]);
              i++)
    {
        if (UnimplementedModules[i] == m_Id)
        {
            return true;
        }
    }
    return false;
}

nn::Result PmModule::SendRequestToModule(PmState state, PmFlagSet flags)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ModuleLock);

    m_NextState      = state;
    m_RequestedState = state;
    m_Flags          = flags;

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    m_StartTick      = nn::os::GetSystemTick();
#endif

    m_DoneEvent.Clear();
    m_StartEvent.Signal();

    return nn::psc::ResultWaitTransition();
}

nn::Result PmModule::DispatchDescendingOrder(nn::os::SystemEvent** pEventOut, PmState state, PmFlagSet flags, uint32_t iterationCounter)
NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    *pEventOut = nullptr;

    if (iterationCounter >= MaximumDependencyLevels)
    {
        result = nn::psc::ResultMaxDependencyLevels();
    }
    else if (!m_Initialized)
    {
        NN_DETAIL_PSC_WARN("uninitialized dependent module %d.\n", m_Id);

        if (IsUnimplementedModule())
        {
            NN_DETAIL_PSC_WARN("module %d is not implemented.\n", m_Id);
            result = ResultSuccess();
        }
        else
        {
            m_DoneEvent.Clear();
            result = nn::psc::ResultWaitTransition();
        }
    }
    else if (m_RequestedState == state)
    {
        result = nn::psc::ResultWaitTransition();
    }
    else
    {
        for (auto pParent  = m_ParentList.begin();
                  pParent != m_ParentList.end();
                  pParent++)
        {
            result = pParent->DispatchDescendingOrder(
                                pEventOut,
                                state,
                                flags,
                                iterationCounter + 1);

            if (result.IsFailure())
            {
                return result;
            }
        }

        if (m_CurrentState != state)
        {
            result = SendRequestToModule(state, flags);
        }
    }

    if (result <= nn::psc::ResultWaitTransition())
    {
        *pEventOut = &m_DoneEvent;
    }

    return result;
}

nn::Result PmModule::DispatchAscendingOrder(nn::os::SystemEvent** pEventOut, PmState state, PmFlagSet flags, uint32_t iterationCounter)
NN_NOEXCEPT
{
    nn::Result result = ResultSuccess();

    *pEventOut = nullptr;

    if (iterationCounter >= MaximumDependencyLevels)
    {
        result = nn::psc::ResultMaxDependencyLevels();
    }
    else if (!m_Initialized)
    {
        NN_DETAIL_PSC_WARN("uninitialized dependent module %d.\n", m_Id);

        if (IsUnimplementedModule())
        {
            NN_DETAIL_PSC_WARN("module %d is not implemented.\n", m_Id);
            result = ResultSuccess();
        }
        else
        {
            m_DoneEvent.Clear();
            result = nn::psc::ResultWaitTransition();
        }
    }
    else if (m_RequestedState == state)
    {
        result = nn::psc::ResultWaitTransition();
    }
    else
    {
        for (auto pChild  = m_ChildList.begin();
                  pChild != m_ChildList.end();
                  pChild++)
        {
            result = pChild->DispatchAscendingOrder(
                                pEventOut,
                                state,
                                flags,
                                iterationCounter + 1);

            if (result.IsFailure())
            {
                return result;
            }
        }

        if (m_CurrentState != state)
        {
            result = SendRequestToModule(state, flags);
        }
    }

    if (result <= nn::psc::ResultWaitTransition())
    {
        *pEventOut = &m_DoneEvent;
    }

    return result;
}

nn::Result PmModule::DispatchRequest(nn::os::SystemEvent** pEventOut, PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    nn::Result result;

    if (!m_Initialized)
    {
        return ResultSuccess();
    }

    switch (order)
    {
    case PmTransitionOrder_ToHigherPowerState:
        result = DispatchAscendingOrder(pEventOut, state, flags, 0);
        break;
    case PmTransitionOrder_ToLowerPowerState:
        result = DispatchDescendingOrder(pEventOut, state, flags, 0);
        break;
    default:
        result = nn::psc::ResultInvalidArgument();
        break;
    }

    return result;
}

nn::Result PmModule::GetRequest(PmState* pStateOut, PmFlagSet* pFlagsOut)
NN_NOEXCEPT
{
    *pStateOut = m_RequestedState;
    *pFlagsOut = m_Flags;
    return ResultSuccess();
}

nn::Result PmModule::Acknowledge(PmState state)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ModuleLock);

#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    m_EndTick        = nn::os::GetSystemTick();
#endif

    // Acknowledge must be a proper response to current transition request
    // Module may call Acknowledge() twice for single transaction by mistake,
    // and it could break the sequence without this check (see SIGLO-82079)
    if (m_RequestedState == PmState_Unknown || m_RequestedState != state)
    {
        NN_SDK_ASSERT(false,
            "Unsolicited acknowledge was sent from module %d.\n"
            "state = %d, m_RequestedState = %d.\n", m_Id, state, m_RequestedState);
        return ResultSuccess();
    }
    m_CurrentState   = m_RequestedState;
    m_RequestedState = PmState_Unknown;
    m_StartEvent.Clear();
    m_DoneEvent.Signal();

    return ResultSuccess();
}

nn::os::MultiWaitHolderType* PmModule::GetMultiWaitHolderPointer()
NN_NOEXCEPT
{
    return &m_Holder;
}

nn::Result PmModule::Initialize(nn::os::SystemEvent** pEventOut, const PmModuleId* pChildren, const uint32_t childCount)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ModuleListLock.IsLockedByCurrentThread());

    if (m_Initialized)
    {
        return nn::psc::ResultAlreadyInitialized();
    }

    for (uint32_t i = 0; i < childCount; i++)
    {
        PmModuleId id           = pChildren[i];
        PmModule*  pChild       = GetModuleById(id);
        uint32_t   childLevels  = 0;
        uint32_t   parentLevels = 0;
        nn::Result result;

        if (pChild == nullptr)
        {
            return nn::psc::ResultOutOfMemory();
        }

        pChild->AddReference();

        if (pChild == this)
        {
            result = nn::psc::ResultSelfDependency();
        }
        else if (pChild->ChildDependencyLevels(&childLevels, 0) +
                 this->ParentDependencyLevels(&parentLevels, 0) >=
                 MaximumDependencyLevels - 1)
        {
            result = nn::psc::ResultMaxDependencyLevels();
        }
        else if ((result = pChild->ValidateDependency(this, 0)).IsFailure() ||
                 (result = this->AddChild(pChild)).IsFailure() ||
                 (result = pChild->AddParent(this)).IsFailure())
        {
            // If this block failed in first or second operation,
            // child did not get referenced and may be safely deleted after
            // RemoveReference() call below. Failure in the third operation guarantees
            // that extra ref was added to this child. It will be deleted only
            // after both RemoveReference() and ReleaseChildren() are called below.
        }

        if (pChild->RemoveReference())
        {
            delete pChild;
        }

        if (result.IsFailure())
        {
            // module is invalid, roll back all child registrations
            ReleaseChildren();
            return result;
        }
    }

    // Setting current state to FullAwake will avoid dispatch of FullAwake
    // request during system startup. Modules are expected to be in FullAwake
    // at the time of registration.
    m_CurrentState   = PmState_FullAwake;
    m_Initialized    = true;
    *pEventOut       = &m_StartEvent;

    m_DoneEvent.Signal();
    m_StartEvent.Clear();
#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    m_Level = 0;
#endif
    return ResultSuccess();
}

void PmModule::ReleaseChildren()
NN_NOEXCEPT
{
    for (auto pChild  = m_ChildList.begin();
              pChild != m_ChildList.end();
              )
    {
        auto tmp = pChild;
        pChild->RemoveParent(this);
        pChild = m_ChildList.erase(pChild);
        delete &(*tmp);
    }
}

nn::Result PmModule::Finalize()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ModuleListLock.IsLockedByCurrentThread());

    if (!m_Initialized)
    {
        return ResultSuccess();
    }

    if (!(m_ParentList.empty()))
    {
        NN_DETAIL_PSC_ERROR("can't finalize module %d, will break dependencies...\n", m_Id);
        return nn::psc::ResultBreaksDependency();
    }

    ReleaseChildren();

    m_CurrentState   = PmState_Unknown;
    m_RequestedState = PmState_Unknown;
    m_Initialized    = false;
    m_DoneEvent.Signal();

    return ResultSuccess();
}

PmModule::PmModule(PmModuleId id)
NN_NOEXCEPT :
    m_Id(id),
    m_CurrentState(PmState_Unknown),
    m_RequestedState(PmState_Unknown),
    m_NextState(PmState_Unknown),
    m_Initialized(false),
    m_LinkedToMultiWait(false),
    m_ModuleLock(false),
    m_StartEvent(nn::os::EventClearMode_ManualClear, true),
    m_DoneEvent(nn::os::EventClearMode_ManualClear, false)
#if defined(NN_DETAIL_PSC_ENABLE_PERFORMANCE_CHECK)
    , m_Level(0)
#endif
{
    m_ModuleList.push_front(*this);
}

PmModule::~PmModule()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ModuleListLock.IsLockedByCurrentThread());
    NN_ABORT_UNLESS(m_ChildList.empty());
    NN_ABORT_UNLESS(m_ParentList.empty());
    m_ModuleList.erase(m_ModuleList.iterator_to(*this));
}

}}}
