﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_HipcServer.h>

#include <nn/psc/psc_Result.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmModuleId.h>

#include "psc_PmControlSfObject.h"
#include "psc_PmControl.h"

namespace nn     {
namespace psc    {
namespace server {
namespace        {

PmControl g_PmControl;

}

nn::Result PmControlSfObject::Initialize(nn::sf::Out<nn::sf::NativeHandle> handleOut)
NN_NOEXCEPT
{
    nn::Result           result;
    nn::os::SystemEvent* pEvent;

    if ((result = g_PmControl.Initialize(&pEvent)).IsSuccess())
    {
        handleOut.Set(nn::sf::NativeHandle(pEvent->GetReadableHandle(), false));
        m_Initialized = true;
    }

    return result;
}

nn::Result PmControlSfObject::PrintModuleInformation()
NN_NOEXCEPT
{
    return m_Initialized ? g_PmControl.PrintModuleInformation() : nn::psc::ResultNotInitialized();
}

nn::Result PmControlSfObject::DispatchRequest(PmState state, PmFlagSet flags, PmTransitionOrder order)
NN_NOEXCEPT
{
    return m_Initialized ? g_PmControl.DispatchRequest(state, flags, order) : nn::psc::ResultNotInitialized();
}

nn::Result PmControlSfObject::GetState(nn::sf::Out<PmControlState> state)
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        PmControlState controlState;
        g_PmControl.GetState(&controlState);
        state.Set(controlState);
        return ResultSuccess();
    }

    return nn::psc::ResultNotInitialized();
}

nn::Result PmControlSfObject::GetResult()
NN_NOEXCEPT
{
    return m_Initialized ? g_PmControl.GetResult() : nn::psc::ResultNotInitialized();
}

nn::Result PmControlSfObject::Cancel()
NN_NOEXCEPT
{
    return m_Initialized ? g_PmControl.Cancel() : nn::psc::ResultNotInitialized();
}

nn::Result PmControlSfObject::GetModuleInformation(
      nn::sf::Out<nn::psc::util::TransitionInformation> transitionInfo,
      nn::sf::Out<::std::int32_t> moduleCount,
      const nn::sf::OutBuffer& moduleInfo,
      nn::sf::Out<std::int32_t> dependencyCount,
      const nn::sf::OutBuffer& dependencyInfo
) NN_NOEXCEPT
{
    if (m_Initialized)
    {
        nn::psc::util::TransitionInformation info;

        int localModuleCount;
        auto pModuleInfo = reinterpret_cast<nn::psc::util::ModuleTransitionInformation*>(moduleInfo.GetPointerUnsafe());
        int maxModuleCount = moduleInfo.GetSize() / sizeof(nn::psc::util::ModuleTransitionInformation);

        int localDependencyCount;
        auto pDependencyInfo = reinterpret_cast<nn::psc::util::DependencyInformation*>(dependencyInfo.GetPointerUnsafe());
        int maxDependencyCount = dependencyInfo.GetSize() / sizeof(nn::psc::util::DependencyInformation);

        auto result = g_PmControl.GetModuleInformation(
            &info,
            &localModuleCount,
            pModuleInfo,
            maxModuleCount,
            &localDependencyCount,
            pDependencyInfo,
            maxDependencyCount
        );

        if (result.IsFailure())
        {
            return result;
        }

        transitionInfo.Set(info);
        moduleCount.Set(localModuleCount);
        dependencyCount.Set(localDependencyCount);

        return ResultSuccess();
    }

    return nn::psc::ResultNotInitialized();
}

PmControlSfObject::PmControlSfObject()
NN_NOEXCEPT :
    m_Initialized(false)
{

}

PmControlSfObject::~PmControlSfObject()
NN_NOEXCEPT
{
    g_PmControl.Finalize();
}

}}}
