﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/psc/server/psc_Server.h>
#include <nn/psc/psc_PmModule.h>
#include <nn/psc/psc_Result.h>
#include <nn/psc/sfdl/psc.sfdl.h>

#include "psc_SessionManager.h"

namespace nn     {
namespace psc    {
namespace        {

SessionManager g_SessionManager;

} // namespace

nn::Result PmModule::Initialize(
    const  PmModuleId  moduleId,
    const  PmModuleId* pChildList,
    uint32_t           childCount,
    nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    nn::Result result;
    nn::sf::NativeHandle sfHandle;
    nn::sf::SharedPointer<nn::psc::sf::IPmService> sessionObject;
    nn::sf::InBuffer childListForSf(reinterpret_cast<const char*>(pChildList), sizeof(PmModuleId) * childCount);

    if (m_Initialized)
    {
        return nn::psc::ResultAlreadyInitialized();
    }

    if ((result = g_SessionManager.GetObject(&sessionObject)).IsSuccess())
    {
        if ((result = sessionObject->Initialize(&m_InterfaceImpl)).IsSuccess() &&
            (result = m_InterfaceImpl->Initialize(&sfHandle, moduleId, childListForSf)).IsSuccess())
        {
            m_SystemEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
            sfHandle.Detach();
            m_Id = moduleId;
            m_Initialized = true;
        }
        else
        {
            sessionObject   = nullptr;
            m_InterfaceImpl = nullptr;
            g_SessionManager.Release();
        }
    }

    return result;
}

nn::Result PmModule::Finalize()
NN_NOEXCEPT
{
    nn::Result result;

    if (m_Initialized)
    {
        if ((result = m_InterfaceImpl->Finalize()).IsSuccess())
        {
            nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
            m_InterfaceImpl = nullptr;
            m_Initialized   = false;
            g_SessionManager.Release();
        }
    }
    else
    {
        result = nn::psc::ResultNotInitialized();
    }

    return result;
}

nn::Result PmModule::GetRequest(PmState* pStateOut, PmFlagSet* pFlagsOut)
NN_NOEXCEPT
{
    return m_Initialized ?
           m_InterfaceImpl->GetRequest(pStateOut, pFlagsOut) :
           nn::psc::ResultNotInitialized();
}

nn::Result PmModule::Acknowledge(PmState state, nn::Result result)
NN_NOEXCEPT
{
    // We may decide to use result in the future,
    // e.g. to notify of a failure, pass module's state after failure.
    // Modyfing public API at that time will not be possible.
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return m_Initialized ?
           m_InterfaceImpl->AcknowledgeEx(state) :
           nn::psc::ResultNotInitialized();
}

nn::os::SystemEvent* PmModule::GetEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

PmModule::PmModule()
NN_NOEXCEPT
    : m_InterfaceImpl(nullptr),
      m_Initialized(false),
      m_Reserved(0)
{

}

PmModule::~PmModule()
NN_NOEXCEPT
{
    if (m_Initialized)
    {
        m_InterfaceImpl = nullptr;
        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
        g_SessionManager.Release();
    }
}

}}
