﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Version.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/pcm/pcm.h>

#include <nn/profiler/profiler_Result.h>

#include "pmu/profiler_PerfCounterGroups.h"
#include "pmu/profiler_PerfCounterThread.h"
#include "profiler_CodeRewriting.h"
#include "profiler_Comms.h"
#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_HeaderWriter.h"
#include "profiler_Hipc.h"
#include "profiler_LibModule.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Main.h"
#include "profiler_Messages.h"
#include "profiler_Memory.h"
#include "profiler_MemoryApi.h"
#include "profiler_Nvn.h"
#include "profiler_ResultPrivate.h"
#include "profiler_SamplingThread.h"
#include "profiler_SamplingOffset.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_Time.h"
#include "profiler_Vsync.h"


// We assume that this is true in many places. Make sure that it is actually true.
NN_STATIC_ASSERT(sizeof(uintptr_t) <= sizeof(uint64_t));


namespace nn { namespace profiler {


void DumpSettings(SettingsFromThePcGui* settings);


//-------------------------------------------------------------------
// Internal Variables
//-------------------------------------------------------------------
namespace /*anonymous*/
{
    enum ProfileStartType : int
    {
        ProfileStartType_Error,
        ProfileStartType_Pccom,
    };

    struct WaitObjects
    {
        nn::os::MultiWaitType multi_wait;
    };

    struct Globals
    {
        WaitObjects waitObjects;
        SettingsFromThePcGui settings;
        SamplingOffset samplingOffset;
        nn::os::EventType exitEvent;
    };
    Globals* globals;


    int wait_for_other_event(nn::os::EventType* other_event)
    {
        nn::Result result;

        DUMP_CURRENT_LINE();

        nn::os::UnlinkAllMultiWaitHolder(&globals->waitObjects.multi_wait);

        DUMP_CURRENT_LINE();

        nn::os::MultiWaitHolderType other_event_holder;
        nn::os::InitializeMultiWaitHolder(&other_event_holder, other_event);
        nn::os::LinkMultiWaitHolder(&globals->waitObjects.multi_wait, &other_event_holder);

        DUMP_CURRENT_LINE();

        nn::os::MultiWaitHolderType exit_holder;
        nn::os::InitializeMultiWaitHolder(&exit_holder, &globals->exitEvent);
        nn::os::LinkMultiWaitHolder(&globals->waitObjects.multi_wait, &exit_holder);

        DUMP_CURRENT_LINE();

        ProfileStartType start_type;

        while (NN_STATIC_CONDITION(true))
        {
            DEBUG_LOG("Waiting for event...\n");

            nn::os::MultiWaitHolderType *trigger = nullptr;
            trigger = nn::os::WaitAny(&globals->waitObjects.multi_wait);

            if (trigger == &other_event_holder)
            {
                // the thingie we were waiting for is ready now
                DEBUG_LOG("got the signal from whatever event we were waiting for.\n");

                start_type = ProfileStartType_Pccom;
                break;
            }
            else /*if (trigger == &exit_holder)*/
            {
                DEBUG_LOG("Exit requested!\n");
                start_type = ProfileStartType_Error;
                break;
            }
        }

        // Cleanup wait objects created at function start
        DUMP_CURRENT_LINE();
        nn::os::UnlinkAllMultiWaitHolder(&globals->waitObjects.multi_wait);
        nn::os::FinalizeMultiWaitHolder(&other_event_holder);
        nn::os::FinalizeMultiWaitHolder(&exit_holder);
        DUMP_CURRENT_LINE();

        return start_type;
    }


    // VerifySettings
    //  Makes sure that the settings stored in settings packet are okay
    //  If not, correct them so that they are acceptable
    //  If not correctable, return false
    //  If all is good, return true
    bool VerifySettings(SettingsFromThePcGui* settings)
    {
        INFO_LOG("Initial settings:\n--\n");
        DumpSettings(settings);
        INFO_LOG("--\n");

        // Verify that the version numbers match
        // If they do not, return back an error
        if (settings->version < ProfilerMinGuiVersion)
        {
            SendNotificationToPC(
                NotificationErrorVersionMismatch,
                nn::profiler::ResultInvalidArgument());
            INFO_LOG("Incompatible plugin version\n"
                "\tSettings Version : %d\n"
                "\tExpected at least: %d\n",
                settings->version, ProfilerMinGuiVersion);
            return false;
        }

        // Fix up invalid flag settings
        // If attempting an instrumented profile, give this priority
        if (settings->flags & SettingsFromThePcGui::UseInstrumented)
        {
            settings->flags |= SettingsFromThePcGui::UseSimple;
            settings->flags &= static_cast<uint32_t>(~SettingsFromThePcGui::SampleByPerfCounter);

            // Make sure that the trampoline and function to instrument are valid address
            // This means that they should be in the range of valid addresses
            if (!IsValidCodeAddress(static_cast<uintptr_t>(settings->arm_trampoline1)) ||
                !IsValidCodeAddress(static_cast<uintptr_t>(settings->func_to_instrument)))
            {
                if (settings->func_to_instrument != 0)
                {
                    WARNING_LOG("Instrumentation validity test failed: 0x%llx -> 0x%llx\n",
                        settings->func_to_instrument,
                        settings->arm_trampoline1);
                }


                // A code-block only profile is being attempted
                settings->arm_trampoline1 = 0;
                settings->func_to_instrument = 0;
            }
        }

        // Only allow Performance Counters to be enabled if the PMU access has been enabled by
        // the user in Dev Menu.
        bool triedUsingPmu = (settings->perf_counters[0] != pmu::PerfCounter_Disabled);
        if (!pmu::IsAvailable() && triedUsingPmu)
        {
            SendNotificationToPC(NotificationWarningPmuNotAvailable, nn::profiler::ResultInvalidArgument());

            for (int i = 0; i < pmu::PerformanceCounterCount; ++i)
            {
                settings->perf_counters[i] = pmu::PerfCounter_Disabled;
            }
            settings->perf_counter_cycle = 0;
        }

        // Verify that the thread ID that is in the settings packet is valid
        // If it is not, replace it with the main thread
        // At the same time, note which core the thread came from
        //if (settings->thread_id_to_profile != ProfileActiveThreadId)
        //{
        //    settings->thread_id_to_profile = ProfileActiveThreadId;
        //}
        //DEBUG_LOG("profiling thread %d\n", settings->thread_id_to_profile);

        // Ensure that the mask of cores to record only contains valid values
        settings->coreMask &= TargetApplication::GetCoreMask();

        INFO_LOG("Settings to be used for profiling\n--\n");
        DumpSettings(settings);
        INFO_LOG("--\n");

        return true;
    }


    //-------------------------------------------------------------------
    // Profiler Header setup
    //-------------------------------------------------------------------
    void SetupMasterHeader(SettingsFromThePcGui* settings)
    {
        // Now fill the header into the buffer
        Header* header = GetMasterHeader();
        header->Initialize();
        header->WriteControlValueOnly(HeaderSpecialValues_MasterHeaderBegin);
        header->Write(HeaderSpecialValues_HeaderVersion, ProfilerHeaderVersion);

        header->Write(HeaderSpecialValues_RuntimeVersion, ProfilerRuntimeVersion);

        header->Write(HeaderSpecialValues_RuntimeSdkVersion, static_cast<uint32_t>(NN_SDK_CURRENT_VERSION_NUMBER));

        header->Write(HeaderSpecialValues_ApplicationSdkVersion, TargetApplication::GetCurrent()->GetSdkVersion());

        header->Write(HeaderSpecialValues_ApplicationPointerSize, static_cast<uint32_t>(sizeof(uintptr_t)));

        // Section code for Dlls
        // Neb_Note: This is inserted here, but managed by the PC side.
        // ref: ProfilerInput.cs@832 in SaveProfileToFile()
        header->Write(HeaderSpecialValues_Dlls, 0u);

        header->Write(HeaderSpecialValues_CoresRecorded, settings->coreMask);

        // Neb_Note : Passing the Units Per Second before the base time so that the
        //    base time can be properly converted to microseconds.
        header->Write(
            HeaderSpecialValues_TimeUnitsPerSecond,
            GetTimeFrequency());

        header->Write(HeaderSpecialValues_BaseTime, GetCurrentTime());
    }
}


void ProfilerMainThread(/* nn::os::EventType* */ void* arg)
{
    nn::Result result;
    bool hasPerfCounters = false;

    INFO_LOG("Made it into ProfilerMainThread!\n");

    globals = Memory::GetInstance()->Allocate<Globals>();
    memset(globals, 0, sizeof(Globals));
    new (globals) Globals;

    nn::os::InitializeEvent(&globals->exitEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::ClearEvent(&globals->exitEvent);

    // Give the profiler some time to stabilize
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    // Dump debugging information
    nn::os::SetThreadName(nn::os::GetCurrentThread(), "NX CPU Profiler: Main Thread");
    DumpThreadInformation();

    // start PCCOM
    InitializeCommunicationsLayer();
    DEBUG_LOG("Communications online\n");

    // setup NVN storage
    nn::profiler::InitializeNvnSupport();

    // Set up our thread that will be in charge of getting performance counter
    //  data
    DEBUG_LOG("Setting up performance counter threads\n");
    nn::profiler::pmu::Initialize();

    // Give PCCOM some time to start up
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    DEBUG_LOG("NX CPU Profiler Started\n");

    TargetApplication::Initialize();
    DEBUG_LOG("Target application initialized\n");

    // Initialize Core and Master Headers
    InitializeHeaders();

    DEBUG_LOG("Starting sampling threads\n");
    result = InitializeSamplingThreads();
    if (result.IsFailure())
    {
        ERROR_LOG("Error starting sampling threads!\n");
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT();
    }

    if (MemoryHeapWasRegistered())
    {
        DEBUG_LOG("Start memory server\n");
        ConnectToMemoryProfiler();
    }

    InitializeVsyncSupport();

    DEBUG_LOG("Initialization nominal\n");

    // Initialize wait objects
    nn::os::InitializeMultiWait(&globals->waitObjects.multi_wait);

    {
        // After we signal this event it will be destroyed, don't attempt to use it again
        nn::os::EventType* waitEvent = reinterpret_cast<nn::os::EventType*>(arg);
        nn::os::SignalEvent(waitEvent);
        arg = nullptr;
        if (GetMainThreadLastError().IsFailure())
        {
            FinalizeMainThread();
        }
    }

    {
        const char* nvnFuncs;
        size_t nvnFuncSize;
        bool found = nn::profiler::GetNvnFunctionPointers(&nvnFuncs, &nvnFuncSize);
        INFO_LOG("NVN Functions - %d - 0x%p - %lld\n", found, nvnFuncs, nvnFuncSize);
    }

    DUMP_CURRENT_LINE();
    while (!nn::os::TryWaitEvent(&globals->exitEvent))
    {
        DEBUG_LOG("** main loop / stage 1.\n");
        nn::os::EventType *beginEvent = CheckForBegin(&globals->settings);
        DUMP_CURRENT_LINE();
        int startType = wait_for_other_event(beginEvent);

        if (startType == ProfileStartType_Error)
        {
            nn::os::ClearEvent(beginEvent);
            continue;
        }

        SampleBuffers::GetInstance()->Reset();

        DEBUG_LOG("** received start request\n");

        TargetApplication::GetCurrent()->FindCodeRegions();
        TargetApplication::GetCurrent()->ClearThreadList();

        if (VerifySettings(&globals->settings) == false)
        {
            INFO_LOG("Invalid settings... Restarting loop.\n");
            nn::os::ClearEvent(beginEvent);
            continue;
        }

        SetupMasterHeader(&globals->settings);

        if (IsPCConnected())
        {
            DEBUG_LOG("profiler: Sending message that profiling has started\n");
            SendProfilingHasBegun();
            DEBUG_LOG("profiler: message sent\n");
        }

        globals->samplingOffset.SetBaseValue(
            globals->settings.requested_time_between_samples_in_nanoseconds);

        DUMP_CURRENT_LINE();

        if (globals->settings.flags & SettingsFromThePcGui::UseInstrumented
            && IsValidCodeAddress(static_cast<uintptr_t>(globals->settings.func_to_instrument)))
        {
            DEBUG_LOG("Attempting to instrument\n");
        }

        // and then... begin.
        DEBUG_LOG("** main loop / stage 2.\n");

        // MAIN PROFILING LOOP
        DUMP_CURRENT_LINE();

        StartProfilingSamplingThreads(&globals->settings);

        nn::os::ClearEvent(beginEvent);
        nn::os::WaitEvent(GetStopEvent());

        DUMP_CURRENT_LINE();

        StopProfilingSamplingThreads();

        DUMP_CURRENT_LINE();

        CheckForNewModules();
        CloseActiveModules();

        if (globals->settings.flags & SettingsFromThePcGui::UseInstrumented
            && IsValidCodeAddress(static_cast<uintptr_t>(globals->settings.func_to_instrument)))
        {
            //InstrumentRemoval(0);
        }

        hasPerfCounters = false;

        // Wait for all buffers to be closed out
        WaitCoresClosed();

        // ok, we're all done profiling now. Beam the data out.
        // Don't bother beaming out the data if we are trying to exit
        if (!nn::os::TryWaitEvent(&globals->exitEvent))
        {
            TargetApplication::GetCurrent()->RegisterAllThreads();

            // Now send the raw data.
            DUMP_CURRENT_LINE();
            DEBUG_LOG("** main loop / stage 3.\n");
            {
                nn::os::EventType *other_event = SendDataToPC();
                if (other_event != nullptr)
                {
                    wait_for_other_event(other_event);
                    SampleBuffers::GetInstance()->Reset();
                }
            }
        }
        // Transfer is now completed.

        DUMP_CURRENT_LINE();
    }

    DumpResultInformation(LOG_AS_DEBUG, result);

    FinalizeVsyncSupport();

    nn::os::FinalizeMultiWait(&globals->waitObjects.multi_wait);

    DisconnectFromMemoryProfiler();

    FinalizeSamplingThreads();

    FinalizeHeaders();

    TargetApplication::Finalize();

    nn::profiler::pmu::Finalize();

    nn::profiler::FinalizeNvnSupport();

    FinalizeCommunicationsLayer();

    nn::os::FinalizeEvent(&globals->exitEvent);

    Memory::GetInstance()->Free(globals);
    globals = nullptr;

    DEBUG_LOG("****** WE ARE DONE ****** \n");
} // NOLINT(impl/function_size)


nn::Result GetMainThreadLastError()
{
    return nn::ResultSuccess();
}


void FinalizeMainThread()
{
    // Kick the main profiling thread so that it will shutdown gracefully
    SendBasicIpcMessage(ProfilerIpcMessage_StopProfiling, 0);
    StopProfilingSamplingThreads();
    nn::os::SignalEvent(&globals->exitEvent);
}


SettingsFromThePcGui* GetProfilerSettingsPointer()
{
    return &globals->settings;
}


void DumpSettings(SettingsFromThePcGui* settings)
{
    NN_UNUSED(settings); // when logging is disabled

    INFO_LOG("Settings:\n");
    INFO_LOG(" Version: %d\n", settings->version);
    INFO_LOG(" Flags: 0x%p\n", settings->flags);
    INFO_LOG(" Perf Counters:\n");
    for (int i = 0; i < pmu::PerformanceCounterCount; ++i)
    {
        INFO_LOG("    %d: 0x%02x\n", i, settings->perf_counters[i]);
    }
    INFO_LOG(" PerfCounterCycle: %d\n", settings->perf_counter_cycle);
    INFO_LOG(" Instrument: 0x%p\n", settings->func_to_instrument);
    INFO_LOG(" Trampoline: 0x%p\n", settings->arm_trampoline1);
    INFO_LOG(" Profile Cores: %x\n", settings->coreMask);
}

} // profiler
} // nn
