﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>

#include <nn/nn_Common.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_BitUtil.h>
#include <nn/profiler/profiler_Memory.h>
#include <nn/profiler/profiler_Api.h>

#include "profiler_Defines.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_MemoryApi.h"
#include "profiler_Time.h"

namespace nn { namespace profiler {

namespace
{
    struct HeapHandles
    {
        void const* startAddress;
        void const* endAddress;
        nn::profiler::MemoryHeapHandle handle;
    };
    HeapHandles gHeapHandles[NN_PROFILER_MEMORY_MAX_HEAPS];

    size_t gHandleCount = 0;

    nn::profiler::MemoryHeapHandle GetHeapHandle(void const* address)
    {
        for (int i = int(gHandleCount) - 1; i >= 0; --i)
        {
            auto item = gHeapHandles[i];
            if (item.startAddress <= address && address < item.endAddress)
            {
                return item.handle;
            }
        }
        return gHeapHandles[0].handle;
    }

    const size_t TrackingIdListSize = SupportedCoreCount;

    struct TrackingIdList
    {
        uintptr_t m_freeList;

        struct TrackingData
        {
            uint64_t time;
        } m_data[TrackingIdListSize];

        TrackingIdList()
        {
            m_freeList = ((1 << TrackingIdListSize) - 1);
        }

        TrackingData* GetFreeData()
        {
            bool failure;
            uint64_t time;
            uintptr_t node;

            while (true)
            {
                do
                {
                    uintptr_t value = __builtin_arm_ldaex(&m_freeList);
                    node = nn::util::isols1b(value);
                    time = GetCurrentTime();
                    value &= ~node;
                    failure = __builtin_arm_stlex(value, &m_freeList);
                } while (failure);

                if (node != 0) { break; }

                // todo: Make this a waitable event instead of polling?
                //       Would need to work without profiler being initialized.
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));
            }

            int index = nn::util::cntt0(node);
            return &m_data[index];
        }

        void ReleaseData(TrackingData* data)
        {
            if (data == nullptr) { return; }

            bool failure = false;
            ptrdiff_t index = (data - m_data);

            NN_SDK_ASSERT(static_cast<size_t>(index) < TrackingIdListSize);

            do
            {
                uintptr_t value = __builtin_arm_ldaex(&m_freeList);
                value |= (1 << index);
                failure = __builtin_arm_stlex(value, &m_freeList);
            } while (failure);
        }
    } gTrackingList;
}


void TrackMalloc(void const* address, size_t size)
{
    TrackMalloc(address, size, nullptr, 0);
}


void TrackMalloc(void const* address, size_t size, char const* systemName)
{
    TrackMalloc(address, size, systemName, 0);
}


void TrackMalloc(void const* address, size_t size, char const* systemName, int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    BeginMemoryAllocation(heap);
    EndMemoryAllocation(heap, address, size);
}


void TrackCalloc(void const* address, size_t num, size_t size)
{
    TrackCalloc(address, num, size, nullptr, 0);
}


void TrackCalloc(void const* address, size_t num, size_t size, char const* systemName)
{
    TrackCalloc(address, num, size, systemName, 0);
}


void TrackCalloc(void const* address, size_t num, size_t size, char const* systemName, int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    BeginMemoryAllocation(heap);
    EndMemoryAllocation(heap, address, num * size);
}


void TrackRealloc(void const* oldAddress, void const* address, size_t size)
{
    TrackRealloc(oldAddress, address, size, nullptr, 0);
}


void TrackRealloc(void const* oldAddress, void const* address, size_t size, char const* systemName)
{
    TrackRealloc(oldAddress, address, size, systemName, 0);
}


void TrackRealloc(void const* oldAddress, void const* address, size_t size, char const* systemName, int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    BeginMemoryReallocation(heap);
    EndMemoryReallocation(heap, oldAddress, address, size);
}


void TrackFree(void const* address)
{
    TrackFree(address, nullptr);
}


void TrackFree(void const* address, char const* systemName)
{
    NN_UNUSED(systemName);

    auto heap = GetHeapHandle(address);
    BeginMemoryFree(heap);
    EndMemoryFree(heap, address);
}


void TrackNew(void const* address, size_t size)
{
    TrackNew(address, size, nullptr, 0, false);
}


void TrackNew(void const* address, size_t size, char const* systemName)
{
    TrackNew(address, size, systemName, 0, false);
}


void TrackNew(void const* address, size_t size, char const* systemName, int alignment)
{
    TrackNew(address, size, systemName, alignment, false);
}


void TrackNew(void const* address, size_t size, char const* systemName, bool isVector)
{
    TrackNew(address, size, systemName, 0, isVector);
}


void TrackNew(void const* address, size_t size, char const* systemName, int alignment, bool isVector)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);
    NN_UNUSED(isVector);

    auto heap = GetHeapHandle(address);
    BeginMemoryAllocation(heap);
    EndMemoryAllocation(heap, address, size);
}


void TrackDelete(void const* address)
{
    TrackDelete(address, nullptr, false);
}


void TrackDelete(void const* address, char const* systemName)
{
    TrackDelete(address, systemName, false);
}


void TrackDelete(void const* address, char const* systemName, bool isVector)
{
    NN_UNUSED(systemName);
    NN_UNUSED(isVector);

    auto heap = GetHeapHandle(address);
    BeginMemoryFree(heap);
    EndMemoryFree(heap, address);
}


TrackingId const GenerateNextTrackingId()
{
    TrackingIdList::TrackingData* id = gTrackingList.GetFreeData();
    NN_SDK_ASSERT(id != nullptr);
    SetMemoryApiLock();
    return reinterpret_cast<TrackingId const>(id);
}


void TrackMalloc(TrackingId const &id, void const* address, size_t size)
{
    TrackMalloc(id, address, size, nullptr, 0);
}


void TrackMalloc(TrackingId const &id, void const* address, size_t size, char const* systemName)
{
    TrackMalloc(id, address, size, systemName, 0);
}


void TrackMalloc(TrackingId const &id, void const* address, size_t size, char const* systemName, int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    EndMemoryAllocation(heap, address, size);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void TrackCalloc(TrackingId const &id, void const* address, size_t num, size_t size)
{
    TrackCalloc(id, address, num, size, nullptr, 0);
}


void TrackCalloc(TrackingId const &id, void const* address, size_t num, size_t size, char const* systemName)
{
    TrackCalloc(id, address, num, size, systemName, 0);
}


void TrackCalloc(TrackingId const &id, void const* address, size_t num, size_t size, char const* systemName, int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    EndMemoryAllocation(heap, address, num * size);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void TrackRealloc(TrackingId const &id, void const* oldAddress, void const* address, size_t size)
{
    TrackRealloc(id, oldAddress, address, size, nullptr, 0);
}


void TrackRealloc(TrackingId const &id, void const* oldAddress, void const* address, size_t size, char const* systemName)
{
    TrackRealloc(id, oldAddress, address, size, systemName, 0);
}


void TrackRealloc(
    TrackingId const &id,
    void const* oldAddress,
    void const* address,
    size_t size,
    char const* systemName,
    int alignment)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);

    auto heap = GetHeapHandle(address);
    EndMemoryReallocation(heap, oldAddress, address, size);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void TrackFree(TrackingId const &id, void const* address)
{
    TrackFree(id, address, nullptr);
}


void TrackFree(TrackingId const &id, void const* address, char const* systemName)
{
    NN_UNUSED(systemName);

    auto heap = GetHeapHandle(address);
    EndMemoryFree(heap, address);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void TrackNew(TrackingId const &id, void const* address, size_t size)
{
    TrackNew(id, address, size, nullptr, 0, false);
}


void TrackNew(TrackingId const &id, void const* address, size_t size, char const* systemName)
{
    TrackNew(id, address, size, systemName, 0, false);
}


void TrackNew(TrackingId const &id, void const* address, size_t size, char const* systemName, int alignment)
{
    TrackNew(id, address, size, systemName, alignment, false);
}


void TrackNew(TrackingId const &id, void const* address, size_t size, char const* systemName, bool isVector)
{
    TrackNew(id, address, size, systemName, 0, isVector);
}


void TrackNew(TrackingId const &id, void const* address, size_t size, char const* systemName, int alignment, bool isVector)
{
    NN_UNUSED(systemName);
    NN_UNUSED(alignment);
    NN_UNUSED(isVector);

    auto heap = GetHeapHandle(address);
    EndMemoryAllocation(heap, address, size);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void TrackDelete(TrackingId const &id, void const* address)
{
    TrackDelete(id, address, nullptr, false);
}


void TrackDelete(TrackingId const &id, void const* address, char const* systemName)
{
    TrackDelete(id, address, systemName, false);
}


void TrackDelete(TrackingId const &id, void const* address, char const* systemName, bool isVector)
{
    NN_UNUSED(systemName);
    NN_UNUSED(isVector);

    auto heap = GetHeapHandle(address);
    EndMemoryFree(heap, address);

    auto data = reinterpret_cast<TrackingIdList::TrackingData*>(id);
    gTrackingList.ReleaseData(data);
}


void RecordHeapRange(void const* startAddress, void const* endAddress, char const* heapName)
{
    size_t index;

    do
    {
        index = __builtin_arm_ldaex(&gHandleCount);
    } while (__builtin_arm_stlex(index + 1, &gHandleCount));

    if (index < NN_PROFILER_MEMORY_MAX_HEAPS)
    {
        HeapHandles& item = gHeapHandles[index];
        item.handle = RegisterMemoryHeap(heapName);
        item.startAddress = startAddress;
        item.endAddress = endAddress;
    }

    // The user has registered a heap.
    // If we haven't already tried to start the memory profiler, start it now.
    // After this point, the memory profiler will be started when the profiler is initialized.
    if (index == 0 && GetProfilerStatus() != ProfilerStatus_Offline)
    {
        ConnectToMemoryProfiler();
    }
}


bool MemoryHeapWasRegistered()
{
    return gHandleCount > 0;
}


}} // nn::profiler
