﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Event.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/profiler/profiler_Result.h>

#include "profiler_CommMessages.h"
#include "profiler_Hipc.h"
#include "profiler_IpcEvent.h"
#include "profiler_IpcEventLoop.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Main.h"
#include "profiler_Memory.h"
#include "profiler_Nvn.h"
#include "profiler_ResultPrivate.h"
#include "profiler_SamplingThread.h"
#include "profiler_SettingsFromThePcGui.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_Workarea.h"

namespace nn { namespace profiler {

namespace /*anonymous*/
{
    const size_t IpcEventStackSize = 8 * 1024;
    struct Globals
    {
        nn::os::ThreadType ipcEventThread;
        void *ipcEventStack;

        OnSendDataCompleteCallback sendDataCompleteCallback;
        OnPcMessageRecievedCallback receivedMessageCallback;

        bool shouldExit;

        nn::os::EventType exitEvent;

    } globals;


    void IpcEventHandler_Status(IpcEventInfo& info);
    void IpcEventHandler_MessageCallback(IpcEventInfo& info);
    void IpcEventHandler_PcMessageForward(IpcEventInfo& info);
    void IpcEventHandler_InstrumentationBuffer(IpcEventInfo& info);
    void IpcEventHandler_SendData(IpcEventInfo& info);


    void IpcEventHandler_Status(IpcEventInfo& info)
    {
        INFO_LOG("Server changed status to %d\n", info.info.status.status);
        SetProfilerStatus(info.info.status.status);
    }


    void IpcEventHandler_MessageCallback(IpcEventInfo& info)
    {
        INFO_LOG("Message from process: %d, 0x%08x\n",
            info.info.messageCallback.id,
            info.info.messageCallback.resultCode);
        nn::Result messageResult = nn::result::detail::ConstructResult(
            info.info.messageCallback.resultCode);
        if (globals.sendDataCompleteCallback != nullptr)
        {
            globals.sendDataCompleteCallback(info.info.messageCallback.id, messageResult);
        }
    }


    void IpcEventHandler_PcMessageForward(IpcEventInfo& info)
    {
        void* data = nullptr;
        bool allocated = false;
        if (info.info.pcMessageForward.size > 0)
        {
            if (info.info.pcMessageForward.message == ProfilerCommMessage_Start)
            {
                auto settings = GetProfilerSettingsPointer();
                NN_SDK_ASSERT_NOT_NULL(settings);
                NN_SDK_ASSERT(info.info.pcMessageForward.size == sizeof(*settings));
                nn::Result result = GetGlobalProfileSettings(settings, sizeof(*settings));
                if (result.IsSuccess())
                {
                    data = settings;
                }
                else
                {
                    WARNING_LOG("Failed to get current profile settings.\n");
                    DumpResultInformation(LOG_AS_WARNING, result);
                }
            }
            else if (info.info.pcMessageForward.message == ProfilerCommMessage_Argv)
            {
                NN_SDK_ASSERT(info.info.pcMessageForward.size <= sizeof(info.info.pcMessageForward.smallPayload));
                data = &info.info.pcMessageForward.smallPayload;
            }
        }

        INFO_LOG("PcMessageForward: %d\n", info.info.pcMessageForward.message);

        if (globals.receivedMessageCallback != nullptr)
        {
            globals.receivedMessageCallback(
                data,
                info.info.pcMessageForward.size,
                info.info.pcMessageForward.message);
        }

        if (allocated)
        {
            NN_SDK_ASSERT_NOT_NULL(data);
            Memory::GetInstance()->Free(data);
        }
    }


    void IpcEventHandler_InstrumentationBuffer(IpcEventInfo& info)
    {
        switch (info.info.instrumentationBuffer.type)
        {
        case InstrumentationBufferType_Start:
            {
                SampleBuffers::GetInstance()->Reset();

                auto settings = GetProfilerSettingsPointer();
                nn::Result result = GetGlobalProfileSettings(settings, sizeof(*settings));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                PrepareInstrumentationBuffer(settings);
            }
            break;

        case InstrumentationBufferType_Stop:
            CloseInstrumentationBuffer();
            break;

        default:
            ERROR_LOG("Invalid type for Instrumentation Buffer IPC\n");
            break;
        }
    }


    void IpcEventHandler_SendData(IpcEventInfo& info)
    {
        NN_UNUSED(info);
        {
            StringTable* table = GetMarkerNamesStringTable();
            nn::Result result = SendBufferToProcess(
                table->GetTableAddress(),
                table->GetUsedTableSize(),
                SendBufferType_MarkerNames);
            if (result.IsFailure())
            {
                DumpResultInformation(LOG_AS_FATAL, result);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
        {
            const char* nvnFuncs;
            size_t nvnSize;
            bool foundNvnFuncs = GetNvnFunctionPointers(&nvnFuncs, &nvnSize);
            if (foundNvnFuncs)
            {
                NN_SDK_ASSERT(nvnFuncs != nullptr && nvnSize > 0);
                nn::Result result = SendBufferToProcess(nvnFuncs, nvnSize, SendBufferType_NvnTable);
                if (result.IsFailure())
                {
                    DumpResultInformation(LOG_AS_FATAL, result);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            }
        }
        {
            WorkArea* ws = GetWorkAreaForCore(SampleBufferIndex_Instrumentation);
            if (ws->startPtr != nullptr)
            {
                while (ws->recordCount > 0)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
                }

                size_t size = ws->curPtr - ws->startPtr;
                size = nn::util::align_up(size, nn::os::MemoryPageSize);
                NN_SDK_ASSERT(size <= SampleBuffers::GetInstance()->GetSize());

                nn::Result result = SetupTransferMemory(ws->startPtr, size);
                if (result.IsFailure())
                {
                    DumpResultInformation(LOG_AS_FATAL, result);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
            }
        }
        SendBasicIpcMessage(ProfilerIpcMessage_AllDataSent, 0);
    }


    void IpcEventLoopThread(void* arg)
    {
        nn::Result result = nn::ResultSuccess();

        nn::os::MultiWaitType multiWait;
        nn::os::InitializeMultiWait(&multiWait);

        nn::os::MultiWaitHolderType ipcHolder;
        nn::os::InitializeMultiWaitHolder(&ipcHolder, &g_ProfilerProcessEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &ipcHolder);

        nn::os::MultiWaitHolderType exitHolder;
        nn::os::InitializeMultiWaitHolder(&exitHolder, &globals.exitEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &exitHolder);

        while (!globals.shouldExit)
        {
            DUMP_CURRENT_LINE();

            auto signaled = nn::os::WaitAny(&multiWait);
            if (globals.shouldExit) { break; }
            nn::os::ClearSystemEvent(&g_ProfilerProcessEvent);

            NN_SDK_ASSERT(signaled == &ipcHolder);
            NN_UNUSED(signaled);

            DUMP_CURRENT_LINE();

            IpcEventInfo info;
            while (!globals.shouldExit)
            {
                result = nn::profiler::GetIpcEvent(&info);
                if (nn::profiler::ResultNoEvent::Includes(result))
                {
                    break;
                }

                // Process events
                INFO_LOG("Received profiler IPC event: %d\n", info.event);
                if (info.event == IpcEvent_Status)
                {
                    IpcEventHandler_Status(info);
                }
                else if (info.event == IpcEvent_MessageCallback)
                {
                    IpcEventHandler_MessageCallback(info);
                }
                else if (info.event == IpcEvent_PcMessageForward)
                {
                    IpcEventHandler_PcMessageForward(info);
                }
                else if (info.event == IpcEvent_InstrumentationBuffer)
                {
                    IpcEventHandler_InstrumentationBuffer(info);
                }
                else if (info.event == IpcEvent_SendData)
                {
                    IpcEventHandler_SendData(info);
                }
            }
        }

        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
        nn::os::FinalizeMultiWaitHolder(&exitHolder);
        nn::os::FinalizeMultiWaitHolder(&ipcHolder);
        nn::os::FinalizeMultiWait(&multiWait);
    }

} // anonymous

void InitializeIpcEventLoop(
    OnSendDataCompleteCallback messageSentCallback,
    OnPcMessageRecievedCallback messageReceivedCallback)
{
    nn::Result result;

    globals.sendDataCompleteCallback = messageSentCallback;
    globals.receivedMessageCallback = messageReceivedCallback;

    globals.ipcEventStack =
        Memory::GetInstance()->Allocate(IpcEventStackSize, nn::os::GuardedStackAlignment);
    globals.shouldExit = false;

    result = nn::os::CreateThread(
        &globals.ipcEventThread,
        IpcEventLoopThread,
        nullptr,
        globals.ipcEventStack,
        IpcEventStackSize,
        TargetApplication::GetMaximumThreadPriority());
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_ERROR, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    nn::os::InitializeEvent(&globals.exitEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::SetThreadName(&globals.ipcEventThread, "NX CPU Profiler: IPC Loop");
    nn::os::StartThread(&globals.ipcEventThread);
}

void FinalizeIpcEventLoop()
{
    globals.shouldExit = true;
    nn::os::SignalEvent(&globals.exitEvent);
    nn::os::WaitThread(&globals.ipcEventThread);
    nn::os::DestroyThread(&globals.ipcEventThread);

    globals.sendDataCompleteCallback = nullptr;
    globals.receivedMessageCallback = nullptr;

    Memory::GetInstance()->Free(globals.ipcEventStack);
}

}} // nn::profiler
