﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/prepo_PrepoService.h>
#include <nn/prepo/detail/service/prepo_Common.h>
#include <nn/prepo/detail/prepo_ApiDetail.h>
#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>
#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>
#include <nn/prepo/detail/service/core/prepo_ReportBuffer.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>
#include <nn/prepo/detail/service/core/prepo_ThroughputHistory.h>
#include <nn/prepo/detail/service/core/prepo_UploadThread.h>
#include <nn/prepo/detail/service/core/prepo_DebugSettings.h>
#include <nn/arp/arp_Api.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        NN_RESULT_THROW_UNLESS(                                     \
            0 < size && size <= maxSize && value[size - 1] == '\0', \
            ResultInvalidArgument());                               \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

#define CHECK_UID(uid) \
    do                                          \
    {                                           \
        NN_RESULT_THROW_UNLESS(                 \
            uid != nn::account::InvalidUid,     \
            ResultInvalidArgument());           \
    }                                           \
    while (NN_STATIC_CONDITION(false))

#define CHECK_REPORT_DATA(eventId, data) \
    do                                                                                                                              \
    {                                                                                                                               \
        size_t length;                                                                                                              \
        NN_RESULT_THROW_UNLESS(detail::VerifyEventId(&length, eventId.GetData()),                                                   \
            ResultInvalidEventId());                                                                                                \
        NN_RESULT_THROW_UNLESS(detail::VerifyReport(reinterpret_cast<const nn::Bit8*>(data.GetPointerUnsafe()), data.GetSize()),    \
            ResultInvalidReportData());                                                                                             \
    }                                                                                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace prepo { namespace detail { namespace service {

namespace
{
    const nn::TimeSpan ThrouputInterval = nn::TimeSpan::FromMinutes(1);
    const int HistoryCount = 60;
    detail::service::core::ThroughputHistory<HistoryCount> g_ThroughputHistory(ThrouputInterval);
}

PrepoService::PrepoService(const char* serviceName, const Capability& capability) NN_NOEXCEPT :
    m_ServiceName(serviceName),
    m_Capability(capability)
{
    NN_UNUSED(m_ServiceName);
    // NN_DETAIL_FRIENDS_INFO("[friends] Get prepo service. (%s)\n", m_ServiceName ? m_ServiceName : "");
}

PrepoService::~PrepoService() NN_NOEXCEPT
{
}

nn::Result PrepoService::SaveReport(const nn::sf::InArray<char>& eventId, const nn::sf::InBuffer& data, nn::Bit64 processId) NN_NOEXCEPT
{
    // システム権限では利用できない。
    CHECK_CAPABILITY(Capability::Flag_Generic);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_REPORT_DATA(eventId, data);

    nn::Bit8 sysInfo[detail::service::core::SystemInfo::RequiredBufferSize];
    size_t sysInfoSize = 0;

    nn::ApplicationId appId;
    NN_RESULT_DO(GetApplicationId(&appId, processId));

    OutputSaveLog(appId, eventId.GetData());
    g_ThroughputHistory.AddForApplication(1u);

    NN_RESULT_DO(detail::service::core::SystemInfo::Collect(&sysInfoSize, sysInfo, sizeof (sysInfo), appId, eventId.GetData()));
    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Add(ReportCategory_Normal, data.GetPointerUnsafe(), data.GetSize(), sysInfo, sysInfoSize));

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::SaveReportWithUser(const nn::account::Uid& uid,
    const nn::sf::InArray<char>& eventId, const nn::sf::InBuffer& data, nn::Bit64 processId) NN_NOEXCEPT
{
    // システム権限では利用できない。
    CHECK_CAPABILITY(Capability::Flag_Generic);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_UID(uid);

    CHECK_REPORT_DATA(eventId, data);

    nn::Bit8 sysInfo[detail::service::core::SystemInfo::RequiredBufferSize];
    size_t sysInfoSize = 0;

    nn::ApplicationId appId;
    NN_RESULT_DO(GetApplicationId(&appId, processId));

    OutputSaveLog(appId, eventId.GetData());
    g_ThroughputHistory.AddForApplication(1u);

    NN_RESULT_DO(detail::service::core::SystemInfo::Collect(&sysInfoSize, sysInfo, sizeof (sysInfo), appId, eventId.GetData(), uid));
    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Add(ReportCategory_Normal, data.GetPointerUnsafe(), data.GetSize(), sysInfo, sysInfoSize));

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::RequestImmediateTransmission() NN_NOEXCEPT
{
    detail::service::core::UploadThread::RequestTransmission();

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::GetTransmissionStatus(nn::sf::Out<std::int32_t> outStatus) NN_NOEXCEPT
{
    outStatus.Set(detail::service::core::UploadThread::GetTransmissionStatus(ReportCategory_Normal));

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::SaveSystemReport(const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_REPORT_DATA(eventId, data);

    OutputSaveLog(appId, eventId.GetData());
    g_ThroughputHistory.AddForSystem(1u);

    nn::Bit8 sysInfo[detail::service::core::SystemInfo::RequiredBufferSize];
    size_t sysInfoSize = 0;

    NN_RESULT_DO(detail::service::core::SystemInfo::Collect(&sysInfoSize, sysInfo, sizeof (sysInfo), appId, eventId.GetData()));
    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Add(ReportCategory_Normal, data.GetPointerUnsafe(), data.GetSize(), sysInfo, sysInfoSize));

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::SaveSystemReportWithUser(const nn::account::Uid& uid,
    const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_UID(uid);

    CHECK_REPORT_DATA(eventId, data);

    OutputSaveLog(appId, eventId.GetData());
    g_ThroughputHistory.AddForSystem(1u);

    nn::Bit8 sysInfo[detail::service::core::SystemInfo::RequiredBufferSize];
    size_t sysInfoSize = 0;

    NN_RESULT_DO(detail::service::core::SystemInfo::Collect(&sysInfoSize, sysInfo, sizeof (sysInfo), appId, eventId.GetData(), uid));
    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Add(ReportCategory_Normal, data.GetPointerUnsafe(), data.GetSize(), sysInfo, sysInfoSize));

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::SetOperationMode(std::int64_t mode) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    detail::service::core::SystemInfo::SetOperationMode(mode);

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::ClearStorage() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Manage);

    detail::service::core::FileSystem::ClearStorage("prepo");

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::ClearStatistics() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Manage);

    NN_RESULT_DO(detail::service::core::StatisticsManager::GetInstance().Clear());

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::IsUserAgreementCheckEnabled(nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Manage);

    bool isEnabled;

    NN_RESULT_DO(detail::service::core::UserAgreementChecker::GetInstance().IsEnabled(&isEnabled));

    outIsEnabled.Set(isEnabled);

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::SetUserAgreementCheckEnabled(bool isEnabled) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Manage);

    NN_RESULT_DO(detail::service::core::UserAgreementChecker::GetInstance().SetEnabled(isEnabled));

    detail::service::core::UploadThread::RequestTransmission();

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::GetStorageUsage(nn::sf::Out<std::int64_t> outUsed, nn::sf::Out<std::int64_t> outCapacity) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    const int64_t capacity = detail::service::core::FileSystem::GetTotalSpaceSize("prepo") - detail::service::core::FileSystem::EntryInfoSize;

    int64_t freeSpace;
    NN_RESULT_DO(detail::service::core::FileSystem::GetFreeSpaceSize(&freeSpace, NN_DETAIL_PREPO_DATA_MOUNT_NAME));

    *outUsed = capacity - freeSpace;
    *outCapacity = capacity;

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::GetStatistics(nn::sf::Out<nn::prepo::Statistics> outStatistics) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    Statistics statistics;

    detail::service::core::StatisticsManager::GetInstance().GetStatistics(&statistics);

    *outStatistics = statistics;

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::GetThroughputHistory(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<ThroughputRecord>& outArray) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    int count;

    g_ThroughputHistory.Get(&count, outArray.GetData(), static_cast<int>(outArray.GetLength()));

    *outCount = count;

    NN_RESULT_SUCCESS;
}

nn::Result PrepoService::GetLastUploadError() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Debug);

    return detail::service::core::UploadThread::GetLastError(ReportCategory_Normal);
}

nn::Result PrepoService::GetApplicationId(nn::ApplicationId* out, nn::Bit64 processId) NN_NOEXCEPT
{
    nn::os::ProcessId osProcessId = {processId};

    nn::arp::ApplicationLaunchProperty property;

    NN_RESULT_TRY(nn::arp::GetApplicationLaunchProperty(&property, osProcessId))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(ResultApplicationInfoNotRegistered());
        }
    NN_RESULT_END_TRY;

    out->value = property.id.value;

    NN_RESULT_SUCCESS;
}

void PrepoService::OutputSaveLog(nn::ApplicationId appId, const char* eventId) NN_NOEXCEPT
{
#if defined (NN_SDK_BUILD_RELEASE)

    NN_UNUSED(appId);
    NN_UNUSED(eventId);

#else

    NN_FUNCTION_LOCAL_STATIC(bool, s_IsOutputSaveLogEnabled,
        = detail::service::core::DebugSettings::IsOutputSaveLogEnabled());

    if (!s_IsOutputSaveLogEnabled)
    {
        return;
    }

    nn::time::PosixTime now = {};
    nn::time::StandardUserSystemClock::GetCurrentTime(&now);

    nn::time::CalendarTime calendar;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendar, nullptr, now));

    char calendarSting[20] = {};

    nn::util::SNPrintf(calendarSting, sizeof (calendarSting), "%04d-%02d-%02dT%02d:%02d:%02d",
        calendar.year, calendar.month, calendar.day, calendar.hour, calendar.minute, calendar.second);

    NN_DETAIL_PREPO_TRACE("[prepo] %s 0x%016llx %s\n", calendarSting, appId.value, eventId);

#endif
}

}}}}
