﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_DeviceAuthenticationToken.h>
#include <nn/prepo/detail/service/core/prepo_ReportFileManager.h>
#include <nn/prepo/detail/service/core/prepo_ReportUploader.h>
#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>
#include <nn/prepo/detail/service/core/prepo_UploadExecuter.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

ReportFileManager* UploadExecuter::RegisterReportFileManager(ReportFileManager* pManager) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_ReportFileManagerMutex);

    auto pCurrent = m_pReportFileManager;
    m_pReportFileManager = pManager;

    return pCurrent;
}

nn::Result UploadExecuter::Execute(ReportDataSummary* outSummary, nn::Bit8* buffer, size_t bufferSize, bool isImmediate) NN_NOEXCEPT
{
    NN_RESULT_TRY(ExecuteImpl(outSummary, buffer, bufferSize, isImmediate))
        NN_RESULT_CATCH_ALL
        {
            m_TransmissionStatus = TransmissionStatus_Pending;
            m_LastError = NN_RESULT_CURRENT_RESULT;

            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    m_TransmissionStatus = TransmissionStatus_Idle;
    m_LastError = nn::ResultSuccess();

    NN_RESULT_SUCCESS;
}

nn::Result UploadExecuter::ExecuteImpl(ReportDataSummary* outSummary, nn::Bit8* buffer, size_t bufferSize, bool isImmediate) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_ReportFileManagerMutex);

    std::memset(outSummary, 0, sizeof (*outSummary));

    if (!m_pReportFileManager)
    {
        NN_RESULT_SUCCESS;
    }

    ReportDataSummary reportDataSummary;

    NN_RESULT_TRY(m_pReportFileManager->LoadUploadFile(&reportDataSummary, buffer, bufferSize))
        NN_RESULT_CATCH(ResultReportFileNotFound)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(UserAgreementChecker::GetInstance().Update());
    if (m_pReportFileManager->IsUserAgreementCheckRequired())
    {
        NN_RESULT_DO(UserAgreementChecker::GetInstance().IsAnyoneAgreedWithAnalystics());
    }
    else
    {
        NN_RESULT_DO(UserAgreementChecker::GetInstance().IsAnyoneAgreedWithNaEula());
    }

    nn::nifm::NetworkConnection nc;
    NN_RESULT_THROW_UNLESS(Connect(nc, isImmediate), ResultNetworkUnavailable());

    m_TransmissionStatus = TransmissionStatus_Processing;

    const char* authToken = nullptr;

    NN_RESULT_TRY(DeviceAuthenticationToken::GetInstance().Get(&authToken))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW_UNLESS(nc.IsAvailable(), ResultNetworkUnavailable());
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    while (NN_STATIC_CONDITION(true))
    {
        if (reportDataSummary.restCount > 0)
        {
            NN_RESULT_TRY(ReportUploader::GetInstance().Upload(authToken, buffer, reportDataSummary.dataSize))
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW_UNLESS(nc.IsAvailable(), ResultNetworkUnavailable());
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY;
        }

        *outSummary += reportDataSummary;

        NN_RESULT_DO(m_pReportFileManager->CleanupUploadedFile());

        NN_RESULT_TRY(m_pReportFileManager->LoadUploadFile(&reportDataSummary, buffer, bufferSize))
            NN_RESULT_CATCH(ResultReportFileNotFound)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
    }
}

TransmissionStatus UploadExecuter::GetTransmissionStatus() NN_NOEXCEPT
{
    return m_TransmissionStatus;
}

nn::Result UploadExecuter::GetLastError() NN_NOEXCEPT
{
    return m_LastError;
}

bool UploadExecuter::Connect(nn::nifm::NetworkConnection& nc, bool isImmediate) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(nc.GetRequestHandle(),
        isImmediate
            ? nn::nifm::RequirementPreset_InternetForSystemProcessSharable
            : nn::nifm::RequirementPreset_InternetForSystemProcess));

    nc.SubmitRequestAndWait();

    return nc.IsAvailable();
}

}}}}}
