﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>
#include <nn/prepo/detail/service/util/prepo_Time.h>
#include <nn/prepo/detail/service/util/prepo_FirmwareVersion.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

namespace
{
    nn::Result WriteAny64BitId(msgpack::OutputStreamParam& stream, const Any64BitId& any64BitId) NN_NOEXCEPT
    {
        Bit64 id;
        nn::util::StoreBigEndian(&id, any64BitId.id);

        NN_RESULT_THROW_UNLESS(msgpack::WriteExtension(&stream, MsgpackExtensionType_Any64BitId, &id, 8), ResultOutOfResource());

        NN_RESULT_SUCCESS;
    }
}

namespace
{
    nn::os::Event g_OperationModeSetEvent(nn::os::EventClearMode_ManualClear);
    std::atomic<int64_t> g_OperationMode;
}

void SystemInfo::SetOperationMode(int64_t mode) NN_NOEXCEPT
{
    g_OperationMode = mode;
    g_OperationModeSetEvent.Signal();
}

nn::Result SystemInfo::GetOperationMode(int64_t* out) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(g_OperationModeSetEvent.TryWait(), ResultOperationModeUndefined());

    *out = g_OperationMode;

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::Collect(size_t* outSize, void* buffer, size_t size,
    nn::ApplicationId appId, const char* eventId) NN_NOEXCEPT
{
    return Collect(outSize, buffer, size, appId, eventId, nn::account::InvalidUid);
}

nn::Result SystemInfo::Collect(size_t* outSize, void* buffer, size_t size,
    nn::ApplicationId appId, const char* eventId, const nn::account::Uid& uid) NN_NOEXCEPT
{
    msgpack::OutputStreamParam stream = {static_cast<nn::Bit8*>(buffer), size};

    const int itemCount = 9;

    NN_RESULT_THROW_UNLESS(msgpack::WriteMapAutoSize(&stream, itemCount), ResultOutOfResource());

    NN_RESULT_DO(CollectApplicationId(stream, appId));
    NN_RESULT_DO(CollectEventId(stream, eventId));
    NN_RESULT_DO(CollectOperationMode(stream));
    NN_RESULT_DO(CollectLocalClockRecordAt(stream));
    NN_RESULT_DO(CollectNetworkClockRecordAt(stream));
    NN_RESULT_DO(CollectNetworkServiceAccountId(stream, uid));
    NN_RESULT_DO(CollectOsVersion(stream));
    NN_RESULT_DO(CollectReportIdentifier(stream));
    NN_RESULT_DO(CollectRebootlessSystemUpdateVersion(stream));

    *outSize = stream.position;

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectSync(size_t* outSize, void* buffer, size_t size,
    nn::ApplicationId appId, const char* eventId) NN_NOEXCEPT
{
    return CollectSync(outSize, buffer, size, appId, eventId, nn::account::InvalidUid);
}

nn::Result SystemInfo::CollectSync(size_t* outSize, void* buffer, size_t size,
    nn::ApplicationId appId, const char* eventId, const nn::account::Uid& uid) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        NN_RESULT_TRY(Collect(outSize, buffer, size, appId, eventId, uid))
            NN_RESULT_CATCH(ResultOperationModeUndefined)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                continue;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }
}

nn::Result SystemInfo::CollectApplicationId(msgpack::OutputStreamParam& stream, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "application_id", sizeof ("application_id") - 1), ResultOutOfResource());

    Any64BitId any64BitId = {appId.value};
    NN_RESULT_DO(WriteAny64BitId(stream, any64BitId));

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectEventId(msgpack::OutputStreamParam& stream, const char* eventId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "event_id", sizeof ("event_id") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, eventId, nn::util::Strnlen(eventId, EventIdLengthMax)), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectOperationMode(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    int64_t operationMode;
    NN_RESULT_DO(GetOperationMode(&operationMode));

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "operation_mode", sizeof ("operation_mode") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteSignedIntegerAutoSize(&stream, operationMode), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectLocalClockRecordAt(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    char currentTime[detail::service::util::DateTimeStringSize];
    NN_RESULT_DO(detail::service::util::GetCurrentUserTimeString(currentTime, sizeof(currentTime)));

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "lc_recorded_at", sizeof ("lc_recorded_at") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, currentTime, sizeof (currentTime) - 1), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectNetworkClockRecordAt(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "nc_recorded_at", sizeof ("nc_recorded_at") - 1), ResultOutOfResource());

    char currentTime[detail::service::util::DateTimeStringSize];

    if (detail::service::util::GetCurrentNetworkTimeString(currentTime, sizeof(currentTime)).IsSuccess())
    {
        NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, currentTime, sizeof (currentTime) - 1), ResultOutOfResource());
    }
    else
    {
        // 時刻補正されていない場合は、空文字を指定する。
        NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "", 0), ResultOutOfResource());
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectNetworkServiceAccountId(msgpack::OutputStreamParam& stream, nn::account::Uid uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "nsa_id", sizeof ("nsa_id") - 1), ResultOutOfResource());

    if (uid)
    {
        nn::account::NetworkServiceAccountId accountId = {};
        NN_RESULT_DO(GetAccountId(&accountId, uid));

        Any64BitId any64BitId = {accountId.id};
        NN_RESULT_DO(WriteAny64BitId(stream, any64BitId));
    }
    else
    {
        NN_RESULT_THROW_UNLESS(msgpack::WriteNil(&stream), ResultOutOfResource());
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::GetAccountId(nn::account::NetworkServiceAccountId* out, const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(manager.GetNetworkServiceAccountId(&accountId));

    *out = accountId;

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectOsVersion(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    auto version = detail::service::util::GetFirmwareVersion();

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "os_version", sizeof ("os_version") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, version.displayVersion, nn::util::Strnlen(version.displayVersion, sizeof(version.displayVersion))), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectReportIdentifier(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    const auto uuid = nn::util::GenerateUuid();

    char uuidString[nn::util::Uuid::StringSize];
    uuid.ToString(uuidString, sizeof(uuidString));

    NN_SDK_ASSERT_EQUAL(nn::util::Strnlen(uuidString, sizeof(uuidString)), static_cast<int>(nn::util::Uuid::StringSize - 1));

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "report_identifier", sizeof ("report_identifier") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, uuidString, nn::util::Uuid::StringSize - 1), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

nn::Result SystemInfo::CollectRebootlessSystemUpdateVersion(msgpack::OutputStreamParam& stream) NN_NOEXCEPT
{
    nn::settings::system::RebootlessSystemUpdateVersion version;
    nn::settings::system::GetRebootlessSystemUpdateVersion(&version);

    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, "rebootless_system_update_version", sizeof ("rebootless_system_update_version") - 1), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, version.displayVersion, nn::util::Strnlen(version.displayVersion, sizeof(version.displayVersion))), ResultOutOfResource());

    NN_RESULT_SUCCESS;
}

}}}}}
