﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_Storage.h>
#include <nn/prepo/detail/service/prepo_Common.h>
#include <nn/fs/fs_FileSystemPrivate.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

void Storage::EnableTestMode() NN_NOEXCEPT
{
    g_IsTestMode = true;
}

std::atomic<bool> Storage::g_IsTestMode(false);

Storage::Storage(const char* mountName, nn::fs::SystemSaveDataId saveDataId,
    int64_t totalSize, int64_t journalSize) NN_NOEXCEPT
    : m_MountName(mountName)
    , m_SaveDataId(saveDataId)
    , m_TotalSize(totalSize)
    , m_JournalSize(journalSize)
{
}

nn::Result Storage::Mount()
{
    if (g_IsTestMode)
    {
        char root[128] = {};
        nn::util::SNPrintf(root, sizeof (root), "C:\\siglo\\%s", m_MountName);
        NN_RESULT_DO(nn::fs::MountHost(m_MountName, root));
    }
    else
    {
        NN_RESULT_TRY(nn::fs::MountSystemSaveData(m_MountName, m_SaveDataId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateSystemSaveData(m_SaveDataId, GetTotalSpaceSize(), m_JournalSize, 0));
                NN_RESULT_DO(nn::fs::MountSystemSaveData(m_MountName, m_SaveDataId));
            }
        NN_RESULT_END_TRY;
    }
    NN_RESULT_SUCCESS;
}

void Storage::Unmount() NN_NOEXCEPT
{
    nn::fs::Unmount(m_MountName);
}

void Storage::Lock() NN_NOEXCEPT
{
    m_Mutex.lock();
}

void Storage::Unlock() NN_NOEXCEPT
{
    m_Mutex.unlock();
}

nn::Result Storage::Commit() NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        NN_RESULT_SUCCESS;
    }

    return nn::fs::CommitSaveData(m_MountName);
}

nn::Result Storage::Rollback() NN_NOEXCEPT
{
    Unmount();

    return Mount();
}

nn::Result Storage::Clear() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsLockedByCurrentThread());

    char root[nn::fs::MountNameLengthMax + 3] = {};
    const int rootLength = nn::util::SNPrintf(root, sizeof (root), "%s:/", m_MountName);
    NN_UNUSED(rootLength);
    NN_SDK_ASSERT_LESS(static_cast<size_t>(rootLength), sizeof (root));

    nn::fs::DirectoryHandle handle = {};

    NN_RESULT_DO(nn::fs::OpenDirectory(&handle, root, nn::fs::OpenDirectoryMode_All));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    while (NN_STATIC_CONDITION(true))
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry = {};

        NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, &entry, handle, 1));

        if (!(entryCount > 0))
        {
            NN_RESULT_SUCCESS;
        }

        char path[nn::fs::EntryNameLengthMax + 1];
        const int pathLength = nn::util::SNPrintf(path, sizeof (path), "%s:/%s", m_MountName, entry.name);
        NN_UNUSED(pathLength);
        NN_SDK_ASSERT_LESS(static_cast<size_t>(pathLength), sizeof (path));

        if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
        {
            NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
        }
        else
        {
            NN_SDK_ASSERT_EQUAL(entry.directoryEntryType, nn::fs::DirectoryEntryType_File);

            NN_RESULT_DO(nn::fs::DeleteFile(path));
        }
    }
}

const char* Storage::GetMountName() const NN_NOEXCEPT
{
    return m_MountName;
}

nn::Result Storage::GetUsageImpl(int64_t* outFileUsage, int64_t* outFileCount, int64_t* outDirectoryCount, char* path, bool isRoot) const NN_NOEXCEPT
{
    auto RoundUpFileSize = [](int64_t fileSize) -> int64_t { return (fileSize + (FsBlockSize - 1)) & ~(FsBlockSize - 1); };

    nn::fs::DirectoryHandle directoryHandle = {};

    NN_RESULT_DO(nn::fs::OpenDirectory(&directoryHandle, path, nn::fs::OpenDirectoryMode_All));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(directoryHandle);
    };

    while (NN_STATIC_CONDITION(true))
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry = {};

        NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, &entry, directoryHandle, 1));

        if (!(entryCount > 0))
        {
            NN_RESULT_SUCCESS;
        }

        if (!isRoot)
        {
            std::strcat(path, "/");
        }
        std::strcat(path, entry.name);

        if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
        {
            NN_RESULT_DO(GetUsageImpl(outFileUsage, outFileCount, outDirectoryCount, path, false));

            (*outDirectoryCount)++;
        }
        else
        {
            NN_SDK_ASSERT_EQUAL(entry.directoryEntryType, nn::fs::DirectoryEntryType_File);

            nn::fs::FileHandle fileHandle = {};

            NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            int64_t fileSize = 0;

            NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, fileHandle));

            *outFileUsage += RoundUpFileSize(fileSize);

            (*outFileCount)++;
        }

        if (isRoot)
        {
            *(std::strrchr(path, '/') + 1) = '\0';
        }
        else
        {
            *(std::strrchr(path, '/')) = '\0';
        }

    }
}

int64_t Storage::GetUsage() const NN_NOEXCEPT
{
    char root[nn::fs::EntryNameLengthMax + 1] = {};
    nn::util::SNPrintf(root, sizeof(root), "%s:/", m_MountName);

    int64_t fileUsage = 0;
    int64_t fileCount = 0;
    int64_t directoryCount = 1; // ルートディレクトリを含める。

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetUsageImpl(&fileUsage, &fileCount, &directoryCount, root, true));

    // 計算方法は以下を参考にしつつ、実測に基づいて求めている。
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=107786293
    //
    // 実際の消費量は FS の実装に依存するが、テスト用の関数なので不問とする。
    //
    // 1 つのアロケーションテーブルには、171 エントリを格納できる。
    // ただし、最初のアロケーションテーブルは、管理領域に 2 エントリ消費する。
    // さらに、512 エントリごとに 1 エントリ消費する。

    const int64_t FsEntryCountForManagementArea = 2;
    const int64_t FsEntryCountMaxInAllocationTable = 171;
    const int64_t FsEntryCountMaxInAllocationTableUnit = 512;

    const int64_t fileEntryCount = fileCount + FsEntryCountForManagementArea;
    const int64_t fileEntryTableCount = ((fileEntryCount + ((fileEntryCount - 1) / FsEntryCountMaxInAllocationTableUnit)) / FsEntryCountMaxInAllocationTable) + 1;

    const int64_t directoryEntryCount = directoryCount + FsEntryCountForManagementArea;
    const int64_t directoryEntryTableCount = ((directoryEntryCount + ((directoryEntryCount - 1) / FsEntryCountMaxInAllocationTableUnit)) / FsEntryCountMaxInAllocationTable) + 1;

    return fileUsage + (fileEntryTableCount * FsBlockSize) + (directoryEntryTableCount * FsBlockSize);
}

int64_t Storage::GetFreeSpaceSize() const NN_NOEXCEPT
{
    if (g_IsTestMode)
    {
        return std::max(GetTotalSpaceSize() - GetUsage(), static_cast<int64_t>(0));
    }
    else
    {
        char root[nn::fs::MountNameLengthMax + 3] = {};
        nn::util::SNPrintf(root, sizeof (root), "%s:/", m_MountName);
        int64_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&size, root));
        return size;
    }
}

int64_t Storage::GetTotalSpaceSize() const NN_NOEXCEPT
{
    return m_TotalSize - m_JournalSize;
}

bool Storage::IsLockedByCurrentThread() const NN_NOEXCEPT
{
    return m_Mutex.IsLockedByCurrentThread();
}

}}}}}
