﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_ReportUploader.h>
#include <nn/prepo/detail/service/util/prepo_HttpErrorHandler.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/settings/system/settings_ProductModel.h>
#include <nn/ssl/ssl_Context.h>

#define CURL_EASY_DO(exp) NN_RESULT_DO(detail::service::util::HandleHttpError(exp))

#define CURL_SLIST_APPEND_DO(list, data) \
    do \
    { \
        auto _temp = curl_slist_append(list, data); \
        NN_RESULT_THROW_UNLESS(_temp, ResultHttpErrorOutOfMemory()); \
        list = _temp; \
    } while (NN_STATIC_CONDITION(false))

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

namespace
{
    const char* ServerUrl = "https://receive-%.dg.srv.nintendo.net/post";
}

ReportUploader::ReportUploader() NN_NOEXCEPT
{
}

nn::Result ReportUploader::Upload(const char* authToken, const Bit8* data, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(authToken);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    m_Stream.data = data;
    m_Stream.size = size;
    m_Stream.position = 0;

    NN_RESULT_DO(Perform(authToken));

    NN_DETAIL_PREPO_INFO("[prepo] Success to upload the file.\n");

    NN_RESULT_SUCCESS;
}

nn::Result ReportUploader::SetupCurl(curl_slist** outHeaders, CURL* curl, const char* authToken) NN_NOEXCEPT
{
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, ServerUrl));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_READFUNCTION, HttpReadFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_READDATA, this));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_POST, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE_LARGE, static_cast<curl_off_t>(m_Stream.size)));

    // 開発中に HTTP 通信ログを出したい場合、有効にする。
    // CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    // 最大リダイレクト回数を制限する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));
    // 1 B/s 以下の転送速度の通信が 30 秒以上続いたら切断する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));
    // 接続タイムアウト（CONNECT 完了までの時間）を 30 秒にする。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));
    // 証明書の有効期限確認のみ無効化する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    curl_slist* headers = nullptr;

    NN_UTIL_SCOPE_EXIT
    {
        if (headers)
        {
            curl_slist_free_all(headers);
        }
    };

    {
        // スタックの消費が大きいので注意。
        char header[2048];
        nn::util::SNPrintf(header, sizeof (header), "X-Nintendo-Device-Authentication-Token: %s", authToken);

        CURL_SLIST_APPEND_DO(headers, header);
    }

    {
        nn::settings::system::ProductModel model = nn::settings::system::GetProductModel();

        char header[30];
        nn::util::SNPrintf(header, sizeof (header), "X-Nintendo-Product-Model: %d", model);

        CURL_SLIST_APPEND_DO(headers, header);
    }

    {
        nn::nsd::EnvironmentIdentifier env;
        nn::nsd::GetEnvironmentIdentifier(&env);

        char header[40];
        nn::util::SNPrintf(header, sizeof (header), "X-Nintendo-Environment: %s", env.value);

        CURL_SLIST_APPEND_DO(headers, header);
    }

    {
        CURL_SLIST_APPEND_DO(headers, "Content-Type: application/x-msgpack");
    }

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers));

    *outHeaders = headers;
    headers = nullptr;

    NN_RESULT_SUCCESS;
}

nn::Result ReportUploader::Perform(const char* authToken) NN_NOEXCEPT
{
    CURL* curl = curl_easy_init();

    NN_RESULT_THROW_UNLESS(curl, ResultHttpErrorFailedInit());

    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);
    };

    curl_slist* headers = nullptr;

    NN_RESULT_DO(SetupCurl(&headers, curl, authToken));

    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };

    CURL_EASY_DO(curl_easy_perform(curl));

    long statusCode = 0;
    CURL_EASY_DO(curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &statusCode));

    NN_RESULT_TRY(detail::service::util::HandleHttpStatusCode(statusCode))
        NN_RESULT_CATCH(ResultServerError400)
        {
            // 何らかの原因で、ファイルが破損していてアップロードに失敗した場合は、
            // 破損したファイルを復旧する方法がないため成功扱いにして、
            // 正常系の後処理で破損したファイルを削除する。
            NN_DETAIL_PREPO_ERROR(
                NN_DETAIL_PREPO_STRING_ERROR(
                    "Failed to upload with HTTP400 error. Skip uploading.\n"));
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

CURLcode ReportUploader::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    auto result = context->Create(nn::ssl::Context::SslVersion_Auto);
    if (result.IsFailure())
    {
        NN_DETAIL_PREPO_ERROR(
            NN_DETAIL_PREPO_STRING_ERROR("Failed to create SSL context with error %03d-%04d\n"),
            result.GetModule(),
            result.GetDescription());
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t ReportUploader::HttpReadFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    ReportUploader* uploader = reinterpret_cast<ReportUploader*>(param);

    size_t bufferSize = size * count;
    size_t read = std::min(bufferSize, uploader->m_Stream.GetRemainSize());

    std::memcpy(buffer, uploader->m_Stream.GetReadPointer(), read);

    uploader->m_Stream.position += read;

    return read;
}

size_t ReportUploader::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    // 空の WriteFunction を設定して、レスポンスデータがログに出力されないようにする。
    NN_UNUSED(buffer);
    NN_UNUSED(param);

    return size * count;
}

}}}}}
