﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "perflog_DaemonImpl.h"
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/os/os_Result.public.h>


// Implemention for the perflog daemon ServiceFramework interface
// This class keeps a dictionary of handles, each associated with a name
// provided by clients.
// It then provides the service of triggering these event handles
// by name on behalf of other clients

namespace nn { namespace perflog {

// static nn::os::NativeHandle g_xdbg_hh;
DaemonImpl::DaemonImpl() NN_NOEXCEPT :
    m_NextDumpCookie(0)
{
    Entry* pCurr = m_Entries;
    for( uint32_t ix = 0; ix < MaxEntries; ++ix )
    {
        pCurr->name[0] = 0; // mark free
        ++pCurr;
    }
}


DaemonImpl::Entry* DaemonImpl::FindFreeEntry( ) NN_NOEXCEPT
{
    Entry* pCurr = m_Entries;
    for( uint32_t ix = 0; ix < MaxEntries; ++ix )
    {
        if( pCurr->name[0] == 0 )
        {
            return pCurr;
        }
        ++pCurr;
    }
    return nullptr;
}


// finds an entry given its name
bool DaemonImpl::MatchEntry( const char* mask, const char* candidateName ) NN_NOEXCEPT
{
    // find first difference, but stop when strings end
    while( ((*mask)          != 0) &&
           ((*candidateName) != 0) &&
           ( (*mask) == (*candidateName)) )
    {
        ++mask;
        ++candidateName;
    }


    if( (*mask) == '*' )
    {
        // first difference is a star
        return true;
    }

    if( ((*candidateName) == 0) && ((*mask) == 0) )
    {
        // exact match
        return true;
    }

    // any other case (different chars, incomplete on either side)
    return false;
}

bool DaemonImpl::IsNameInUse(const char* evName) NN_NOEXCEPT
{
    Entry *pCurr = m_Entries;
    for( uint32_t ix = 0; ix < MaxEntries; ++ix )
    {
        if( (pCurr->name[0] != 0) && (strcmp(evName, pCurr->name) == 0) )
        {
            return true;
        }
        ++pCurr;
    }
    return false;
}

nn::Result DaemonImpl::RegisterTrigger(
    const nn::sf::InBuffer& pNameAsBuffer,
    nn::sf::NativeHandle&& signalHandle ) NN_NOEXCEPT
{
    const char* evName;
    Entry *pNext = FindFreeEntry();
    if( pNext == nullptr )
    {
        return ResultOutOfResource();
    }
    else if ( nullptr == (evName = pNameAsBuffer.GetPointerUnsafe()) )
    {
        return ResultInvalidParameter(); // empty name
    }
    else if( (*evName) == 0 )
    {
        return ResultInvalidParameter(); // invalid name
    }
    else if( IsNameInUse(evName) )
    {
        return ResultInvalidParameter(); // already taken
    }
    else
    {
        strncpy( pNext->name, evName, MaxNameLen );
        pNext->name[ sizeof(pNext->name) - 1 ] = 0;
        pNext->event = std::move( signalHandle );
        NN_SDK_LOG( "XDBG: v4: RegisterTrigger with name=%s handle=%x\n", evName, pNext->event.GetOsHandle() );
        NN_SDK_ASSERT( pNext->event.IsManaged() ); // we need this to close the handle
    }
    NN_RESULT_SUCCESS;
}

nn::Result DaemonImpl::BeginDump(
    nn::sf::Out<uint64_t> cookie ) NN_NOEXCEPT
{
    ++m_NextDumpCookie;
    *cookie = m_NextDumpCookie;
    NN_SDK_LOG( "BeginDump\n" );
    NN_RESULT_SUCCESS;
}

nn::Result DaemonImpl::DumpLine(
    uint64_t cookie,
    uint64_t thread,
    uint64_t timestamp,
    uint32_t id,
    const nn::sf::InBuffer& pContextAsBuffer ) NN_NOEXCEPT
{

    const char* ctx;
    if ( nullptr == (ctx = pContextAsBuffer.GetPointerUnsafe()) )
    {
        return ResultInvalidParameter();
    }
    else
    {
        // log disappears on release builds, so reference vars here
        NN_UNUSED(cookie);
        NN_UNUSED(thread);
        NN_UNUSED(timestamp);
        NN_UNUSED(id);
        NN_SDK_LOG( "[perflog] cookie[%llx] thr[%llx] ts[%lld] id[%x] ctx[%s]\n",
             cookie, thread, timestamp, id, ctx  );
    }
    NN_RESULT_SUCCESS;
}

nn::Result DaemonImpl::EndDump(
    uint64_t cookie ) NN_NOEXCEPT
{
    NN_UNUSED(cookie); // log disappears on release build
    NN_SDK_LOG( "EndDump with cookie[%llx]\n", cookie);
    NN_RESULT_SUCCESS;
}


nn::Result DaemonImpl::TriggerDump(
    const nn::sf::InBuffer& pNameAsBuffer ) NN_NOEXCEPT
{
    const char* evName;
    if ( NULL == (evName = pNameAsBuffer.GetPointerUnsafe()) )
    {
        return ResultInvalidParameter();
    }
    else
    {
        bool bSignaledAny = false;
        Entry *pCurr = m_Entries;
        for( uint32_t ix = 0; ix < MaxEntries ; ++ix )
        {
            if( (pCurr->name[0] != 0) && MatchEntry(evName, pCurr->name) )
            {
                nn::os::SystemEventType evt;
                nn::os::AttachWritableHandleToSystemEvent(&evt,
                                                          pCurr->event.GetOsHandle(), false,
                                                          nn::os::EventClearMode_AutoClear  );
                nn::os::SignalSystemEvent(&evt);
                nn::os::DetachWritableHandleOfSystemEvent(&evt);
                NN_SDK_LOG( "XDBG: v4: TriggerDump with name=%s, will signal handle %x\n", pCurr->name, pCurr->event.GetOsHandle()  );
                bSignaledAny = true;
            }
            ++pCurr;
        }
        if( !bSignaledAny)
        {
            return ResultNotSupported();
        }
    }
    NN_RESULT_SUCCESS;
}


nn::Result DaemonImpl::UnregisterTrigger(
    const nn::sf::InBuffer& pNameAsBuffer ) NN_NOEXCEPT
{
    const char* evName;
    if ( NULL == (evName = pNameAsBuffer.GetPointerUnsafe()) )
    {
        return ResultInvalidParameter();
    }
    else
    {
        Entry *pCurr = m_Entries;
        uint32_t ix = 0;
        for( ix = 0; ix < MaxEntries; ++ix )
        {
            if( (pCurr->name[0] != 0) && (strcmp(evName, pCurr->name) == 0) )
            {
                nn::sf::NativeHandle destroy = std::move( pCurr->event );
                NN_SDK_LOG( "XDBG: v4: Unregister name=%s, will signal handle %x\n", pCurr->name, destroy.GetOsHandle()  );
                pCurr->name[0] = 0; // mark as unused
                NN_SDK_ASSERT( !pCurr->event.IsManaged() );
                NN_SDK_ASSERT( pCurr->event.GetOsHandle() == nn::os::InvalidNativeHandle );
                break;
            }
            ++pCurr;
        }
        if( ix >= MaxEntries ) // entry was found
        {
            return ResultNotSupported();
        }
    }
    NN_RESULT_SUCCESS;
}


}}  // nn::perflog

