﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryLastPlayTimeApi.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_Types.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_IQueryService.sfdl.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_ServiceNames.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/srv/pdm_QueryServiceImpl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/fs/detail/fs_AccessLog.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_SaveDataCommitThread.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/time.h>
#endif

namespace nn { namespace pdm {

namespace
{
    void InitializeEventTimeData(nn::pdm::EventTimeData* outValue) NN_NOEXCEPT
    {
        outValue->eventIndex = 0;
        outValue->userClockTime = 0;
        outValue->userClockTime = 0;
    }

    void InitializePlayStatistics(PlayStatistics* outValue) NN_NOEXCEPT
    {
        outValue->applicationId.value = 0;
        outValue->totalPlayCount = 0;
        outValue->totalPlayTime = 0;
        InitializeEventTimeData(&outValue->firstEventTime);
        InitializeEventTimeData(&outValue->latestEventTime);
    }

    sf::ShimLibraryObjectHolder<pdm::detail::IQueryService> g_QueryHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    typedef void(*HolderInitializer)(sf::ShimLibraryObjectHolder<pdm::detail::IQueryService>&);

    detail::InitializationManager g_Initialization = NN_PDM_INITIALIZATION_INITIALIZER;

    void Initialize(HolderInitializer holderInitializer) NN_NOEXCEPT
    {
        g_Initialization.Initialize([&](){
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
            detail::InitializeFs();
            detail::InitializeTime();
            detail::SaveDataCommitThread::Start();
#endif
            (*holderInitializer)(g_QueryHolder);
        });
    }

    void Finalize() NN_NOEXCEPT
    {
        g_Initialization.Finalize([]()
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            detail::SaveDataCommitThread::Stop();
            detail::FinalizeFs();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
#endif
            g_QueryHolder.FinalizeHolder();
        });
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    const uint32_t ReadTemporaryByteCapacity = 100 * sizeof(pdm::PlayEvent);
    pdm::detail::LockableMemoryBuiltIn<ReadTemporaryByteCapacity> g_ReadTemporaryBuffer;
    sf::UnmanagedServiceObject<pdm::detail::IQueryService, pdm::srv::QueryServiceImpl> g_ServiceObject(&g_ReadTemporaryBuffer);

    void InitializeForDfc() NN_NOEXCEPT
    {
        Initialize([](sf::ShimLibraryObjectHolder<pdm::detail::IQueryService>& holder) {
            holder.InitializeHolderDirectly(g_ServiceObject.GetShared());
        });
    }

#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    sf::SimpleAllInOneHipcClientManager<17> g_Manager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;

    void InitializeForHipc() NN_NOEXCEPT
    {
        Initialize([](sf::ShimLibraryObjectHolder<pdm::detail::IQueryService>& holder) {
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager.InitializeShimLibraryHolder(&holder, detail::ServiceNameForQuery));
        });
    }
#else
#error "unsupported os"
#endif
}


void InitializeForQuery() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    InitializeForDfc();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    InitializeForHipc();
#else
#error "unsupported os"
#endif
}

void FinalizeForQuery() NN_NOEXCEPT
{
    Finalize();
}

int QueryApplicationEvent(ApplicationEvent outValue[], int outCountMax, int eventIndexOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(eventIndexOffset, 0);

    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryApplicationEvent(&outCount, sf::OutArray<ApplicationEvent>(outValue, outCountMax), eventIndexOffset));

    return outCount;
}

int QueryAccountEvent(AccountEvent outValue[], int outCountMax, int eventIndexOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(eventIndexOffset, 0);

    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryAccountEvent(&outCount, sf::OutArray<AccountEvent>(outValue, outCountMax), eventIndexOffset));

    return outCount;
}

int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);

    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatistics(&outCount, sf::OutArray<PlayStatistics>(outValue, outCountMax)));

    return outCount;
}

int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax, const account::Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);
    NN_SDK_REQUIRES(user);

    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatisticsByUserAccountId(&outCount, sf::OutArray<PlayStatistics>(outValue, outCountMax), user));

    return outCount;
}

int QueryPlayStatistics(PlayStatistics outValue[], int outCountMax, const account::NetworkServiceAccountId& networkServiceAccountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);

    int outCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatisticsByNetworkServiceAccountId(&outCount, sf::OutArray<PlayStatistics>(outValue, outCountMax), networkServiceAccountId));

    return outCount;
}

nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationId != ncm::ApplicationId::GetInvalidId());

    PlayStatistics outValue;
    InitializePlayStatistics(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatisticsByApplicationId(&outValue, applicationId));

    if( outValue.totalPlayCount > 0 )
    {
        return outValue;
    }
    else
    {
        return nn::util::optional<PlayStatistics>();
    }
}

nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId, const account::Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationId != ncm::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES(user);

    PlayStatistics outValue;
    InitializePlayStatistics(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatisticsByApplicationIdAndUserAccountId(&outValue, applicationId, user));

    if( outValue.totalPlayCount > 0 )
    {
        return outValue;
    }
    else
    {
        return nn::util::optional<PlayStatistics>();
    }
}

nn::util::optional<PlayStatistics> QueryPlayStatistics(const ncm::ApplicationId& applicationId, const account::NetworkServiceAccountId& networkServiceAccountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(applicationId != ncm::ApplicationId::GetInvalidId());

    PlayStatistics outValue;
    InitializePlayStatistics(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayStatisticsByApplicationIdAndNetworkServiceAccountId(&outValue, applicationId, networkServiceAccountId));

    if( outValue.totalPlayCount > 0 )
    {
        return outValue;
    }
    else
    {
        return nn::util::optional<PlayStatistics>();
    }
}

int QueryLastPlayTime(LastPlayTime outValue[], const nn::ApplicationId applicationIds[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(applicationIds);
    NN_SDK_REQUIRES_GREATER(count, 0);

    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_QueryHolder->QueryLastPlayTime(&outCount, sf::OutArray<LastPlayTime>(outValue, count), sf::InArray<nn::ApplicationId>(applicationIds, count)));

    return outCount;
}

int QueryPlayEvent(PlayEvent outValue[], int outCountMax, int offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(outCountMax, 0);
    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryPlayEvent(&outCount, sf::OutArray<PlayEvent>(outValue, outCountMax), offset));
    return outCount;
}

int GetAvailablePlayEventRange(int* outStartIndex, int* outLastIndex) NN_NOEXCEPT
{
    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->GetAvailablePlayEventRange(&outCount, outStartIndex, outLastIndex));
    return outCount;
}

int QueryAccountPlayEvent(AccountPlayEvent outValue[], int outCountMax, int offset, const account::Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(static_cast<bool>(user));
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_GREATER(outCountMax, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);

    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->QueryAccountPlayEvent(&outCount, sf::OutArray<AccountPlayEvent>(outValue, outCountMax), offset, user));
    return outCount;
}

int GetAvailableAccountPlayEventRange(int* outStartIndex, int* outLastIndex, const account::Uid& user) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(static_cast<bool>(user));
    NN_SDK_REQUIRES_NOT_NULL(outStartIndex);
    NN_SDK_REQUIRES_NOT_NULL(outLastIndex);

    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_QueryHolder->GetAvailableAccountPlayEventRange(&outCount, outStartIndex, outLastIndex, user));
    return outCount;
}

int QueryApplicationPlayStatisticsForSystem(ApplicationPlayStatistics outValues[], const ncm::ApplicationId applicationIds[], int count) NN_NOEXCEPT
{
    int outCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_SYSTEM(
            g_QueryHolder->QueryApplicationPlayStatisticsForSystem(&outCount, sf::OutArray<ApplicationPlayStatistics>(outValues, count), sf::InArray<ncm::ApplicationId>(applicationIds, count)),
            nullptr, ""
        )
    );
    return outCount;
}

int QueryRecentlyPlayedApplication(ncm::ApplicationId out[], int outCount, const account::Uid& user) NN_NOEXCEPT
{
    int actualOutCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_QueryHolder->QueryRecentlyPlayedApplication(&actualOutCount, sf::OutArray<ncm::ApplicationId>(out, outCount), user));
    return actualOutCount;
}

os::SystemEvent*  GetRecentlyPlayedApplicationUpdateEvent() NN_NOEXCEPT
{
    static os::SystemEvent e;
    sf::NativeHandle h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_QueryHolder->GetRecentlyPlayedApplicationUpdateEvent(&h));
    e.AttachReadableHandle(h.GetOsHandle(), h.IsManaged(), os::EventClearMode_ManualClear);
    h.Detach();
    return &e;
}

}}
