﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/detail/pdm_PlayEventFactory.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pdm { namespace detail {

namespace {
    PlayEvent MakePlayEventCommon(PlayEventCategory category, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
    {
        PlayEvent playEvent;
        memset(&playEvent, 0, sizeof(PlayEvent));
        playEvent.eventCategory = category;
        playEvent.userTime = userClockTime;
        playEvent.networkTime = networkClockTime;
        playEvent.steadyTime = steadyClockTime;
        return playEvent;
    }
}

PlayEvent MakeAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    return MakeAppletEvent(eventType, programId, version, appletId, storageId, logPolicy, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::Applet, userClockTime, networkClockTime, steadyClockTime);
    pe.appletEventData.eventType = eventType;
    pe.appletEventData.programIdHi = Hi(programId.value);
    pe.appletEventData.programIdLow = Low(programId.value);
    pe.appletEventData.version = version;
    pe.appletEventData.appletId = static_cast<uint8_t>(appletId);
    pe.appletEventData.storageId = storageId;
    pe.appletEventData.playLogPolicy = logPolicy;
    return pe;
}

PlayEvent MakeLibraryAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& mainProgramId, nn::applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    return MakeLibraryAppletEvent(eventType, mainProgramId, libraryAppletMode, appletId, storageId, logPolicy, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeLibraryAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& mainProgramId, nn::applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::Applet, userClockTime, networkClockTime, steadyClockTime);
    pe.appletEventData.eventType = eventType;
    pe.appletEventData.programIdHi = Hi(mainProgramId.value);
    pe.appletEventData.programIdLow = Low(mainProgramId.value);
    pe.appletEventData.laInfo.version = 1;
    pe.appletEventData.laInfo.libraryAppletMode = static_cast<Bit8>(libraryAppletMode);
    pe.appletEventData.appletId = static_cast<uint8_t>(appletId);
    pe.appletEventData.storageId = storageId;
    pe.appletEventData.playLogPolicy = logPolicy;
    return pe;
}

PlayEvent MakePowerStateChangeEvent(PowerStateChangeEventType eventType) NN_NOEXCEPT
{
    return MakePowerStateChangeEvent(eventType, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakePowerStateChangeEvent(PowerStateChangeEventType eventType, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::PowerStateChange, userClockTime, networkClockTime, steadyClockTime);
    pe.powerStateChangeEventData.eventType = eventType;
    return pe;
}

PlayEvent MakeOperationModeChangeEvent(pdm::OperationMode operationMode) NN_NOEXCEPT
{
    return MakeOperationModeChangeEvent(operationMode, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeOperationModeChangeEvent(pdm::OperationMode operationMode, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::OperationModeChange, userClockTime, networkClockTime, steadyClockTime);
    pe.operationModeEventData.operationMode = operationMode;
    return pe;
}

PlayEvent MakeUserAccountEvent(UserAccountEventType eventType, const nn::account::Uid& user) NN_NOEXCEPT
{
    return MakeUserAccountEvent(eventType, user, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeUserAccountEvent(UserAccountEventType eventType, const nn::account::Uid& user, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT(eventType != UserAccountEventType::NetworkServiceAccountAvailable);
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::UserAccount, userClockTime, networkClockTime, steadyClockTime);
    pe.userAccountEventData.eventType = eventType;
    pe.userAccountEventData.userId0Hi = Hi(user._data[0]);
    pe.userAccountEventData.userId0Low = Low(user._data[0]);
    pe.userAccountEventData.userId1Hi = Hi(user._data[1]);
    pe.userAccountEventData.userId1Low = Low(user._data[1]);
    return pe;
}

PlayEvent MakeNetworkServiceAccountAvailableEvent(const nn::account::Uid& user, const nn::account::NetworkServiceAccountId& nid) NN_NOEXCEPT
{
    return MakeNetworkServiceAccountAvailableEvent(user, nid, GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeNetworkServiceAccountAvailableEvent(const nn::account::Uid& user, const nn::account::NetworkServiceAccountId& nid, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::UserAccount, userClockTime, networkClockTime, steadyClockTime);
    pe.userAccountEventData.eventType = UserAccountEventType::NetworkServiceAccountAvailable;
    pe.userAccountEventData.userId0Hi = Hi(user._data[0]);
    pe.userAccountEventData.userId0Low = Low(user._data[0]);
    pe.userAccountEventData.userId1Hi = Hi(user._data[1]);
    pe.userAccountEventData.userId1Low = Low(user._data[1]);
    pe.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdHi = Hi(nid.id);
    pe.userAccountEventData.networkServiceAccountAvailableData.networkServiceAccountIdLow = Low(nid.id);
    return pe;
}

PlayEvent MakeSteadyClockResetEvent() NN_NOEXCEPT
{
    return MakeSteadyClockResetEvent(GetUserClockTimeForPlayEvent(), GetNetworkClockTimeForPlayEvent(), GetSteadyClockTimeForPlayEvent());
}

PlayEvent MakeSteadyClockResetEvent(const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    return MakePlayEventCommon(PlayEventCategory::SteadyClockReset, userClockTime, networkClockTime, steadyClockTime);
}

}}}
