﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/dd.h>
#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/result/result_HandlingUtility.h>

#include "pcv_Driver.h"
#include "pcv_Sdmmc1PowerController-hardware.nx.h"

namespace nn { namespace pcv { namespace driver { namespace detail {

void Sdmmc1PowerController::WaitMicroseconds(uint32_t us) NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(us));
}

Result Sdmmc1PowerController::ControlVddioSdmmc1(BusPower busPower) NN_NOEXCEPT
{
    const nn::pcv::PowerDomain powerDomain = nn::pcv::PowerDomain_Max77620_Ldo2;
    switch (busPower)
    {
    case BusPower_Off:
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageEnabled(powerDomain, false));
        break;
    case BusPower_1_8V:
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageValue(powerDomain, 1800000));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageEnabled(powerDomain, true));
        break;
    case BusPower_3_3V:
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageValue(powerDomain, 3300000));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageEnabled(powerDomain, true));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

void Sdmmc1PowerController::SetSdmmc1IoMode(bool isBusPower_3_3V) NN_NOEXCEPT
{
    const nn::dd::PhysicalAddress ApbdevPmcPwrDetVal0Address = 0x7000e4e4ull; // PMC - APBDEV_PMC_PWR_DET_VAL_0
    const int ApbdevPmcPwrDetVal0Sdmmc1Shift = 12;
    Bit32 value = nn::dd::ReadIoRegister(ApbdevPmcPwrDetVal0Address);
    if (isBusPower_3_3V)
    {
        value |= (0x1U << ApbdevPmcPwrDetVal0Sdmmc1Shift);
    }
    else
    {
        value &= (~(0x1U << ApbdevPmcPwrDetVal0Sdmmc1Shift));
    }
    nn::dd::WriteIoRegister(ApbdevPmcPwrDetVal0Address, value);
    (void)nn::dd::ReadIoRegister(ApbdevPmcPwrDetVal0Address); // PMC レジスタ書き込み完了を確かにする
}

void Sdmmc1PowerController::ControlRailSdmmc1Io(bool isPowerOn) NN_NOEXCEPT
{
    const nn::dd::PhysicalAddress ApbdevPmcNoIopower0Address = 0x7000e444ull; // PMC - APBDEV_PMC_NO_IOPOWER_0
    const int ApbdevPmcPwrDetVal0Sdmmc1Shift = 12;
    Bit32 value = nn::dd::ReadIoRegister(ApbdevPmcNoIopower0Address);
    if (isPowerOn)
    {
        value &= (~(0x1U << ApbdevPmcPwrDetVal0Sdmmc1Shift));
    }
    else
    {
        value |= (0x1U << ApbdevPmcPwrDetVal0Sdmmc1Shift);
    }
    nn::dd::WriteIoRegister(ApbdevPmcNoIopower0Address, value);
    (void)nn::dd::ReadIoRegister(ApbdevPmcNoIopower0Address); // PMC レジスタ書き込み完了を確かにする
}

Sdmmc1PowerController::Sdmmc1PowerController() NN_NOEXCEPT
: m_BusPower(BusPower_Off)
{
    nn::gpio::Initialize();
    #if (defined(NN_BUILD_CONFIG_SPEC_NX))
        nn::gpio::OpenSession(&m_GpioPadPowSdEnSession, nn::gpio::GpioPadName_PowSdEn);
    #else
        nn::gpio::OpenSession(&m_GpioPadPowSdEnSession, nn::gpio::GpioPadName_EnablePowerToTheSdCard);
    #endif

    // 出力 LOW （電源 OFF）状態から開始する
    nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_Low);
    nn::gpio::SetDirection(&m_GpioPadPowSdEnSession, nn::gpio::Direction_Output);

    ControlRailSdmmc1Io(false);
}

Sdmmc1PowerController::~Sdmmc1PowerController() NN_NOEXCEPT
{
    nn::gpio::CloseSession(&m_GpioPadPowSdEnSession);
    nn::gpio::Finalize();
}

Result Sdmmc1PowerController::PowerOn(BusPower busPower) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_BusPower == BusPower_Off);

    if ( m_BusPower != BusPower_Off )
    {
        NN_RESULT_SUCCESS;
    }

    // 電源 ON では 3.3V しか想定しない
    NN_ABORT_UNLESS(busPower == BusPower_3_3V);

    ControlRailSdmmc1Io(true);

    // SD カード電源 3.3V
    nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_High);
    WaitMicroseconds(10 * 1000); // 安定待ち

    // IO 電源 3.3V
    SetSdmmc1IoMode(true);
    NN_RESULT_DO(ControlVddioSdmmc1(BusPower_3_3V));
    WaitMicroseconds(130); // 安定待ち
    m_BusPower = busPower;

    NN_RESULT_SUCCESS;
}

Result Sdmmc1PowerController::PowerOff() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_BusPower != BusPower_Off);

    if ( m_BusPower == BusPower_Off )
    {
        NN_RESULT_SUCCESS;
    }

    NN_SDK_ASSERT(m_BusPower == BusPower_1_8V);

    // 電源 OFF は 1.8V からのみ受け付ける。
    if ( m_BusPower != BusPower_1_8V )
    {
        return ResultIllegalRequest();
    }

    // IO 電源 0V
    (void)ControlVddioSdmmc1(BusPower_Off);
    WaitMicroseconds(4 * 1000); // 安定待ち

    // SD カード電源 0V
    nn::gpio::SetValue(&m_GpioPadPowSdEnSession, nn::gpio::GpioValue_Low);
    WaitMicroseconds(239 * 1000); // 安定待ち

    ControlRailSdmmc1Io(false);

    // WAR: ダメージ回避のため、3.3V モードにし、元の端子設定に戻す
    SetSdmmc1IoMode(true);

    m_BusPower = BusPower_Off;

    NN_RESULT_SUCCESS;
}

Result Sdmmc1PowerController::LowerBusPower() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_BusPower == BusPower_3_3V);

    if ( m_BusPower != BusPower_3_3V )
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(ControlVddioSdmmc1(BusPower_1_8V));
    WaitMicroseconds(150); // 安定待ち
    SetSdmmc1IoMode(false);

    m_BusPower = BusPower_1_8V;

    NN_RESULT_SUCCESS;
}

Result Sdmmc1PowerController::IsPoweredOn(bool* pOutIsPoweredOn) NN_NOEXCEPT
{
    return nn::pcv::driver::detail::GetVoltageEnabled(pOutIsPoweredOn, nn::pcv::PowerDomain_Max77620_Ldo2);
}

}}}} // namespace nn::pcv::driver::detail
