﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>

#include <nn/nn_TimeSpan.h>

#include <nn/os/os_Thread.h>
#include <nn/pcv/pcv_Types.h>
#include <nn/pcv/pcv_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Types.h>
#include <nn/util/util_TypedStorage.h>

#include "pcv_Driver.h"
#include "pcv_Sdmmc1PowerController-hardware.nx.h"

namespace nn { namespace pcv { namespace driver { namespace detail {

namespace {

nn::util::TypedStorage<Sdmmc1PowerController, sizeof(Sdmmc1PowerController), NN_ALIGNOF(Sdmmc1PowerController)> g_Sdmmc1PowerControllerStorage;
Sdmmc1PowerController* g_pSdmmc1PowerController = nullptr;

nn::spl::HardwareType g_HardwareType = nn::spl::HardwareType_Icosa;

bool IsHdmiSupported() NN_NOEXCEPT
{
    return (g_HardwareType == nn::spl::HardwareType_Icosa)
        || (g_HardwareType == nn::spl::HardwareType_Copper)
        || (g_HardwareType == nn::spl::HardwareType_Iowa);
}

} // namespace

// TODO: 無名名前空間外に出ているのは SIGLO-79931 のワークアラウンドです。
// Raptor の display ドライバの修正が完了され次第無名名前空間に戻します。
bool IsSioMcuSupported() NN_NOEXCEPT
{
    return (g_HardwareType == nn::spl::HardwareType_Hoag);
}

namespace {

// 単純に PowerControlTarget と PowerDomain が対応する場合を解決する
Result ConvertPowerControlTargetToPowerDomain(PowerDomain* pOutPowerDomain, PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    if ( powerControlTarget == PowerControlTarget_SioMcu && IsSioMcuSupported() )
    {
        *pOutPowerDomain = PowerDomain_Max77620_Ldo7;
        NN_RESULT_SUCCESS;
    }
    else if ( (powerControlTarget == PowerControlTarget_Hdmi && IsHdmiSupported())
        || (powerControlTarget == PowerControlTarget_SioMcuA && IsSioMcuSupported()) )
    {
        *pOutPowerDomain = PowerDomain_Max77620_Ldo8;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultInvalidPowerControlTarget());
}

} // namespace

void InitializePowerController() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(nullptr, g_pSdmmc1PowerController);

    nn::spl::Initialize();
    g_HardwareType = nn::spl::GetHardwareType();
    nn::spl::Finalize();

    g_pSdmmc1PowerController = new (&g_Sdmmc1PowerControllerStorage) Sdmmc1PowerController();
}

void FinalizePowerController() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_pSdmmc1PowerController);
    g_pSdmmc1PowerController->~Sdmmc1PowerController();
    g_pSdmmc1PowerController = nullptr;
}

Result PowerOn(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    if ( powerControlTarget == PowerControlTarget_SdCard )
    {
        NN_SDK_ASSERT_NOT_NULL(g_pSdmmc1PowerController);
        if ( microVolt == 3300000 )
        {
            return g_pSdmmc1PowerController->PowerOn(Sdmmc1PowerController::BusPower_3_3V);
        }
        else
        {
            return ResultUnsupportedVoltageOnTarget();
        }
    }
    else
    {
        PowerDomain powerDomain;
        NN_RESULT_DO(ConvertPowerControlTargetToPowerDomain(&powerDomain, powerControlTarget));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageValue(powerDomain, microVolt));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageEnabled(powerDomain, true));
        NN_RESULT_SUCCESS;
    }
}

Result PowerOff(PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    if ( powerControlTarget == PowerControlTarget_SdCard )
    {
        NN_SDK_ASSERT_NOT_NULL(g_pSdmmc1PowerController);
        return g_pSdmmc1PowerController->PowerOff();
    }
    else
    {
        PowerDomain powerDomain;
        NN_RESULT_DO(ConvertPowerControlTargetToPowerDomain(&powerDomain, powerControlTarget));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageEnabled(powerDomain, false));
        NN_RESULT_SUCCESS;
    }
}

Result ChangeVoltage(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    if ( powerControlTarget == PowerControlTarget_SdCard )
    {
        NN_SDK_ASSERT_NOT_NULL(g_pSdmmc1PowerController);
        if ( microVolt == 1800000 )
        {
            return g_pSdmmc1PowerController->LowerBusPower();
        }
        else
        {
            return ResultUnsupportedVoltageOnTarget();
        }
    }
    else
    {
        PowerDomain powerDomain;
        NN_RESULT_DO(ConvertPowerControlTargetToPowerDomain(&powerDomain, powerControlTarget));
        NN_RESULT_DO(nn::pcv::driver::detail::SetVoltageValue(powerDomain, microVolt));
        NN_RESULT_SUCCESS;
    }
}

Result IsPoweredOn(bool* pOutIsPoweredOn, PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    if ( powerControlTarget == PowerControlTarget_SdCard )
    {
        NN_SDK_ASSERT_NOT_NULL(g_pSdmmc1PowerController);
        return g_pSdmmc1PowerController->IsPoweredOn(pOutIsPoweredOn);
    }
    else
    {
        PowerDomain powerDomain;
        NN_RESULT_DO(ConvertPowerControlTargetToPowerDomain(&powerDomain, powerControlTarget));
        NN_RESULT_DO(nn::pcv::driver::detail::GetVoltageEnabled(pOutIsPoweredOn, powerDomain));
        NN_RESULT_SUCCESS;
    }
}

}}}} // namespace nn::pcv::driver::detail
